#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::More;
use List::Util qw(any);

use qbit;

$ENV{LAZY_LOAD} = FALSE;

my $exclude_fields = {pcode_settings => TRUE};

run_tests(
    sub {
        my ($app) = @_;

        my @rest_api_dbmodels = grep {blessed($app->{$_}) && $app->{$_}->isa('RestApi::DBModel')} keys(%$app);
        ok(@rest_api_dbmodels, "RestApi::DBModels list is not null");
        ok(any(sub {m/_action_log$/}, @rest_api_dbmodels), "RestApi::DBModels list contains action_logs models");
        foreach my $accessor (@rest_api_dbmodels) {
            next if defined($ARGV[0]) && $accessor ne $ARGV[0];

            subtest $accessor => sub {

                my $fields = $app->$accessor->get_model_fields() // {};

                fail(gettext('%s. Field "public_id" not found in %s', $accessor, ref($app->$accessor)))
                  unless exists($fields->{'public_id'});

                my $filters = $app->$accessor->get_model_filter_fields() // {};

                unless (grep {$filters->{$_}{'type'} && $filters->{$_}{'type'} eq 'publicid'} keys(%$filters)) {
                    fail(
                        sprintf(
                            '%s. Filter "public_id" not found and use composite primary key in %s',
                            $accessor, ref($app->$accessor)
                        )
                    ) if @{$app->$accessor->get_pk_fields} > 1;
                }

                foreach (qw(api_available_fields api_available_actions get_product_name api_check_public_id)) {
                    fail(sprintf('%s. Method "%s" not found in %s', $accessor, $_, ref($app->$accessor)))
                      unless $app->$accessor->can($_);
                }

                my $config = $app->get_option('api_configs', {$accessor => {}})->{$accessor} // {};

                my @api_available_fields = $app->$accessor->api_available_fields($config);

                _check_field_types($app, $accessor, $fields, \@api_available_fields);

                _check_field_types_depends($app, $accessor, $fields, \@api_available_fields);

                my %available_fields = map {$_ => TRUE} @api_available_fields;

                if ($app->$accessor->api_can_edit($config)) {
                    my @bad_fields = grep {!$available_fields{$_}} qw(editable_fields),
                      (      $app->$accessor->isa('Application::Model::Page')
                          || $app->$accessor->isa('Application::Model::Block') ? 'status' : ());
                    fail(
                        sprintf(
                            '%s. For use "edit" in %s need add in api_available_fields following fields: %s',
                            $accessor, ref($app->$accessor), join(', ', @bad_fields)
                        )
                    ) if @bad_fields;
                }

                if ($app->$accessor->api_available_actions($config)) {
                    my @bad_fields = grep {!$available_fields{$_}} qw(multistate multistate_name actions);
                    fail(
                        sprintf(
                            '%s. For use "do_action" in %s need add in api_available_fields following fields: %s',
                            $accessor, ref($app->$accessor), join(', ', @bad_fields)
                        )
                    ) if @bad_fields;
                }

                # fix: No tests run for subtest
                pass();
              }
        }

        my @rest_api_multistate_models =
          grep {$app->{$_}->isa('RestApi::MultistateModel')} @rest_api_dbmodels;

        foreach my $accessor (@rest_api_multistate_models) {
            next if defined($ARGV[0]) && $accessor ne $ARGV[0];

            subtest $accessor => sub {
                fail(gettext('%s. Field "fields_depends" not found in %s', $accessor, ref($app->$accessor)))
                  unless exists($app->$accessor->get_model_fields()->{'fields_depends'});

                my $config = $app->get_option('api_configs', {$accessor => {}})->{$accessor} // {};

                if ($app->$accessor->api_available_actions($config)) {
                    my %available_fields = map {$_ => TRUE} $app->$accessor->api_available_fields($config);

                    my @bad_fields = grep {!$available_fields{$_}} qw(multistate multistate_name actions);
                    fail(
                        sprintf(
                            'For use "do_action" in %s need add in api_available_fields following fields: %s',
                            ref($app->$accessor), join(', ', @bad_fields)
                        )
                    ) if @bad_fields;
                }

                pass();
              }
        }

        my @rest_api_simple_models =
          grep {blessed($app->{$_}) && $app->$_->isa('RestApi::SimpleModel')} keys(%$app);

        ok(@rest_api_simple_models, "RestApi::SimpleModels list is not null");
        foreach my $accessor (@rest_api_simple_models) {
            next if defined($ARGV[0]) && $accessor ne $ARGV[0];

            foreach (qw(api_available_fields api_available_actions api_get_all get_product_name api_check_public_id)) {
                fail(sprintf('Method "%s" not found in %s', $_, ref($app->$accessor))) unless $app->$accessor->can($_);
            }
        }

        my @rest_api_multi_models =
          grep {blessed($app->{$_}) && $app->$_->isa('Application::Model::MultiModel')} keys(%$app);

        foreach my $accessor (@rest_api_multi_models) {
            next if defined($ARGV[0]) && $accessor ne $ARGV[0];

            my @models_without_available_fields = ();
            foreach my $model ($app->$accessor->get_models()) {
                my $config = $app->get_option('api_configs', {$model => {}})->{$model} // {};

                my %api_available_fields = map {$_ => TRUE} $app->$model->api_available_fields($config);

                push(@models_without_available_fields, $model) unless $api_available_fields{'available_fields'};
            }

            if (@models_without_available_fields) {
                fail(
                    sprintf(
                        'Multi model "%s" has not field "available_fieds" in following models: %s',
                        ref($app->$accessor), join(', ', @models_without_available_fields)
                    )
                );
            }
        }
    },
    user               => 'yndx-developer',
    do_not_die_on_fail => TRUE,
    locale             => 'C'
);

sub _check_field_types {
    my ($app, $accessor, $fields, $api_available_fields) = @_;

    my @no_type = ();
    foreach my $field_name (@$api_available_fields) {
        if ($exclude_fields->{$field_name}) {
            next;
        }

        my $field_data = $fields->{$field_name};

        fail(sprintf('%s. Field "%s" not found in model "%s"', $accessor, $field_name, ref($app->$accessor)))
          unless $field_data;

        my $type = $field_data->{'type'};
        if (!$type) {
            push(@no_type, $field_name);
        } elsif ($type eq 'array') {

            fail(sprintf '%s. You have to specify "sub_type" for the type="array" (field="%s")', $accessor, $field_name)
              unless $field_data->{'sub_type'};
        }
    }

    fail(
        sprintf(
            '%s. In model "%s" fields: %s have not attribute "type". Example: type => "string"',
            $accessor, ref($app->$accessor), join(', ', @no_type)
        )
    ) if @no_type;
}

sub _check_field_types_depends {
    my ($app, $accessor, $fields, $api_available_fields) = @_;

    my %fields_skip = map {$_ => TRUE} qw(actions available_fields editable_fields);
    my %types_skip  = map {$_ => TRUE} qw(boolean complex number string);

    foreach my $field (@{$api_available_fields}) {
        if ($exclude_fields->{$field}) {
            next;
        }

        if (exists $fields->{$field}->{'type'}) {

            next if exists $fields_skip{$field} || exists $types_skip{$fields->{$field}->{'type'}};

            my $fields_depends =
              (exists $fields->{$field}->{depends_on})
              ? $fields->{$field}->{depends_on}
              : $fields->{$field}->{forced_depends_on};

            if ($fields->{$field}->{'type'} eq 'array') {

                my $type = $fields->{$field}->{'sub_type'};

                next if exists $types_skip{$type} || !defined $fields_depends;

                if (ref($fields_depends) eq 'ARRAY') {

                    foreach (@{$fields_depends}) {

                        if ($_ =~ /\./) {

                            my @split = split(/\./, $_);

                            if (
                                $app->$accessor->can('get_structure_model_accessors')
                                && (   exists $app->$accessor->get_structure_model_accessors()->{$split[0]}
                                    || exists $app->$accessor->get_structure_model_accessors()
                                    ->{$fields->{$field}->{'type'}})
                               )
                            {
                                #Поле с префиксом
                                _check_field_types_depends_first_part($app, $fields, $field, $_);
                            }

                        } else {

                            #Поле без префикса

                            _check_field_types_depends_second_part($app, $accessor, $fields, $_);
                        }
                    }
                } else {
                    fail(
                        sprintf(
'%s. In model "%s" field: %s have reference of depends_on/forced_depends_on not equal ARRAY',
                            $accessor, ref($app->$accessor), $field
                        )
                    );
                }
            } else {

                next unless defined $fields_depends;

                foreach (@{$fields_depends}) {

                    if ($_ =~ /\./) {

                        #Поле с префиксом

                        _check_field_types_depends_first_part($app, $fields, $field, $_);

                    } else {

                        #Поле без префикса

                        _check_field_types_depends_second_part($app, $accessor, $fields, $_);

                    }
                }
            }
        } else {
            warn(
                sprintf(
                    '%s. In model "%s" field: %s have not attribute "type". Example: type => "string"',
                    $accessor, ref($app->$accessor), $field
                )
            );
        }
    }
}

sub _check_field_types_depends_first_part {
    my ($app, $fields, $field, $field_depends) = @_;

    my @split = split(/\./, $field_depends);

    my $related_model_accessor = ($split[0] eq 'pages') ? $fields->{$field}->{type} : $split[0];

    my $field_of_related_model = $app->$related_model_accessor->get_model_fields()->{$split[1]};
    fail(
        sprintf(
            '%s. In model "%s" field: %s have not attribute "type". Example: type => "string"',
            $related_model_accessor, ref($app->$related_model_accessor),
            $split[1]
        )
    ) unless defined $field_of_related_model->{type};

    my $fields_from_new_model = $app->$related_model_accessor->get_model_fields();

    _check_field_types_depends($app, $related_model_accessor, $fields_from_new_model, [$split[1]]);
}

sub _check_field_types_depends_second_part {
    my ($app, $accessor, $fields, $field) = @_;

    fail(
        sprintf(
            '%s. In model "%s" field: %s have not attribute "type". Example: type => "string"',
            $accessor, ref($app->$accessor), $field
        )
    ) unless defined $fields->{$field}->{type};

    _check_field_types_depends($app, $accessor, $fields, [$field]);
}
