package Test::Partner::DB::Mock::db;

use strict;
use warnings FATAL => 'all';

use qbit;

use base ('DBI::db');

use Test::MockObject::Extends::Easy;
use Test::Partner::Utils qw(get_hash_values);

use SQL::Translator;
use SQL::Translator::Parser::QBit;

my %db;
my @locales;

my $inside_brackets;
$inside_brackets = qr{\(((?:(?>[^()]+)|(??{ $inside_brackets }))*)\)};

our $track_sql = sub { };

sub method__connect {
    my ($dbh, $self) = @_;

    $self->{'__DBH__'}{$self->get_key()} = $dbh unless defined($self->{'__DBH__'}{$self->get_key()});

    return FALSE;
}

sub method__get_sql {
    my ($self, $sql_query) = @_;

    return $sql_query;
}

sub method__get_all {
    my ($dbh, $self, $sql_query) = @_;

    $sql_query =~ s/FOR UPDATE//g;
    $sql_query =~ s/= ANY/IN/g;
    $sql_query =~ s/STRAIGHT_JOIN/JOIN/g;
    $sql_query =~ s/$inside_brackets(\s+UNION(?: ALL)?)/$1$2/gs;
    $sql_query =~ s/(UNION(?: ALL)?\s+)$inside_brackets/$1$2/gs;
    $sql_query =~ s/LAST_DAY\((.+?)\)/date($1, 'start of month', '+1 month', '-1 day')/g;

    if (my ($group_by_fields) = $sql_query =~ /GROUP BY ((?:`\w+`(?:, )?)+)/) {
        my @group_by_fields = split(', ', $group_by_fields);

        my ($select_fields) = $sql_query =~ /^SELECT(.*?)FROM/ms;
        $select_fields =~ s/\s+/ /g;
        $select_fields =~ s/^\s+|\s+$//g;
        my %select_fields = reverse map {/(.+) AS (.+)/} split(', ', $select_fields);

        foreach my $group_by_field (@group_by_fields) {
            next unless exists($select_fields{$group_by_field});
            $sql_query =~ s/(GROUP BY (?:`\w+`(?:.`\w+`)?(?:, )?)*)($group_by_field)/$1$select_fields{$group_by_field}/;
        }
    }

    $track_sql->($sql_query);

    my $sth = $dbh->prepare($sql_query) || throw $dbh->errstr() . "\n$sql_query\n";
    $sth->execute();

    my $data = [];
    while (my $data_piece = $sth->fetchrow_hashref()) {
        foreach (keys(%$data_piece)) {
            $data_piece->{$_} = fix_utf($data_piece->{$_});
        }

        push(@$data, $data_piece);
    }

    return $data;
}

sub method__do {
    my ($dbh, $self, $sql_query, @data) = @_;

    $sql_query =~ s/TRUNCATE TABLE/DELETE FROM/g;

    my $sth = $dbh->prepare($sql_query) || die $dbh->errstr();
    $sth->execute(@data);
    return TRUE;
}

sub mock {
    my ($self, $app) = @_;

    $QBit::Application::Model::DB::mysql::Table::ADD_CHUNK = 50;

    foreach my $database (qw(partner_db partner_logs_db banks_db kladr)) {
        $app->$database;
        $app->{$database} = Test::MockObject::Extends::Easy->new($app->{$database});

        $app->$database->mock('_connect', sub {return &method__connect($self, @_)});
        $app->$database->mock('_get_sql', \&method__get_sql);
        $app->$database->mock('_get_all', sub {&method__connect($self, @_); return &method__get_all($self, @_)});
        $app->$database->mock('_do', sub {&method__connect($self, @_); return &method__do($self, @_)});

        $db{$database} = $app->{$database};
        weaken($db{$database});
    }

    @locales = keys %{$app->get_option('locales') // {}};
}

sub insert {
    my ($self, $table, $fields, @row_hashes) = @_;
    @$fields = keys %{$row_hashes[0]} unless $fields;

    $self->do(
        "INSERT INTO $table (" . join(q{, }, @$fields) . ')
           VALUES  ' . join(q{, }, ('(' . join(q{, }, ('?') x @$fields) . ')') x @row_hashes) . '
        ;',
        {},
        map {(get_hash_values($_, $fields)),} @row_hashes
    );
}

sub create_tables {
    my ($self, %db_tables) = @_;

    $self->sqlite_create_function('LAST_INSERT_ID', 0, sub {$self->function_last_insert_id()});
    $self->sqlite_create_function('DATE_FORMAT',    2, sub {$self->function_date_format(@_)});

    foreach my $db_name (keys(%db_tables)) {
        die "Call 'mock' method before the 'create_tables' one.\nNo '$db_name' accessor is registered.\n"
          unless defined($db{$db_name});

        my $translator = SQL::Translator->new();
        my $meta = SQL::Translator::Parser::QBit::generate_meta($db{partner_db}, @{$db_tables{$db_name}});
        SQL::Translator::Parser::QBit::parse($translator, $meta, \@locales);

        foreach my $sql ($translator->translate(producer => 'SQLite')) {
            next if $sql =~ /^CREATE INDEX/;

            # fix default values
            $sql =~ s/timestamp NOT NULL/$& DEFAULT CURRENT_TIMESTAMP/g;
            $sql =~ s/datetime NOT NULL/$& DEFAULT '1970-01-01 03:00:00'/g;

            $sql =~ s/\b$_\b/'$&'/g foreach (qw(limit from to));

            $self->do($sql) || die Dumper $sql;
        }
    }
}

sub function_last_insert_id {
    my ($self) = @_;

    return $self->last_insert_id('', '', '', '');
}

sub function_date_format {
    my ($self, $date, $format) = @_;

    my ($year, $month, $day_of_month, $hour, $minute, $second) =
      ($date =~ /^(\d{4})-(\d{2})-(\d{2})(?: (\d{2}):(\d{2}):(\d{2}))?$/);
    my $formatted_date = $format;
    $formatted_date =~ s/\%Y/$year/g;
    $formatted_date =~ s/\%m/$month/g;
    $formatted_date =~ s/\%d/$day_of_month/g;

    return $formatted_date;
}

TRUE;
