package Test::Partner::Utils;

use Exporter qw(import);

@EXPORT = qw(
  add_bookmaker_filter
  add_roles_rights
  add_templates
  compare_formats
  convert_design_to_object
  convert_design_to_templates
  curdate_add
  curdate_sub
  default_mocks
  eq_or_diff_http_message
  get_bk_data
  get_clickhouse_statistics_test_data
  get_hash_values
  get_internal_context_on_site_campaign_db_filter_fields
  get_internal_search_on_site_campaign_db_filter_fields
  get_last_email
  get_new_an_rtb
  get_new_context_on_site_adblock
  get_new_direct_context
  get_new_direct_search
  get_new_dsp
  get_new_indoor
  get_new_internal_direct_context
  get_new_internal_direct_search
  get_new_internal_mobile_app
  get_new_internal_mobile_rtb
  get_new_internal_rtb
  get_new_mobile_app
  get_new_mobile_rtb
  get_new_outdoor
  get_new_premium
  get_new_rtb
  get_new_site
  get_new_ssp_application
  get_new_ssp_link_mobile_app
  get_new_ssp_mobile_app_rtb
  get_new_ssp_mobile_app_settings
  get_new_stripe
  get_new_user
  get_new_video_an_site
  get_new_video_fullscreen
  get_new_video_inpage
  get_new_video_instream
  get_ports
  get_test_data_and_update_if_needed
  get_test_data_path
  get_user_with_all_rights
  mock_curdate
  mock_time
  request_to_api
  request_to_intapi
  restore_table
  set_bit_by_name
  set_model_status
  set_roles_rights
  set_test_partner_utils_app
  ssp_mock_httpmaas
  ssp_mock_lwp_for_yql
  use_common_blocks_table
  use_perl_do_actions
  use_perl_get_all
  get_hash_from_http_message
  );

use Carp;
use Capture::Tiny ':all';
use File::Path qw(make_path);
use IO::Socket::INET;

use Test::Partner2::Mock qw( mock_http  mock_time  mock_curdate );
use Test::Partner2::Simple qw(need_self_update);
use Test::Partner2::Utils qw(set_locale_in_tests);

use qbit;

use Test::Deep qw(ignore);
use Test::Differences qw( eq_or_diff );

use QBit::WebInterface::FastCGI::Request;

use PiConstants qw(
  $CLICKHOUSE_STATISTICS_PK
  $DO_ACTION_USE_QUEUE_KEY
  $USE_COMMON_BLOCKS_TABLE_KEY
  $USE_JAVA_JSONAPI_KEY
  );

# TODO: выпилить вызовы вне ф-ций - переписать на Test::Partner2::Simple
#       осталось всего 11 тестов: grep -rl  'Test::Partner::Utils' ./t | xargs grep  -L 'Test::Partner2::Simple' | wc -l
mock_http() if $ENV{'mock_http'} || !defined $ENV{'mock_http'};

sub curdate_add {
    my (%opts) = @_;

    return date_add(curdate(), oformat => 'db_time', %opts);
}

sub curdate_sub {
    my (%opts) = @_;

    return date_sub(curdate(), oformat => 'db_time', %opts);
}

sub get_hash_values {
    return map {$_[0]->{$_}} @{$_[1]};
}

my @users = (
    {
        id               => 11,
        login            => 'good',
        client_id        => 111,
        email            => 'good@ya.ru',
        name             => 'I',
        midname          => 'am',
        lastname         => 'good',
        accountant_email => 'good_accountant@ya.ru',
        newsletter       => 0,
    },
    {
        id         => 12,
        login      => 'bad',
        client_id  => 112,
        email      => 'bad@ya.ru',
        name       => 'I',
        midname    => 'am',
        lastname   => 'bad',
        newsletter => 1,
    },
);

my $app;

sub compare_formats {
    my ($got, $expect, $test_name_suffix) = @_;

    # remove get_text before comparison
    my $data = {
        got    => $got,
        expect => $expect
    };

    my $formats = {};
    foreach my $key (keys %$data) {
        foreach my $format (@{$data->{$key}}) {
            map {delete $format->{$_} if $_ =~ /caption/} keys %$format;
            foreach my $site_vertion_format (values %{$format->{settings} // {}}) {
                map {delete $site_vertion_format->{$_} if $_ =~ /caption/} keys %$site_vertion_format;
            }
            $formats->{$format->{id}}->{$key} = $format;
        }
    }

    # compare by id
    my $new_got = [];
    foreach my $format (sort keys %$formats) {
        my ($got_format, $expect_format) = map {$_ // {}} @{$formats->{$format}}{qw( got expect )};

        eq_or_diff(
            $got_format, $expect_format,
            sprintf('block format "%s" %s', $got_format->{id} // 'UNDEF', $test_name_suffix // ''),
            {context => 3}
        );
        push @$new_got, $got_format;
    }

    return $new_got;
}

sub get_new_an_rtb {
    my ($opts) = @_;

    my $rtb = get_new_rtb($opts);

    $rtb->{'alt_width'}  = 240;
    $rtb->{'alt_height'} = 400;

    my $settings = {%$rtb, %$opts};

    if ($settings->{'adfox_block'}) {
        delete($settings->{$_}) foreach qw(
          alternative_code
          alt_width
          alt_height
          );
    }

    return $settings;
}

sub get_new_context_on_site_adblock {
    my ($opts) = @_;

    my $adblock = {
        caption      => 'Test AdBlock',
        mincpm       => 50,
        strategy     => 0,
        page_id      => $opts->{'page_id'},
        direct_block => 'vertical',
        limit        => 5,
    };

    return $adblock;
}

sub get_new_internal_rtb {
    my ($opts) = @_;

    my $rtb = get_new_rtb($opts);

    $rtb->{'blind'} = 2;

    return {%$rtb, %$opts};
}

sub get_new_rtb {
    my ($opts) = @_;
    my @delete_fields = defined($opts->{'delete_fields'}) ? @{delete($opts->{'delete_fields'})} : ();

    my $rtb = {
        site_version => $opts->{'site_version'} // 'desktop',
        campaign_id  => $opts->{'campaign_id'},
        caption      => "RTB block (campaign: $opts->{'campaign_id'})",
        dsp_blocks => defined($opts->{'dsp_blocks'}) ? $opts->{'dsp_blocks'} : ['240x400', '300x600'],
        picategories     => undef,
        alternative_code => '',
        mincpm           => undef,
        media_active     => undef,
        media_blocked    => undef,
        media_cpm        => undef,
        text_active      => undef,
        text_blocked     => undef,
        text_cpm         => undef,
        strategy         => defined($opts->{'strategy'}) ? $opts->{'strategy'} : 1,
        blind            => defined($opts->{'blind'}) ? $opts->{'blind'} : 0,
        (defined($opts->{'dsps'}) ? (dsps => $opts->{'dsps'}) : ()),
        horizontal_align => defined($opts->{'horizontal_align'}) ? $opts->{'horizontal_align'} : 0,
        (defined($opts->{'show_video'}) ? (show_video => $opts->{'show_video'}) : ()),
        design_templates => defined($opts->{'design_templates'})
        ? $opts->{'design_templates'}
        : [
            {
                type                    => 'tga',
                caption                 => 'template',
                is_custom_format_direct => 0,
                page_id                 => $opts->{'campaign_id'},
                block_id                => 0,
                design_settings         => {name => '240x400', limit => 1}
            },
            {
                type            => 'media',
                caption         => 'media design',
                page_id         => $opts->{'campaign_id'},
                block_id        => 0,
                design_settings => {filterSizes => JSON::XS::false, horizontalAlign => JSON::XS::false}
            }
          ],
    };

    if ($rtb->{'strategy'} == 0) {
        $rtb->{'mincpm'} = 100;
    } elsif ($rtb->{'strategy'} == 3) {
        $rtb->{'media_active'}  = 1;
        $rtb->{'media_blocked'} = 1;
        $rtb->{'media_cpm'}     = undef;
        $rtb->{'text_active'}   = 1;
        $rtb->{'text_blocked'}  = 0;
        $rtb->{'text_cpm'}      = 150;
    }

    map {delete($rtb->{$_})} @delete_fields;

    return $rtb;
}

sub get_new_user {
    my (%opts) = @_;

    my $user = shift(@users);

    if (defined($app)) {

        my $cur_user = $app->get_option('cur_user');

        unless ($user) {
            my $tmp_rights = $app->add_tmp_rights(qw(users_view_all));
            my $id = $app->users->get_all(fields => ['id'], order_by => [['id', 1]], limit => 1)->[0]{'id'} + 1;
            undef($tmp_rights);
            $user = {
                id         => $id,
                login      => "login$id",
                client_id  => "$id$id",
                email      => "login$id\@ya.ru",
                name       => "name$id",
                midname    => "midname$id",
                lastname   => "lastname$id",
                newsletter => 0,
            };
        }

        $user = {%$user, %{$opts{'users_add_opts'} // {}}};
        $app->set_cur_user({id => $user->{'id'},});

        {
            $app->api_balance;

            no strict 'refs';
            no warnings 'redefine';
            *{'QBit::Application::Model::API::Yandex::Balance::get_client_id_by_uid'} = sub {
                return [grep {$_->{'id'} == $_[1]} ($user)]->[0]->{'client_id'};
            };

            *{'Application::get_user_lang'} = sub {
                return $app->get_option('locale');
            };

            my $tmp_rights = $app->add_tmp_rights(qw(do_user_action_add));

            $app->users->add(%$user);

            if ($opts{'adfox_id'}) {
                $app->api_adfox;
                *{'Application::Model::API::Yandex::AdFox::get_adfox_user_info'} = sub {
                    return {login => $user->{'login'}};
                };
                $app->users->do_action($user->{'id'}, 'link_adfox_user', %opts);
            }
        }

        if ($opts{'role_id'}) {
            $app->mailer;

            no strict 'refs';
            no warnings 'redefine';
            local (*{'MIME::Lite::send_by_sendmail'})     = sub {TRUE};
            local (*{'MIME::Lite::last_send_successful'}) = sub {TRUE};
            local (*{'Application::get_user_lang'})       = sub {
                return $app->get_option('locale');
            };

            my $tmp_rights = $app->add_tmp_rights(
                qw(do_user_action_set_user_role rbac_user_role_set users_view_field__client_id users_view_all));
            $app->users->do_action($user->{'id'}, 'set_user_role', role_id => $opts{'role_id'}, allow_idm => TRUE);

            undef($tmp_rights);
        }

        $app->{'__CURRENT_USER_RIGHTS__'} = {};
        unless ($opts{'cur_user'}) {
            $app->set_cur_user($cur_user);
        } else {
            $app->set_cur_user($user);
        }
    }

    return $user;
}

sub set_test_partner_utils_app {
    ($app) = @_;
}

my $bk_id = 11;

sub get_bk_data {
    my ($model, $filter, %opts) = @_;

    my $tmp_rights = $model->app->add_all_tmp_rights();

    my $obj = $model->get_all(
        fields => ['*'],
        filter => $filter
    )->[0];

    if ($model->can('_data_for_bk')) {
        #page
        return $model->_data_for_bk($obj, %opts);
    } elsif ($model->can('get_bk_block_data_key')) {
        #block
        return {$model->get_bk_data($obj->{'page_id'})}->{$model->get_bk_block_data_key};
    } elsif ($model->can('_get_data_for_bk')) {
        # dsp
        return $model->_get_data_for_bk($obj);
    } else {
        $model->get_bk_data($obj->{'page_id'}, %opts);
    }
}

sub get_test_data_path {
    my $path = [caller(0)]->[1];
    $path =~ s/\.t$//;
    return $path;
}

sub get_test_data_and_update_if_needed {
    my ($file, $data, %opts) = @_;

    my $path = [caller(0)]->[1];
    $path =~ s/\.t$//;

    my $from = $opts{raw} ? sub {$_[0]} : \&from_json;
    my $to   = $opts{raw} ? sub {$_[0]} : \&to_json;

    my $path_to_data = "$path/$file";
    # Usage: > ./prove *.t :: --self_update 2>&1 | less
    if (defined($data) && need_self_update()) {
        my ($folder) = $path_to_data =~ /^(.+)\/[^\/]+$/;
        make_path($folder) unless -d $folder;
        writefile($path_to_data, $to->($data, pretty => TRUE, %opts));
    }

    return $from->(readfile($path_to_data));
}

sub add_roles_rights {
    my (%roles_rights) = @_;

    my @roles_rights = @{$app->rbac->get_roles_rights()};

    _set_roles_rights(\%roles_rights, \@roles_rights);
}

sub add_templates {
    my @keys    = ();
    my @locales = sort keys %{$app->get_option('locales')};

    my $tmp_rights = $app->add_tmp_rights('do_text_template_add');
    foreach my $key (@keys) {
        $app->text_template->add(
            key     => $key,
            caption => $key,
            content => {map {$_ => "Stub $_"} @locales}
        );
    }
}

sub default_mocks {
    no strict 'refs';
    no warnings 'redefine';
    *{'MIME::Lite::send_by_sendmail'}     = sub {TRUE};
    *{'MIME::Lite::last_send_successful'} = sub {TRUE};
    *{'Application::get_user_lang'}       = sub {return $app->get_option('locale', 'ru');};
}

sub get_last_email {
    my (%opts) = @_;

    $opts{'fields'} //= [qw(body)];
    push(@{$opts{'fields'}}, 'id');

    my $tmp_rights = $app->add_tmp_rights('mailer_view');
    my $email = $app->mailer->get_all(%opts, order_by => [['id', TRUE]], limit => 1)->[0];
    $email //= {};

    if (defined($email->{'body'})) {
        utf8::decode($email->{'body'});
        $email->{'body'} =~ s/..--.*//s;
    }

    return $email;
}

sub get_new_direct_context {
    return (&_get_new_direct, adaptive_width => '', adaptive_height => '');
}

sub get_new_internal_direct_context {
    my ($opts) = @_;

    my @out = _get_new_direct({%$opts, is_internal => 1, type_campaign => 'context'});

    push(@out, adaptive_width => '', adaptive_height => '');

    return @out;
}

sub get_new_direct_search {
    return (&_get_new_direct);
}

sub get_new_internal_direct_search {
    my ($opts) = @_;

    my @out = _get_new_direct({%$opts, is_internal => 1, type_campaign => 'search'});

    return @out;
}

sub get_new_dsp {
    my ($opts) = @_;

    my %dsp = (
        types         => [0, 2],
        short_caption => 'Test DSP',
        display_name  => 'Test DSP',
        url           => 'http://dsp.com/',
    );

    return (%dsp, ($opts ? %$opts : ()));
}

sub get_new_internal_mobile_rtb {
    my ($opts) = @_;

    my $rtb = {
        block_type => $opts->{'block_type'} // 'banner',
        application_id => $opts->{'application_id'},
        caption        => "Internal RTB block (application: $opts->{'application_id'})",
        direct_block   => defined($opts->{'direct_block'}) ? $opts->{'direct_block'} : '320x100',
        media_block    => '320x100',
        dsps => $opts->{'dsps'} // ['1'],
        dsp_blocks => ['320x100', '728x90'],
        articles   => '[]',
        border_type => defined($opts->{'border_type'}) ? $opts->{'border_type'} : 'none',
        border_radius        => undef,
        border_color         => undef,
        favicon              => 1,
        links_underline      => 0,
        sitelinks_color      => '0000CC',
        site_bg_color        => 'FFFFFF',
        title_color          => defined($opts->{'title_color'}) ? $opts->{'title_color'} : '0000CC',
        text_color           => defined($opts->{'text_color'}) ? $opts->{'text_color'} : '000000',
        url_color            => '006600',
        url_background_color => defined($opts->{'url_background_color'}) ? $opts->{'url_background_color'} : undef,
        hover_color          => '0066FF',
        title_font_size      => undef,
        font_family          => undef,
        font_size            => undef,
        mincpm               => undef,
        media_active         => undef,
        media_blocked        => undef,
        media_cpm            => undef,
        text_active          => undef,
        text_blocked         => undef,
        text_cpm             => undef,
        strategy             => defined($opts->{'strategy'}) ? $opts->{'strategy'} : 1,
        no_sitelinks         => 0,
        limit                => undef,
        horizontal_align     => defined($opts->{'horizontal_align'}) ? $opts->{'horizontal_align'} : 0,
        adaptive_height      => '',
        adaptive_width       => '',
        callouts             => defined($opts->{'callouts'}) ? $opts->{'callouts'} : undef,
    };

    if ($rtb->{'block_type'} eq 'interstitial') {
        foreach (
            qw(
            font_size
            title_font_size
            font_family
            url_background_color
            border_radius
            sitelinks_color
            callouts
            )
          )
        {
            $rtb->{$_} = undef;
        }

        foreach (
            qw(
            adaptive_height
            adaptive_width
            limit
            dsp_blocks
            favicon
            media_block
            no_sitelinks
            direct_block
            )
          )
        {
            delete($rtb->{$_});
        }
    }

    if (($rtb->{'direct_block'} // '') eq 'vertical') {
        $rtb->{'limit'}           = 5;
        $rtb->{'font_size'}       = '0.8';
        $rtb->{'title_font_size'} = 1;
        $rtb->{'font_family'}     = 'arial';
    }

    if ($rtb->{'strategy'} == 0) {
        $rtb->{'mincpm'} = 100;
    } elsif ($rtb->{'strategy'} == 3) {
        $rtb->{'media_active'}  = 1;
        $rtb->{'media_blocked'} = 1;
        $rtb->{'media_cpm'}     = undef;
        $rtb->{'text_active'}   = 1;
        $rtb->{'text_blocked'}  = 0;
        $rtb->{'text_cpm'}      = 150;
    }

    if ($rtb->{'border_type'} ne 'none') {
        $rtb->{'border_radius'} = 0;
        $rtb->{'border_color'}  = '0000CC';
    }

    return %$rtb;
}

sub get_new_internal_mobile_app {
    my ($opts) = @_;

    my $mobile_app = {
        store_id    => $opts->{'store_id'},
        block_title => 'Test title',
        comment     => 'Test comment',
        caption     => 'Test caption',
        type        => '1',
    };
    return %$mobile_app;
}

sub get_new_mobile_app {
    my ($opts) = @_;

    $opts //= {};

    my $mobile_app = {
        type => '1',
        %$opts,
    };
    return %$mobile_app;
}

sub get_new_indoor {
    my (%opts) = @_;

    return (
        "address"       => "ул. Поляны, д. 8",
        "business_oid"  => "1132721398",
        "caption"       => "Вива",
        "facility_type" => 6,
        "gps"           => "55.565429,37.556476",
        "login"         => "mocked-indoor-partner",
        %opts,
    );
}

sub get_new_mobile_rtb {
    my ($opts) = @_;

    $opts->{block_type} //= 'banner';

    my $rtb = {
        page_id  => $opts->{'context_page_id'},
        caption  => sprintf('RTB block (campaign: %s)', $opts->{'page_id'} // 'PAGE_ID'),
        mincpm   => undef,
        strategy => 1,
        %$opts,
    };

    if ($rtb->{'strategy'} == 0) {
        $rtb->{'mincpm'} = 100;
    }

    if ($rtb->{'block_type'} eq 'rewarded') {
        $rtb->{'currency_type'}  = $opts->{'currency_type'}  // 'coins';
        $rtb->{'currency_value'} = $opts->{'currency_value'} // 20;
        $rtb->{'callback'}       = $opts->{'callback'};
        $rtb->{'sign'}           = $opts->{'sign'};
    }

    return %$rtb;
}

sub get_new_outdoor {
    my (%opts) = @_;

    return (
        "caption"       => "Вива",
        "facility_type" => 3,
        "login"         => "mocked-outdoor-partner",
        %opts,
    );
}

sub get_new_premium {
    &_get_new_direct;
}

sub get_new_site {
    my ($opts) = @_;

    return (
        domain => 'test.com',
        (defined($opts->{'login_owner'}) ? (login_owner => $opts->{'login_owner'}) : ()),
    );
}

sub get_new_ssp_application {
    my ($opts) = @_;

    return {
        'id'        => 10,
        'bundle_id' => 'com.cleanmaster.mguard.test',
        'store_id'  => 2,
    };
}

sub get_new_ssp_link_mobile_app {
    my ($opts) = @_;

    return {
        'application_id' => 10,
        'source_token'   => ['com.cleanmaster.mguard.test'],
        'hits'           => 101,
        'seller_id'      => 8739185,
    };
}

sub get_new_ssp_mobile_app_settings {
    my ($opts) = @_;

    return (
        'source_id'            => 10,
        'banner_langs'         => ['by', 'en', 'kz', 'ru', 'uk'],
        'behavioral_targeting' => 1,
        'block_title'          => "Реклама test",
        'cpa'                  => 40,
        'excluded_domains'     => [],
        'excluded_phones'      => [],
        'fast_context'         => 0,
        'mirrors'              => ['dsp.yandex.ru'],
        'only_picture'         => 1,
        'owner_id'             => 368880517,
        'page_id'              => 0,
        'seller_id'            => 8739185,
        'view_images'          => 1,
    );
}

sub get_new_ssp_mobile_app_rtb {
    return (
        'articles'      => undef,       # TODO: PI-23148: Перевести на IAB SSP блоки
        'bg_color'      => 'FFFFFF',
        'border_color'  => 'DDDCDA',
        'border_radius' => 0,
        'border_type'   => 'block',
        'campaign_id'   => 0,
        'direct_block'  => '240x400',
        'dsp_blocks' =>
          ['240x400', '300x250', '300x300', '320x100', '320x50', '400x240', '728x90', '320x480', '480x320'],
        'dsps'            => [1],
        'favicon'         => 1,
        'font_family'     => undef,
        'font_size'       => undef,
        'header_bg_color' => 'FFFFFF',
        'hover_color'     => 'DD0000',
        'id'              => 1,
        'limit'           => 1,
        'links_underline' => 0,
        'media_active'    => undef,
        'media_block'     => '240x400',
        'media_blocked'   => undef,
        'media_cpm'       => undef,
        'mincpm'          => '0.8',
        'no_sitelinks'    => 0,
        'site_bg_color'   => 'FFFFFF',
        'sitelinks_color' => '0000CC',
        'strategy'        => 0,
        'text_active'     => undef,
        'text_blocked'    => undef,
        'text_color'      => '000000',
        'text_cpm'        => undef,
        'title_color'     => '0000CC',
        'title_font_size' => undef,
        'url_color'       => '006600',
    );
}

sub get_new_stripe {
    my ($opts) = @_;

    return (
        campaign_id    => $opts->{'campaign_id'},
        title          => '',
        disposition_id => 0,
        type_id        => 1,
        animation      => 1,
    );
}

sub get_new_video_an_site {
    my ($opts) = @_;

    return (
        domain              => 'test.video.com',
        caption             => 'Video site caption',
        login               => $opts->{'login'} // 'Login',
        platform            => $opts->{'platform'} // 3,
        title               => 'Video site title',
        skip_delay          => $opts->{'skip_delay'} // 10,
        skip_time_left_show => $opts->{'skip_time_left_show'} // 1,
        time_left_show      => $opts->{'time_left_show'} // 1,
        vpaid_enabled       => $opts->{'vpaid_enabled'} // 1,
        vpaid_timeout       => $opts->{'vpaid_timeout'} // 500,
        partner_type        => $opts->{'partner_type'} // undef
    );
}

sub mock_api_balance_create_or_update_place {
    unless (ref($app->api_balance) =~ /^T::MO::E::\w+/) {
        $app->api_balance;
        $app->{api_balance} = Test::MockObject::Extends->new($app->{api_balance});
    }
    $app->api_balance->set_always('create_or_update_place', TRUE);
}

sub mock_api_bk_create_or_update_campaigns {
    unless (ref($app->api_bk) =~ /^T::MO::E::\w+/) {
        $app->api_bk;
        $app->{api_bk} = Test::MockObject::Extends->new($app->{api_bk});
    }
    $app->api_bk->mock('create_or_update_campaigns', sub {return [{PageID => $bk_id++}]});
}

sub restore_table {
    my ($dbh, $table_dumps_dir, $table) = @_;

    my $insert = readfile("$table_dumps_dir/$table.sql");

    $insert =~ s/^(.*? VALUES)\s*//;
    my $header = $1;

    chomp($insert);
    $insert =~ s/\(/__BEGIN__/;
    $insert =~ s/\),\s+\(/__SPLITTER__/g;
    $insert =~ s/\),\(/__SPLITTER__/g;
    $insert =~ s/\);$/__END__/;
    $insert =~ s/(?:[^\[\]\{\}A-Za-z0-9\\\.\-\_,`'":;\s]|\\')/A/g;
    $insert =~ s/__BEGIN__/(/;
    $insert =~ s/__SPLITTER__/),(/g;
    $insert =~ s/__END__/);/;
    $insert =~ s/\\"/"/g;

    $dbh->do("DELETE FROM $table");
    foreach ($insert =~ /\(.{1,15000}\)/g) {
        $dbh->do("$header $_") or die $dbh->errstr;
    }
}

sub set_roles_rights {
    my (%roles_rights) = @_;

    _set_roles_rights(\%roles_rights, []);
}

sub set_bit_by_name {
    my ($model, $filter, $multistate_name) = @_;

    my $multistate = $app->$model->get_multistates_by_filter($multistate_name)->[0];

    my $table = $app->$model->partner_db_table();

    my $filter_obj = $app->partner_db->filter($filter);

    $table->edit($filter_obj, {multistate => $multistate});
}

sub _get_new_direct {
    my ($opts) = @_;

    if (defined($opts->{'type_campaign'}) && !in_array($opts->{'type_campaign'}, [qw(context search)])) {
        throw "Unknown type_campaign: '$opts->{'type_campaign'}'";
    }

    my @out = (
        campaign_id => $opts->{'campaign_id'},
        caption     => "Block $opts->{'campaign_id'}",
        (exists($opts->{'comment'}) ? (comment => $opts->{'comment'}) : ()),
        type  => defined($opts->{'type'}) ? $opts->{'type'}  : 'vertical',
        limit => exists($opts->{'limit'}) ? $opts->{'limit'} : 3,
        border_radius        => 0,
        favicon              => 1,
        no_sitelinks         => 0,
        links_underline      => 0,
        border_type          => 'block',
        sitelinks_color      => '0000CC',
        site_bg_color        => 'FFFFFF',
        bg_color             => 'FFFFFF',
        border_color         => 'DDDDDD',
        header_bg_color      => 'FFCC00',
        title_color          => defined($opts->{'title_color'}) ? $opts->{'title_color'} : '2222CC',
        text_color           => defined($opts->{'text_color'}) ? $opts->{'text_color'} : '000000',
        url_color            => '006600',
        hover_color          => '0066FF',
        font_family          => exists($opts->{'font_family'}) ? $opts->{'font_family'} : 'arial',
        font_size            => exists($opts->{'font_size'}) ? $opts->{'font_size'} : '0.8',
        title_font_size      => exists($opts->{'title_font_size'}) ? $opts->{'title_font_size'} : 1,
        url_background_color => defined($opts->{'url_background_color'}) ? $opts->{'url_background_color'} : undef,
    );

    if ($opts->{'type_campaign'} && $opts->{'type_campaign'} eq 'context') {
        push @out, (images_first => 0);
    }

    if ($opts->{'type_campaign'} && $opts->{'type_campaign'} eq 'search' && $opts->{'is_internal'}) {
        push @out, (adaptive_height => '');
    }

    return @out;
}

sub _set_roles_rights {
    my ($roles_rights_hash, $roles_rights_array) = @_;

    my @roles_rights = (
        @$roles_rights_array,
        map {
            my $role_id = $_;
            map {{role_id => $role_id, right => $_}} @{$roles_rights_hash->{$role_id}}
          } keys(%$roles_rights_hash)
    );

    my $tmp_rights = $app->add_tmp_rights('rbac_assign_rigth_to_role');
    $app->rbac->set_roles_rights(\@roles_rights);

    # force check_rights to re-read rights
    delete($app->{'__CURRENT_RIGHTS_USER_ID__'});
    delete($app->get_option('cur_user', {})->{'rights'});
}

sub get_new_video_instream {
    my ($opts) = @_;

    my %set = _get_new_video_block($opts);

    $set{'type'}            = $opts->{'type'} // 2;
    $set{'caption'}         = 'Instream';
    $set{'count_positions'} = $opts->{'count_positions'} // 1;

    return %set;
}

sub get_new_video_inpage {
    my ($opts) = @_;

    my %set = _get_new_video_block($opts);

    $set{'type'}    = 5;
    $set{'caption'} = 'Inpage';
    $set{$_} //= 0 foreach (qw( hide_at_the_end auto_repeat stick_outstream ));

    return %set;
}

sub get_new_video_fullscreen {
    my ($opts) = @_;

    my %set = _get_new_video_block($opts);

    $set{'type'}    = 3;
    $set{'caption'} = 'Fullscreen';

    return %set;
}

sub add_bookmaker_filter {
    my ($app) = @_;
    $app->partner_db->thematic_filters->add(
        {
            id         => 21,
            caption    => 'bookmaker',
            bk_name    => 'bookmaker',
            multistate => 0,
        }
    );
}

sub request_to_fastcgi {
    my (%opts) = @_;

    my (
        $api,  $path_prefix,    $path,         $method,          $ext,  $query_string,
        $mode, $request_method, $content_type, $accept_language, $body, $http_headers
       )
      = @opts{
        qw(app path_prefix path method ext query_string mode request_method content_type accept_language body http_headers)
      };

    $mode //= 'split_answer';

    {
        # чистим заголовки от предыдущих запросов
        delete @ENV{grep {/^HTTP_/} keys %ENV};

        $path_prefix    //= 'api';
        $ext            //= 'json';
        $query_string   //= 'lang=ru';
        $request_method //= 'GET';

        $ENV{'REMOTE_ADDR'}    = '127.0.0.1';
        $ENV{'SERVER_NAME'}    = '127.0.0.1';
        $ENV{'SERVER_PORT'}    = 443;
        $ENV{'REQUEST_METHOD'} = $request_method;
        $ENV{'SCHEME'}         = 'https';
        $ENV{'REQUEST_URI'}    = "/$path_prefix/$path/$method.$ext";
        $ENV{'QUERY_STRING'}   = $query_string;

        $ENV{'HTTP_CONTENT_TYPE'}    = $content_type    if defined($content_type);
        $ENV{'HTTP_ACCEPT_LANGUAGE'} = $accept_language if defined($accept_language);

        if (defined $http_headers) {
            while (my ($k, $v) = each %$http_headers) {
                (my $nk = uc($k)) =~ s/-/_/g;
                $ENV{"HTTP_$nk"} = $v;
            }
        }

        no strict 'refs';
        no warnings 'redefine';

        *{'QBit::Application::set_app_locale'} = sub {
            set_locale_in_tests($api, $_[1]);
        };

        my $request = QBit::WebInterface::FastCGI::Request->new();
        $api->request($request);
    }

    if ($mode eq 'full_answer') {
        $api->run();
    } elsif ($mode eq 'split_answer' || $mode eq 'only_stdout' || $mode eq 'split_json_answer') {

        binmode(STDOUT, ':utf8');
        binmode(STDERR, ':utf8');

        my ($stdout, $stderr, $exit_status) = capture {
            $api->run();
        };

        return ($stdout, $stderr, $exit_status) if $mode eq 'split_answer';

        my @values = split /\n\n/, $stdout, 2;

        $values[1] = from_json($values[1]) if $mode eq 'split_json_answer';

        return @values;
    } else {
        throw sprintf('Unknown mode "%s"', $mode);
    }
}

sub request_to_api {
    my (%opts) = @_;

    no strict 'refs';
    no warnings 'redefine';

    $opts{'app'}->api_blackbox;

    *{'QBit::Application::Model::API::Yandex::BlackBox::oauth'} = sub {
        my ($self, %params) = @_;

        my $user =
          $opts{'app'}->partner_db->users->get_all(fields => [qw(id)], filter => {login => $params{'token'}})->[0];

        return {
            OAuth => {scope   => {content => 'pi:all'}},
            uid   => {content => $user->{'id'}}
        };
    };

    request_to_fastcgi(%opts, path_prefix => 'api');
}

sub request_to_intapi {
    my (%opts) = @_;

    no strict 'refs';
    no warnings 'redefine';

    $opts{'app'}->intapi_acl;

    *{'Application::Model::IntAPI_ACL::check_acl'}      = sub {TRUE};
    *{'Application::Model::IntAPI_ACL::get_acl_cached'} = sub {'[]'};

    my $called;
    *{'QBit::WebInterface::FastCGI::Request::_read_from_stdin'} = sub {
        if ($called) {
            return 0;
        } else {
            ${$_[1]} = $opts{'body'};
            $called = 1;
            return length($opts{'body'});
        }
    };

    if (defined($opts{'body'})) {
        *{'IntAPI::Method::get_body'} = sub {$opts{'body'}};
    }

    request_to_fastcgi(%opts, path_prefix => 'intapi');
}

sub _get_new_video_block {
    my ($opts) = @_;

    return (
        'dsps' => $opts->{'dsps'} // [
            {
                'show_count' => 12,
                'dsp_id'     => '1',
                'interval'   => 1800
            }
        ],
        'category_id' => $opts->{'category_id'} // 0,
        'strategy'    => $opts->{'strategy'}    // 0,
        'mincpm' => exists $opts->{'mincpm'} ? $opts->{'mincpm'} : 1,
        'video_active'              => $opts->{'video_active'}              // undef,
        'video_blocked'             => $opts->{'video_blocked'}             // undef,
        'video_cpm'                 => $opts->{'video_cpm'}                 // undef,
        'video_performance_active'  => $opts->{'video_performance_active'}  // undef,
        'video_performance_blocked' => $opts->{'video_performance_blocked'} // undef,
        'video_performance_cpm'     => $opts->{'video_performance_cpm'}     // undef,
        'adfox_block'               => '0',
        'blind'                     => $opts->{'blind'}                     // 1,
        'max_duration'              => $opts->{'max_duration'}              // 60,
        'geo'                       => '[]',
        'page_id'                   => $opts->{'page_id'},
        'comment'                   => 'Test',
        'alternative_code' => exists $opts->{'alternative_code'} ? $opts->{'alternative_code'} : '',
        (exists $opts->{'geo'} ? (geo => $opts->{'geo'}) : ())
    );
}

sub get_clickhouse_statistics_test_data {
    my ($app, %opts) = @_;

    my ($pk_data, $accessors) = @opts{qw( pk_data  accessors )};

    my $clichouse_ststistics_fields = $app->clickhouse_db->statistics->fields();

    if ($accessors) {
        $accessors = [map {$app->$_->product->accessor} @$accessors];
    } else {
        my ($product_id_field) = grep {$_->{name} eq 'product_id'} @$clichouse_ststistics_fields;
        $accessors = $product_id_field->{values};
    }

    my %pk = map {$_ => 0} @$CLICKHOUSE_STATISTICS_PK;

    my $data = [];
    foreach my $accessor (@$accessors) {
        foreach my $pk_row (@$pk_data) {
            push @$data,
              {
                dt => '1970-01-01',
                %pk,
                (
                    map {$_ => _get_value_by_field_name($_)}
                    sort grep {!exists($pk{$_})} map {$_->name} @$clichouse_ststistics_fields
                ),
                %$pk_row,
                product_id => $accessor,
              };
        }
    }

    return $data;
}

sub _get_value_by_field_name {
    my ($field) = @_;

    my $res = 0;
    map {$res += ord($_)} split(//, $field);

    return $res;
}

sub get_user_db_filter_fields {
    my ($app) = @_;

    my $roles = [map {{'label' => ignore(), 'id' => $_->{'id'}, 'key' => "id$_->{'id'}",}} @{$app->rbac->get_roles()}];

    return {
        'subfields' => {
            'accountant_email' => {
                'label' => ignore(),
                'type'  => 'text'
            },
            'adfox_id' => {
                'label' => ignore(),
                'type'  => 'number'
            },
            'business_unit' => {'type' => 'boolean'},
            'client_id'     => {
                'label' => ignore(),
                'type'  => 'number'
            },
            'contract_id' => {
                'label' => ignore(),
                'type'  => 'contractnumber'
            },
            'current_currency' => {'type' => 'text'},
            'domain_login'     => {'type' => 'text', 'label' => ignore()},
            'email' => {
                'label' => ignore(),
                'type'  => 'text'
            },
            'has_business_rule'   => {'type' => 'boolean', 'label' => ignore(),},
            'has_common_offer'    => {'type' => 'boolean'},
            'has_tutby_agreement' => {'type' => 'boolean'},
            'id'                  => {
                'label' => ignore(),
                'type'  => 'number'
            },
            'is_adfox_partner' => {'type' => 'boolean'},
            'is_tutby'         => {'type' => 'boolean'},
            'is_yandex'        => {'type' => 'boolean'},
            'lastname'         => {
                'label' => ignore(),
                'type'  => 'text'
            },
            'login' => {
                'label' => ignore(),
                'type'  => 'login'
            },
            'moderation_reason' => {
                'label'  => ignore(),
                'type'   => 'dictionary',
                'values' => [
                    {
                        'id'    => 1,
                        'key'   => 'id1',
                        'label' => ignore(),
                    }
                ],
            },
            'multistate' => {
                'label'  => ignore(),
                'type'   => 'multistate',
                'values' => {
                    'blocked'                     => ignore(),
                    'contacts_provided'           => ignore(),
                    'need_create_in_banner_store' => ignore(),
                    'need_yan_contract'           => ignore(),
                },
            },
            'name' => {
                'label' => ignore(),
                'type'  => 'text'
            },
            'need_to_email_processing'  => {'type' => 'boolean'},
            'newsletter'                => {'type' => 'boolean'},
            'no_stat_monitoring_emails' => {'type' => 'boolean'},
            'phone'                     => {
                'label' => ignore(),
                'type'  => 'text'
            },
            'role_id' => {
                'values' => $roles,
                'label'  => ignore(),
                'type'   => 'dictionary'
            },
            'self_employed' => {'type' => 'text'},
            'user_type'     => {
                'label'  => ignore(),
                'type'   => 'dictionary',
                'values' => [
                    {'id' => 1, 'key' => 'id1', 'label' => ignore(),},
                    {'id' => 2, 'key' => 'id2', 'label' => ignore(),},
                    {'id' => 3, 'key' => 'id3', 'label' => ignore(),},
                    {'id' => 4, 'key' => 'id4', 'label' => ignore(),},
                    {'id' => 5, 'key' => 'id5', 'label' => ignore(),},
                    {'id' => 6, 'key' => 'id6', 'label' => ignore(),},
                ],
            },
        },
        'label' => ignore(),
        'type'  => 'subfilter'
    };
}

sub get_internal_context_on_site_campaign_db_filter_fields {
    &_get_internal_campaign_db_filter_fields
}

sub get_internal_search_on_site_campaign_db_filter_fields {
    &_get_internal_campaign_db_filter_fields
}

sub _get_internal_campaign_db_filter_fields {
    my ($app) = @_;

    my $user_db_filter_fields = get_user_db_filter_fields($app);

    return {
        'mirror' => {
            'subfields' => {
                'domain' => {
                    'label' => ignore(),
                    'type'  => 'text'
                },
                'campaign_id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                }
            },
            'label' => ignore(),
            'type'  => 'subfilter'
        },
        'page_id' => {
            'label' => ignore(),
            'type'  => 'number'
        },
        'multistate' => {
            'values' => {
                'balance_registered' => ignore(),
                'check_statistics'   => ignore(),
                'deleted'            => ignore(),
                'read_only'          => ignore(),
                'stopped'            => ignore(),
                'testing'            => ignore(),
                'protected'          => ignore(),
                'working'            => ignore(),
                'need_update'        => ignore(),
                'updating'           => ignore()
            },
            'label' => ignore(),
            'type'  => 'multistate'
        },
        'domain' => {
            'subfields' => {
                'domain' => {
                    'label' => ignore(),
                    'type'  => 'text'
                },
                'id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                },
            },
            'label' => ignore(),
            'type'  => 'subfilter'
        },
        'site_id' => {
            'label' => ignore(),
            'type'  => 'number'
        },
        'all_domain' => {
            'label' => ignore(),
            'type'  => 'domain_mirror'
        },
        'managers' => {
            'subfields' => {
                'manager_id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                },
                'page_id' => {
                    'label' => ignore(),
                    'type'  => 'number'
                },
                'manager' => $user_db_filter_fields,
            },
            'label' => ignore(),
            'type'  => 'subfilter'
        },
        'id' => {
            'label' => ignore(),
            'type'  => 'number'
        },
        'caption' => {
            'label' => ignore(),
            'type'  => 'text'
        },
        'domain_id' => {
            'label' => ignore(),
            'type'  => 'number'
        },
        'update_time' => {
            'label' => ignore(),
            'type'  => 'date'
        },
        'send_time' => {
            'label' => ignore(),
            'type'  => 'date'
        },
        'metrica_counters' => {'type' => 'text'},
        'tier'             => {
            'values' => [
                {
                    'label' => 1,
                    'id'    => 1,
                    'key'   => 'id1'
                },
                {
                    'label' => 2,
                    'id'    => 2,
                    'key'   => 'id2'
                },
                {
                    'label' => 3,
                    'id'    => 3,
                    'key'   => 'id3'
                },
                {
                    'label' => 4,
                    'id'    => 4,
                    'key'   => 'id4'
                }
            ],
            'label' => 'Tier',
            'type'  => 'dictionary'
        },
        'create_date' => {
            'label' => ignore(),
            'type'  => 'date'
        },
        'creator_id' => {
            'label' => ignore(),
            'type'  => 'number'
        },
        'domain_text' => {
            'label' => ignore(),
            'type'  => 'text'
        },
        'caption_text' => {
            'label' => ignore(),
            'type'  => 'text'
        },
    };
}

sub convert_design_to_object {
    my ($design, $design_id, $horizontal_align) = @_;
    if (defined $design_id) {
        my $design_from_json = from_json("{" . $design . "}");
        return +{
            $design_id => {name => "default template", design => $design_from_json},
            0          => {
                design => {
                      horizontalAlign => ($horizontal_align // $design_from_json->{horizontalAlign})
                    ? JSON::XS::true
                    : JSON::XS::false
                },
                name => "",
                type => "media"
            }
        };
    } else {
        return +{0 => {name => "", design => from_json("{" . $design . "}")}};
    }
}

sub convert_design_to_templates {
    my ($design, $template_id) = @_;
    $template_id //= 2;
    return +{
        $template_id => {name => "default template", design => from_json("{" . $design . "}")},
        $template_id +
          1 => {name => "default template media", type => "media", design => from_json("{" . $design . "}")}
    };
}

sub ssp_mock_httpmaas($) {
    my ($app) = @_;

    $app->api_http_maas;
    $app->{api_http_maas} = Test::MockObject::Extends->new($app->api_http_maas)->mock(
        'get_apps_from_yt',
        sub {
            my ($self, $missing_app_list) = @_;
            my %app_list;
            foreach my $store_app_list (values %$missing_app_list) {
                foreach my $app (@{$store_app_list}) {
                    $app_list{$app->{source_app}} = {
                        bundle   => $app->{source_app},
                        app_id   => (2 == $app->{store_id} ? 'apple.' : 'google.') . $app->{source_app} . '.xxx',
                        app_name => 'app name',
                        app_os   => $app->{store_id},
                      }
                      unless exists $app_list{$app->{source_app}};
                }
            }
            return [map {$app_list{$_}} sort keys %app_list];
        }
    );
}

sub ssp_mock_lwp_for_yql($$) {
    my ($DT_EPOCH, $ssp_type) = @_;
    {
        no strict 'refs';
        no warnings 'redefine';
        *{'LWP::UserAgent::request'} = sub {
            my ($self, $url) = @_;

            my $r = HTTP::Response->new(200);
            $r->request(HTTP::Request->new());

            my $mocked_yt_response = '';

            if ('mobile-app' eq $ssp_type) {
                $mocked_yt_response =
qq([["$DT_EPOCH"], ["7944603"], ["159101"], ["com.alterego.skazka.littlekids"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-media"], ["11"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["com.alterego.skazka.littlekids_t1"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-media"], ["33"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["com.alterego.skazka.littlekids_t2"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-media"], ["44"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["1084196918"], ["1084196918"], ["iOS"], ["app-media"], ["22"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["apple.1084196918.xxx"], ["apple.1084196918.xxx"], ["iOS"], ["app-media"], ["22"]]);
            } elsif ('web' eq $ssp_type) {
                $mocked_yt_response =
qq([["$DT_EPOCH"], ["7944603"], ["159101"], ["as dfasdfsadf"], ["as dfasdfsadfasd"], ["-"], ["media"], ["11"]]
[["$DT_EPOCH"], ["7944603"], ["159101"], ["d3.ru"], ["d3.ru"], ["-"], ["media"], ["11"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["www.mail.ru/main"], ["http://www.mail.ru/main"], ["-"], ["media"], ["33"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["mail.ru"], ["mail.ru"], ["-"], ["media"], ["22"]]);
            } elsif ('video' eq $ssp_type) {
                $mocked_yt_response =
qq([["$DT_EPOCH"], ["7944603"], ["159101"], ["com.alterego.skazka.littlekids"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-video"], ["11"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["com.alterego.skazka.littlekids_t1"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-video"], ["33"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["com.alterego.skazka.littlekids_t2"], ["com.alterego.skazka.littlekids"], ["Android"], ["app-video"], ["44"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["1084196918"], ["1084196918"], ["iOS"], ["app-video"], ["22"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["apple.1084196918.xxx"], ["apple.1084196918.xxx"], ["iOS"], ["app-video"], ["22"]]
[["$DT_EPOCH"], ["7952356"], ["159101"], ["apple.1084196918.xxxx"], ["apple.1084196918.xxxx"], ["iOS"], ["app-media"], ["23"]]);
            }

            my $json = {
                data => [
                    {
                        Write => [
                            {
                                Data => from_jsonl $mocked_yt_response,
                                Type => [
                                    undef,
                                    [
                                        undef,
                                        [
                                            map {[$_]} (
                                                'UpdateTime',   'SSPID',    'PageID', 'ExportToken',
                                                'ExportDomain', 'ExportOS', 'Type',   'TotalHits'
                                            )
                                        ]
                                    ]
                                ]
                            }
                        ]
                    }
                ]
            };

            if ($url->uri->as_string =~ /\/operations\?/) {
                $r->content('{"id":12345678}');
            } elsif ($url->uri->as_string =~ /\/operations\/\d+$/) {
                $r->content('{"status":"COMPLETED"}');
            } elsif ($url->uri->as_string =~ /operations\/\d+\/results\?/) {
                $r->content(to_json $json);
            } else {
                die sprintf("do not know this url: %s", $url->uri->as_string);
            }

            return $r;
        };
    }
}

sub get_ports {
    my (%opts) = @_;

    my $count = $opts{'count'} // 1;

    my $min_port = $opts{'min'} // 49152;
    my $max_port = $opts{'max'} // 65535;
    my $possible_ports_count = $max_port - $min_port;

    my %ports = ();
    while ($count) {
        my $port = int($min_port + rand() * $possible_ports_count);

        my $sock = IO::Socket::INET->new(
            LocalAddr => 'localhost',
            LocalPort => $port,
            Proto     => 'tcp',
            ReuseAddr => TRUE,
        );

        if ($sock) {
            $ports{$port} = $sock;

            $count--;
        }
    }

    return [map {close $ports{$_}; $_} keys(%ports)];
}

sub get_user_with_all_rights {
    my ($app, $id, @opts) = @_;
    my $tmp_rights = $app->add_all_tmp_rights();
    return $app->users->get_by_uid($id, @opts);
}

sub eq_or_diff_http_message {
    my ($msg1, $msg2, $test_name, $opts) = @_;

    for my $msg ($msg1, $msg2) {
        if (!ref $msg) {
            utf8::encode($msg) if utf8::is_utf8($msg);
            $msg = HTTP::Message->parse($msg);
        }
        my $content = $msg->decoded_content;
        if ($content) {
            try {
                $msg->content(to_json(from_json($content), pretty => TRUE));
            }
            catch {
                try {
                    $msg->content(
                        {
                            map {@_ = split '=', chomp($_); $_[0] => $_[1]}
                              split('&', $content)
                        }
                    );
                }
            };
        }
    }
    eq_or_diff($msg1, $msg2, $test_name, $opts);
}

sub get_hash_from_http_message {
    my ($msg) = @_;
    if (!ref $msg) {
        utf8::encode($msg) if utf8::is_utf8($msg);
        $msg = HTTP::Message->parse($msg);
    }
    return from_json($msg->decoded_content);
}

sub use_perl_do_actions {
    my ($app, $model) = @_;

    $app->partner_db->kv_store->edit($DO_ACTION_USE_QUEUE_KEY, {value => {JSON_SET => ['value', \"\$.$model", \0]}});
}

sub use_perl_get_all {
    my ($app, $model) = @_;

    $app->partner_db->kv_store->edit($USE_JAVA_JSONAPI_KEY, {value => {JSON_SET => ['value', \"\$.$model", \0]}});
}

sub use_common_blocks_table {
    my ($app) = @_;

    $app->kv_store->set(
        $USE_COMMON_BLOCKS_TABLE_KEY,
        to_json(
            {
                internal_context_on_site_rtb => \1,
                mobile_app_rtb               => \1,
                context_on_site_content      => \1,
                internal_mobile_app_rtb      => \1
            }
        )
    );
}

sub set_model_status {
    my ($app, $requests) = @_;

    foreach my $request (@$requests) {
        my ($model, $filter, $order_by, $limit, $status) = @$request{qw(model filter order_by limit status)};

        my $table = $app->$model->partner_db_table();

        my $items = $table->get_all(
            fields   => $table->primary_key(),
            filter   => $app->partner_db->filter($filter),
            order_by => $order_by,
            limit    => $limit,
        );

        if (@$items == 0) {
            print STDERR sprintf("Nothing found for model '%s'\n", $model);

            next;
        } elsif (@$items > 1) {
            throw 'Expected one item';
        }

        my $multistate = $app->$model->get_multistates_by_filter($status)->[0];

        unless (defined($multistate)) {
            throw 'State is incorrect';
        }

        $table->edit($items->[0], {multistate => $multistate});
    }
}

TRUE;
