#!/usr/bin/perl

use Test::More tests => 12;
use Test::Differences qw(eq_or_diff);

use qbit;

use Utils::DB qw(fields_to_complex_filter);

my $FIXTURES = [
    {
        data     => [],
        for_db   => TRUE,
        expected => ['AND', [\undef]]
    },
    {
        data     => [],
        for_db   => FALSE,
        expected => [undef]
    },
    {
        data   => [{page_id => 1}],
        for_db => TRUE,
        expected => ['OR', [['AND', [['page_id', '=', \1]]]]]
    },
    {
        data   => [{page_id => 1}],
        for_db => FALSE,
        expected => ['OR', [['AND', [['page_id', '=', 1]]]]]
    },
    {
        data   => [{page_id => 1, id => 2}],
        for_db => TRUE,
        expected => ['OR', [['AND', [['id', '=', \2], ['page_id', '=', \1],]]]]
    },
    {
        data   => [{page_id => 1, id => 2}],
        for_db => FALSE,
        expected => ['OR', [['AND', [['id', '=', 2], ['page_id', '=', 1]]]]]
    },
    {
        data     => [{page_id => 1, block_id => 1}, {page_id => 2, block_id => 1}, {page_id => 2, block_id => 2}],
        for_db   => TRUE,
        expected => [
            'OR',
            [
                ['AND', [['block_id', '=', \1], ['page_id', '=', \1]]],
                ['AND', [['block_id', '=', \1], ['page_id', '=', \2]]],
                ['AND', [['block_id', '=', \2], ['page_id', '=', \2]]],
            ]
        ]
    },
    {
        data     => [{page_id => 1, block_id => 1}, {page_id => 2, block_id => 1}, {page_id => 2, block_id => 2}],
        for_db   => FALSE,
        expected => [
            'OR',
            [
                ['AND', [['block_id', '=', 1], ['page_id', '=', 1]]],
                ['AND', [['block_id', '=', 1], ['page_id', '=', 2]]],
                ['AND', [['block_id', '=', 2], ['page_id', '=', 2]]],
            ]
        ]
    },
    {
        data     => [{campaign_id => 1, id => 1}, {campaign_id => 2, id => 1}, {campaign_id => 2, id => 2}],
        for_db   => TRUE,
        expected => [
            'OR',
            [
                ['AND', [['campaign_id', '=', \1], ['id', '=', \1]]],
                ['AND', [['campaign_id', '=', \2], ['id', '=', \1]]],
                ['AND', [['campaign_id', '=', \2], ['id', '=', \2]]],
            ]
        ]
    },
    {
        data     => [{campaign_id => 1, id => 1}, {campaign_id => 2, id => 1}, {campaign_id => 2, id => 2}],
        for_db   => FALSE,
        expected => [
            'OR',
            [
                ['AND', [['campaign_id', '=', 1], ['id', '=', 1]]],
                ['AND', [['campaign_id', '=', 2], ['id', '=', 1]]],
                ['AND', [['campaign_id', '=', 2], ['id', '=', 2]]],
            ]
        ]
    },
    {
        data => [
            {page_id => 1, block_id => 1, dsp_id => 1},
            {page_id => 2, block_id => 1, dsp_id => 2},
            {page_id => 2, block_id => 2, dsp_id => 2}
        ],
        for_db   => TRUE,
        expected => [
            'OR',
            [
                ['AND', [['block_id', '=', \1], ['dsp_id', '=', \1], ['page_id', '=', \1]]],
                ['AND', [['block_id', '=', \1], ['dsp_id', '=', \2], ['page_id', '=', \2]]],
                ['AND', [['block_id', '=', \2], ['dsp_id', '=', \2], ['page_id', '=', \2]]],
            ]
        ]
    },
    {
        data => [
            {page_id => 1, block_id => 1, dsp_id => 1},
            {page_id => 2, block_id => 1, dsp_id => 2},
            {page_id => 2, block_id => 2, dsp_id => 2}
        ],
        for_db   => FALSE,
        expected => [
            'OR',
            [
                ['AND', [['block_id', '=', 1], ['dsp_id', '=', 1], ['page_id', '=', 1]]],
                ['AND', [['block_id', '=', 1], ['dsp_id', '=', 2], ['page_id', '=', 2]]],
                ['AND', [['block_id', '=', 2], ['dsp_id', '=', 2], ['page_id', '=', 2]]],
            ]
        ]
    },
];

foreach my $test (@$FIXTURES) {
    my ($data, $error);
    try {
        $data = fields_to_complex_filter($test->{'data'}, for_db => $test->{'for_db'});
    }
    catch {
        $error = shift->message;
    };

    if ($test->{'error'}) {
        is($error, $test->{'error'}, $test->{'name'});
    } else {
        eq_or_diff($data, $test->{'expected'}, $test->{'name'});
    }
}
