use strict;
use warnings FATAL => 'all';

use File::Temp qw(tempdir);
use Storable;
use Test::Deep;
use Test::Differences;
use Test::More tests => 7;

use Test::Partner2::Mock qw(mock_subs);
use Test::Partner2::Simple;
use Utils::LogParser::Checks;
use Utils::Logger;

my $logs_path = tempdir(CLEANUP => 1);
*Utils::LogParser::Checks::DISK_STORAGE_DIRECTORY_PATH = \$logs_path;
*Utils::LogParser::Checks::LOGS_DIRECTORY_PATH         = \$logs_path;

my $mocked_positions = {
    "$logs_path/partners.yandex.ru/partner2-cron-internal_adv_net-regular_update_statistics.err"        => 1000,
    "$logs_path/partners.yandex.ru/partner2-cron-stat_direct-regular_update_statistics.err"             => 1000,
    "$logs_path/partners.yandex.ru/partner2-cron-adv_net_video-regular_update_statistics_for_month.log" => 1000,
    "$logs_path/partners.yandex.ru/var/log/nginx/frontend-node-v2call.log"                              => 1000,
    "$logs_path/partners.yandex.ru/var/log/kern.log"                                                    => 1000,
};

my $mocked_results = {'mocked_log_name.mocked_check_name' => {value => 5, solomon => {mocked_label => 'value'}}};

my $mocked_new_positions = {
    "$logs_path/partners.yandex.ru/partner2-cron-internal_adv_net-regular_update_statistics.err"        => 1500,
    "$logs_path/partners.yandex.ru/partner2-cron-stat_direct-regular_update_statistics.err"             => 1500,
    "$logs_path/partners.yandex.ru/partner2-cron-adv_net_video-regular_update_statistics_for_month.log" => 1500,
    "$logs_path/partners.yandex.ru/var/log/nginx/frontend-node-v2call.log"                              => 1500,
    "$logs_path/partners.yandex.ru/var/log/kern.log"                                                    => 1500,
};

mock_subs(
    {
        'Utils::LogParser::process_logs' => sub {
            my ($logs, $checks, $positions) = @_;

            my $expected_logs = {
                cron_internal_an_regular_update_statistics => {
                    path =>
                      "$logs_path/partners.yandex.ru/partner2-cron-internal_adv_net-regular_update_statistics.err",
                },
                cron_stat_direct_regular_update_statistics =>
                  {path => "$logs_path/partners.yandex.ru/partner2-cron-stat_direct-regular_update_statistics.err",},
                cron_adv_net_video_regular_update_statistics_for_month => {
                    path =>
"$logs_path/partners.yandex.ru/partner2-cron-adv_net_video-regular_update_statistics_for_month.log",
                },
                kern_log           => {path => "$logs_path/kern.log"},
                partner2_rosetta   => {path => "$logs_path/partners.yandex.ru/partner2_rosetta.log",},
                jsonapi_access_log => {
                    path         => "$logs_path/nginx/jsonapi.partner2.yandex.com.access-post.log",
                    preparse     => ignore(),
                    groups_regex => ignore(),
                    field_labels => ignore(),
                },
                java_jsonapi_access_log => {
                    path         => "$logs_path/nginx/java.jsonapi.partner2.yandex.access-post.log",
                    preparse     => ignore(),
                    groups_regex => ignore(),
                    field_labels => ignore(),
                },
                access_log => {
                    path         => "$logs_path/nginx/partner2.yandex.access-post.log",
                    preparse     => ignore(),
                    groups_regex => ignore(),
                    field_labels => ignore(),
                },
                v2call_log => {
                    path         => "$logs_path/nginx/frontend-node-v2call.log",
                    preparse     => ignore(),
                    field_labels => ignore(),
                },
            };

            my $expected_check_names = [
                'access_log.rps',                 'access_log.status',
                'access_log.timing',              'broken_pipe',
                'fix_statistics',                 'jsonapi_access_log.rps',
                'jsonapi_access_log.status',      'jsonapi_access_log.timing',
                'java_jsonapi_access_log.rps',    'java_jsonapi_access_log.status',
                'java_jsonapi_access_log.timing', 'oom_killer',
                'v2call_log.rps',                 'v2call_log.status',
                'v2call_log.timing',              'video_pages_update_lost_statistics'
            ];

            cmp_deeply($logs,           $expected_logs,              'list of checked logs');
            cmp_deeply([keys %$checks], bag(@$expected_check_names), 'check names');
            cmp_deeply($positions,      $mocked_positions,           'positions from state file');

            return ($mocked_results, $mocked_new_positions);
        },

        'Utils::LogParser::Checks::send_to_graphite' => sub {
            my %data = @_;

            is($data{interval}, 'five_min', 'graphite interval');
            cmp_deeply({$data{path} => {value => $data{value}, solomon => $data{solomon}}},
                $mocked_results, 'results sent to graphite');
        },
    }
);

run_tests(
    sub {
        my ($app) = @_;

        Utils::Logger::init();

        Storable::store($mocked_positions, "$logs_path/log_parser_state.bin");

        Utils::LogParser::Checks::parse_logs($app);

        is_deeply(Storable::retrieve("$logs_path/log_parser_state.bin"), $mocked_new_positions,
            'positions after check');
    },
    dont_create_database => 1,
    do_not_die_on_fail   => 1,
);
