use strict;
use warnings FATAL => 'all';

use Test::More tests => 62;
use Test::Differences;
use File::Slurp qw(write_file);
use File::Temp qw(tempdir);

use QBit::Application;
use Utils::LogParser;
use Utils::Logger;

my $log_content_before = <<LOG;
2017-05-07 11:45:05 28272 INFO [Cron system - check_impossible_multistates] START
2017-05-07 11:45:17 28272 INFO [Cron system - check_impossible_multistates] END (11.82 sec)
2017-05-08 11:45:05 30438 INFO [Cron system - check_impossible_multistates] START
2017-05-08 11:45:16 30438 INFO [Cron system - check_impossible_multistates] END (11.09 sec)
2017-05-09 11:45:05 22292 INFO [Cron system - check_impossible_multistates] START
2017-05-09 11:45:16 22292 INFO [Cron system - check_impossible_multistates] END (10.93 sec)
2017-05-10 11:45:05 19597 INFO [Cron system - check_impossible_multistates] START
2017-05-10 11:45:10 19597 INFO [Cron system - check_impossible_multistates] END (4.99 sec)
LOG

my $log_content_update = <<LOG;
2017-05-11 11:45:06 17562 INFO [Cron system - check_impossible_multistates] START
2017-05-11 11:45:12 17562 INFO [Cron system - check_impossible_multistates] END (6.60 sec)
LOG

my $log_content_full = $log_content_before . $log_content_update;

my $logs_path = tempdir(CLEANUP => 1);
my $mocked_log_path = "$logs_path/mocked_log.log";

open(my $sh, '<', \$log_content_full) or die "Failed to open scalar with log content: $!";

my $logs = {mocked_log => {path => $mocked_log_path,}};

my $before_checks = {
    lines_count => {
        apply_to  => 'mocked_log',
        regex     => qr//,
        check_sub => \&inc,
        final_sub => total_count_final_sub(8),
    },
};

my $update_checks = {
    lines_count => {
        apply_to  => 'mocked_log',
        regex     => qr//,
        check_sub => \&inc,
        final_sub => total_count_final_sub(2),
    },
};

my $full_checks = {
    callback_args => {
        apply_to  => 'mocked_log',
        regex     => qr//,
        check_sub => sub {
            my ($log, $check, $temp_data, $line) = @_;

            is($log->{name},       'mocked_log',              'log name');
            is($log->{path},       $logs->{mocked_log}{path}, 'log path');
            is($check->{apply_to}, 'mocked_log',              'check apply_to');
            is($temp_data->{mocked_log} // 0, $sh->input_line_number, 'check data');

            is($line, readline($sh), 'line content');

            $temp_data->{$log->{name}} += 1;
        },
        final_sub => sub {
            my ($check, $logs, $temp_data, $results) = @_;

            is($check->{apply_to},       'mocked_log', 'final check apply_to');
            is($temp_data->{mocked_log}, 10,           'final check data');

            #is_deeply($log,   $logs->{mocked_log},           'log content');
            #is_deeply($check, $full_checks->{callback_args}, 'check content');

            $results->{"mocked_log.callback_args"} = $temp_data->{mocked_log};
        },
    },
    starts => {
        apply_to  => 'mocked_log',
        regex     => qr/\bSTART\b/,
        check_sub => \&inc,
        final_sub => total_count_final_sub(5),
    },
    applies_to_nothing => {
        apply_to  => 'nonexistent_log',
        regex     => qr//,
        check_sub => sub {
            fail('checking nonexistent log');
        },
        final_sub => sub {
            pass('finalize runs anyways');
        },
    }
};

sub inc {
    my ($log, $check, $temp_data, $line) = @_;
    $temp_data->{$log->{name}} += 1;
}

sub total_count_final_sub {
    my ($count) = @_;
    return sub {
        my ($check, $logs, $temp_data, $results) = @_;
        is($temp_data->{mocked_log}, $count, 'before lines count');
        $results->{"mocked_log.$check->{name}"} = $temp_data->{mocked_log};
    };
}

Utils::Logger::init();

write_file($mocked_log_path, $log_content_before) or die "Failed to write to file $mocked_log_path";

my ($results, $positions) = Utils::LogParser::process_logs($logs, $before_checks, {$mocked_log_path => -1});

is_deeply($results, {'mocked_log.lines_count' => 8}, 'results before');
is_deeply($positions, {$mocked_log_path => length($log_content_before),}, 'positions before');

write_file($mocked_log_path, {append => 1}, $log_content_update) or die "Failed to append to file $mocked_log_path";

($results, $positions) = Utils::LogParser::process_logs($logs, $update_checks, $positions);

is_deeply($results, {'mocked_log.lines_count' => 2}, 'results update');
is_deeply($positions, {$mocked_log_path => length($log_content_full),}, 'positions update');

($results, $positions) = Utils::LogParser::process_logs($logs, $full_checks, {$mocked_log_path => -1});

is_deeply(
    $results,
    {
        'mocked_log.callback_args' => 10,
        'mocked_log.starts'        => 5,
    },
    'results full'
);
is_deeply($positions, {$mocked_log_path => length($log_content_full),}, 'positions full');
