#!/usr/bin/perl -w

# Usage: > ./prove2 *.t :: --self_update 2>&1 | less

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Partner::Utils qw(get_test_data_and_update_if_needed get_test_data_path);
use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner2::Simple;

use Clone;
use Test::Differences;
use Test::Exception;
use Test::More tests => 12;

use Utils::MigrationApplier;

my $COMPLETED;

run_tests(
    sub {
        my ($app) = @_;

        subtest '"type" arg is required' => sub {
            plan tests => 2;
            throws_ok {
                run_migrations($app, {order => 'before',});
            }
            'Exception', 'Raise exception';
            is($@->message(), 'type and order - requred options', 'check message');
        };

        subtest '"order" arg is required' => sub {
            plan tests => 2;
            throws_ok {
                run_migrations($app, {type => 'mysql',});
            }
            'Exception', 'Raise exception';
            is($@->message(), 'type and order - requred options', 'check message');
        };

        subtest 'empty list of migrations' => sub {
            plan tests => 4;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {}, {};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    diag @_;
                    fail 'incorrect call ERROR';
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'skip manual' => sub {
            plan tests => 5;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], TRUE, $COMPLETED], 'check_dependencies args');
                    return {}, {'MANUAL RUN' => ['MANUAL_migration'],};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    diag @_;
                    fail 'incorrect call ERROR';
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    eq_or_diff(\@_, ['SKIP because of MANUAL RUN: MANUAL_migration'], 'correct INFO');
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order          => 'before',
                        type           => 'mysql',
                        without_manual => TRUE,
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'skip corrupted' => sub {
            plan tests => 5;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {}, {'IO ERROR' => ['correpted_migration'],};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    eq_or_diff(\@_, ['SKIP because of IO ERROR: correpted_migration'], 'correct ERROR');
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'skip deps' => sub {
            plan tests => 5;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {migration_with_incoplete_deps => ['unknown_dep'],}, {};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    eq_or_diff(
                        \@_,
                        ['SKIP because of INCOMPLETE DEPENDENCIES: migration_with_incoplete_deps'],
                        'correct ERROR'
                    );
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'two migrations with deps' => sub {
            plan tests => 6;
            my $odd  = TRUE;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {first_migration => [], second_migration => ['first_migration'],}, {};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    shift;    # app
                    if ($odd) {
                        eq_or_diff(\@_, ['first_migration', undef], 'first of all execute first migration');
                    } else {
                        eq_or_diff(\@_, ['second_migration', undef], 'execute second migration after deps');
                    }
                    $odd = !$odd;
                    return TRUE;
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;    # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    diag @_;
                    fail 'incorrect call ERROR';
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'two migrations with deps and fail first' => sub {
            plan tests => 6;
            my $odd  = TRUE;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {first_migration => [], second_migration => ['first_migration'],}, {};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    shift;    # app
                    if ($odd) {
                        eq_or_diff(\@_, ['first_migration', undef], 'first of all execute first migration');
                    } else {
                        diag $_[1];    # migration
                        fail 'incorrect call execute_migration';
                    }
                    $odd = !$odd;
                    return FALSE;
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;             # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    eq_or_diff(\@_, ['SKIP because of INCOMPLETE DEPENDENCIES: second_migration'], 'correct ERROR');
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                -1,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'case with empty skip reason list' => sub {
            plan tests => 5;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], undef, $COMPLETED], 'check_dependencies args');
                    return {}, {'SOME REASON' => [],};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    diag @_;
                    fail 'incorrect call ERROR';
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    eq_or_diff(\@_, ['skip reason "%s" is defined but empty', 'SOME REASON'], 'correct ERRORF');
                },
                'Utils::MigrationApplier::INFO' => sub {
                    diag @_;
                    fail 'incorrect call INFO';
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order => 'before',
                        type  => 'mysql',
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };

        subtest 'skip manual and skip migration that depends on it' => sub {
            plan tests => 6;
            my $subs = {
                'Utils::MigrationApplier::check_dependencies' => sub {
                    eq_or_diff(\@_, [[], TRUE, $COMPLETED], 'check_dependencies args');
                    return {some_migration => ['MANUAL_migration']}, {'MANUAL RUN' => ['MANUAL_migration'],};
                },
                'Utils::MigrationApplier::execute_migration' => sub {
                    diag $_[1];    # migration
                    fail 'incorrect call execute_migration';
                },
                'Utils::MigrationApplier::get_completed_migrations' => sub {
                    shift;         # app
                    eq_or_diff(\@_, ['mysql'], 'get_completed_migrations args');
                    return clone $COMPLETED;
                },
                'Utils::MigrationApplier::get_new_migration_scripts' => sub {
                    eq_or_diff(\@_, ['mysql', 'before', $COMPLETED], 'get_new_migration_scripts args');
                    return [];
                },
                'Utils::MigrationApplier::ERROR' => sub {
                    eq_or_diff(\@_, ['SKIP because of INCOMPLETE DEPENDENCIES: some_migration'], 'correct ERROR');
                },
                'Utils::MigrationApplier::ERRORF' => sub {
                    diag @_;
                    fail 'incorrect call ERRORF';
                },
                'Utils::MigrationApplier::INFO' => sub {
                    eq_or_diff(\@_, ['SKIP because of MANUAL RUN: MANUAL_migration'], 'correct INFO');
                },
            };
            mock_subs($subs);
            is(
                run_migrations(
                    $app,
                    {
                        order          => 'before',
                        type           => 'mysql',
                        without_manual => TRUE,
                    }
                ),
                0,
                'result of run_migration'
              );
            restore_subs([keys %$subs]);
        };
    },
);
