#!/usr/bin/perl

use Test::Most;
use Test::Differences qw(eq_or_diff);

use Test::Partner2::Simple;
use Test::Partner::Utils;
use Test::Partner2::Fixture;
use Test::Partner::DB::Mock;
use Capture::Tiny qw(capture_stdout);

use qbit;
use Utils::Oneshots::ChangeDesignDropOldFormat;
use PiConstants qw($IS_PRECISE);

my %fixtures = (
    '06_oldPosterHorizontal' => {
        name         => "oldPosterHorizontal",
        borderType   => "block",
        hoverColor   => "00FF00",
        textColor    => "00FF00",
        borderColor  => "000000",
        borderRadius => \0,
        limit        => 1,
    },
    '07_oldPosterHorizontal_gm' => {
        name              => "oldPosterHorizontal",
        borderType        => "block",
        hoverColor        => "00FF00",
        textColor         => "00FF00",
        borderColor       => "000000",
        borderRadius      => \0,
        limit             => 1,
        is_custom_bk_data => 1,
    },
    '08_pictureSearch3' => {
        name  => "pictureSearch3",
        limit => 1,
    },
    '09_pictureSearch3_gm' => {
        name              => "pictureSearch3",
        limit             => 1,
        is_custom_bk_data => 1,
    },
    '10_pictureSearch' => {
        name  => "pictureSearch",
        limit => 1,
    },
    '11_pictureSearch_gm' => {
        name              => "pictureSearch",
        limit             => 1,
        is_custom_bk_data => 1,
    },
    '12_newHorizontal' => {
        name  => "newHorizontal",
        limit => 1,
    },
    '13_newHorizontal_gm' => {
        name              => "newHorizontal",
        limit             => 1,
        is_custom_bk_data => 1,
    },
    '14_grid' => {
        name  => "grid",
        limit => 1,
    },
    '15_posterCarousel' => {
        name           => "posterCarousel",
        arrowsCarousel => \0,
        limit          => 2,
        '-extra'       => {
            slider => {
                selectType => "default",
                type       => "default",
            },
        },
    },
    '16_adaptiveCarousel' => {
        name           => "adaptiveCarousel",
        arrowsCarousel => \0,
        limit          => 2,
        '-extra'       => {
            slider => {
                selectType => "default",
                type       => "default",
            },
        },
    },
);

my @custom_bk_data = grep {delete $fixtures{$_}{is_custom_bk_data}} sort keys %fixtures;

run_tests(
    sub {
        my ($app) = @_;
        mock_curdate('2020-09-10 10:10:10');

        my @blocks;
        my @list;
        for my $name (sort keys %fixtures) {
            my $block = get_fixture($name);
            push @blocks, $block;
            push @list,
              {
                page_id  => $block->{campaign_id},
                block_id => $block->{id},
              };
        }

        my $data = {before => get_designs($app, \@blocks)};

        for my $name (@custom_bk_data) {
            my $block = get_fixture($name);
            my $bk_data = get_block_bk_data($app, $block);
            $data->{bk_data}{$name}{before} = $bk_data->{RtbDesign};
            $app->context_on_site_rtb->do_action($block, "edit", is_custom_bk_data => 1);
        }

        my $stdout_1 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['oldPosterHorizontal']});
        };

        my $stdout_2 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['oldPosterHorizontal', 'pictureSearch3']});
        };

        my $stdout_3 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['pictureSearch', 'newHorizontal'], mode => 'add'});
        };

        my $stdout_4 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['newHorizontal'], mode => 'delete'});
        };

        my $stdout_5 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['grid'], fix_sv => {grid => {height => 290}}});
        };

        my $stdout_6 = capture_stdout {
            Utils::Oneshots::ChangeDesignDropOldFormat::change_designs($app,
                {list => \@list, formats => ['posterCarousel', 'adaptiveCarousel']});
        };

        $data->{after} = get_designs($app, \@blocks);

        for my $name (@custom_bk_data) {
            my $block = get_fixture($name);
            my $bk_data = get_block_bk_data($app, $block);
            $data->{bk_data}{$name}{after} = $bk_data->{RtbDesign};
        }

        if ($IS_PRECISE) {
            for my $k (keys %$data) {
                next unless ref $data->{$k} eq 'ARRAY';
                for my $e (@{$data->{$k}}) {
                    for my $f (qw(id multistate block_id page_id)) {
                        $e->{$f} += 0 if exists $e->{$f};
                    }
                }
            }

            for my $stdout ($stdout_1, $stdout_2, $stdout_3, $stdout_4, $stdout_5, $stdout_6) {
                $stdout =~ s/("(?:BlindLevel|DSPID|AdFoxBlock)":)"(\d{1,4})"/$1$2/g;
            }
        }

        my $expected_1 = get_test_data_and_update_if_needed("stdout_1.dump", $stdout_1, raw => TRUE);
        eq_or_diff($stdout_1, $expected_1, "check stdout 1");

        my $expected_2 = get_test_data_and_update_if_needed("stdout_2.dump", $stdout_2, raw => TRUE);
        eq_or_diff($stdout_2, $expected_2, "check stdout 2");

        my $expected_3 = get_test_data_and_update_if_needed("stdout_3.dump", $stdout_3, raw => TRUE);
        eq_or_diff($stdout_3, $expected_3, "check stdout 3");

        my $expected_4 = get_test_data_and_update_if_needed("stdout_4.dump", $stdout_4, raw => TRUE);
        eq_or_diff($stdout_4, $expected_4, "check stdout 4");

        my $expected_5 = get_test_data_and_update_if_needed("stdout_5.dump", $stdout_5, raw => TRUE);
        eq_or_diff($stdout_5, $expected_5, "check stdout 5");

        my $expected_6 = get_test_data_and_update_if_needed("stdout_6.dump", $stdout_6, raw => TRUE);
        eq_or_diff($stdout_6, $expected_6, "check stdout 6");

        my $has = get_test_data_and_update_if_needed("data.json", $data);
        eq_or_diff($data, $has, "check data");
    },
    do_not_die_on_fail => 1,
    fill_databases     => 0,
    fixtures           => ['user_cron', 'tns_dict_article', 'picategories_dict'],
    extra_fixtures     => make_fixtures(),
);

sub make_fixtures {

    my @result;
    for my $name (sort keys %fixtures) {
        push @result, make_fixture($name, $fixtures{$name});
    }

    return +{@result};
}

sub make_fixture {
    my ($name, $design) = @_;

    my $extra = delete $design->{'-extra'} // {};

    return Test::Partner2::Fixture::DB::Block::context_rtb_block_fixtures(
        fixture_name      => $name,
        page_fixture_name => 'partner_context_page',
        site_version      => 'turbo',
        design_templates  => [
            {
                caption                 => 'design ' . $name,
                type                    => 'tga',
                design_settings         => $design,
                is_custom_format_direct => 0,
                filter_tags             => ["adaptive", "horizontal"],
                %$extra,
            },
        ],
    );
}

sub get_designs {
    my ($app, $blocks) = @_;

    my @filter;
    for my $block (@$blocks) {
        push @filter, [AND => [[page_id => '=' => \$block->{campaign_id}], [block_id => '=' => \$block->{id}],]];
    }
    my $list = $app->partner_db->design_templates->get_all(
        fields => [qw(id caption page_id block_id opts multistate)],
        filter => [OR => \@filter],
    );
    for my $row (@$list) {
        $row->{opts} = from_json($row->{opts});
        if (my $ds = $row->{opts}{design_settings}) {
            for my $f (qw(css template)) {
                if (exists $ds->{$f} && $ds->{$f}) {
                    $ds->{$f} = [split /\n/, $ds->{$f}];
                }
            }
        }
    }
    return $list;
}

sub get_block_bk_data {
    my ($app, $block) = @_;

    my $model = $app->context_on_site_rtb;
    my $bk_data;
    eval {$bk_data = get_bk_data($model, $block)};
    if ($@) {
        $bk_data = 'error in get_bk_data context_on_site_rtb';
        warn "context_on_site_rtb\n" . $@;
    }
    return $bk_data->{$block->{id}};
}
