#!/usr/bin/perl

use Test::Most;
use Test::Differences qw(eq_or_diff);

use Test::Partner2::Simple;
use Test::Partner::Utils;
use Test::Partner2::Fixture;
use Test::Partner::DB::Mock;
use Capture::Tiny qw(capture_stdout);

use qbit;
use Utils::Oneshots::ChangeDesignRowToNative;
my $fixture_name = 'block_fixure_with_filter_tags';

my %fixtures = (
    row_empty    => {name => 'row'},
    newrow_empty => {name => 'newRow'},
    row_full     => {
        name         => 'row',
        bgColor      => "FFFFF4",
        urlColor     => "FFFFF7",
        textColor    => "FFFFF3",
        borderType   => "block",
        hoverColor   => "FFFFF1",
        titleColor   => "FFFFF6",
        borderColor  => "FFFFF2",
        siteBgColor  => "FFFFF5",
        borderRadius => \0,
    },
    newrow_full => {
        name         => 'newRow',
        bgColor      => "FFFFF4",
        urlColor     => "FFFFF7",
        textColor    => "FFFFF3",
        borderType   => "block",
        hoverColor   => "FFFFF1",
        titleColor   => "FFFFF6",
        borderColor  => "FFFFF2",
        siteBgColor  => "FFFFF5",
        borderRadius => \0,
    },
    row_border_link => {
        name           => 'row',
        borderType     => "collapse",
        borderRadius   => \1,
        linksUnderline => \0,
    },
    test => {
        name         => "newRow",
        borderType   => "block",
        hoverColor   => "00FF00",
        borderColor  => "000000",
        borderRadius => \0,
    },
);

run_tests(
    sub {
        my ($app) = @_;
        mock_curdate('2020-09-10 10:10:10');

        my @blocks;
        my @list;
        for my $name (sort keys %fixtures) {
            my $block = get_fixture($name);
            push @blocks, $block;
            push @list,
              {
                page_id  => $block->{campaign_id},
                block_id => $block->{id},
              };
        }

        my $data = {before => get_designs($app, \@blocks)};

        my $stdout = capture_stdout {
            Utils::Oneshots::ChangeDesignRowToNative::change_designs($app, {list => \@list});
        };

        $data->{after} = get_designs($app, \@blocks);

        my $expected = get_test_data_and_update_if_needed("stdout.dump", $stdout, raw => TRUE);
        eq_or_diff($stdout, $expected, "check stdout");

        my $has = get_test_data_and_update_if_needed("data.json", $data);
        eq_or_diff($data, $has, "check data");
    },
    do_not_die_on_fail => 1,
    fill_databases     => 0,
    fixtures           => ['tns_dict_article', 'picategories_dict'],
    extra_fixtures     => make_fixtures(),
);

sub make_fixtures {

    my @result;
    for my $name (sort keys %fixtures) {
        my $design = $fixtures{$name};
        push @result, make_fixture($name, $design);
    }

    return +{@result};
}

sub make_fixture {
    my ($name, $design) = @_;

    return Test::Partner2::Fixture::DB::Block::context_rtb_block_fixtures(
        fixture_name      => $name,
        page_fixture_name => 'partner_context_page',
        site_version      => 'turbo',
        design_templates  => [
            {
                caption         => 'design ' . $name,
                type            => 'tga',
                design_settings => {
                    limit => 1,
                    %$design
                },
                is_custom_format_direct => 0,
                filter_tags             => ["adaptive", "horizontal"]
            },
        ],
    );
}

sub get_designs {
    my ($app, $blocks) = @_;

    my @filter;
    for my $block (@$blocks) {
        push @filter, [AND => [[page_id => '=' => \$block->{campaign_id}], [block_id => '=' => \$block->{id}],]];
    }
    my $list = $app->partner_db->design_templates->get_all(
        fields   => [qw(id caption page_id block_id opts multistate)],
        filter   => [OR => \@filter],
        order_by => ['id'],
    );
    for my $row (@$list) {
        $row->{opts} = from_json($row->{opts});
        if (my $ds = $row->{opts}{design_settings}) {
            for my $f (qw(css template)) {
                if (exists $ds->{$f} && $ds->{$f}) {
                    $ds->{$f} = [split /\n/, $ds->{$f}];
                }
            }
        }
    }
    return $list;
}
