#!/usr/bin/perl

use Test::Most;
use Test::Differences qw(eq_or_diff);

use Test::Partner2::Simple;
use Test::Partner::Utils;
use Test::Partner2::Fixture;
use Test::Partner::DB::Mock;
use Capture::Tiny qw(capture_stdout);

use qbit;
use Utils::Oneshots::UpdateContentTemplate;
use PiConstants qw($IS_PRECISE);

my %fixtures = (
    '01_natural_error' => {
        model    => 'context_on_site_natural',
        template => '<ya-units-grid rows=${grid_rows} cols=${grid_columns}>',
    },
    '02_natural_error_gm' => {
        model             => 'context_on_site_natural',
        template          => '<ya-units-grid rows=${grid_rows} cols=${grid_columns}>',
        is_custom_bk_data => 1,
    },
    '03_natural' => {
        model    => 'context_on_site_natural',
        template => '<ya-unit-image/>',
    },
    '04_natural_gm' => {
        model             => 'context_on_site_natural',
        template          => '<ya-unit-image/>',
        is_custom_bk_data => 1,
    },
    '05_rtb' => {
        model    => 'context_on_site_rtb',
        template => '<ya-unit-image/>',
    },
    '06_rtb_gm' => {
        model             => 'context_on_site_rtb',
        template          => '<ya-unit-image/>',
        is_custom_bk_data => 1,
    },
    '07_rtb_error' => {
        model    => 'context_on_site_rtb',
        template => '<ya-units-grid rows=${grid_rows} cols=${grid_columns}>',
    },
    '08_rtb_error_gm' => {
        model             => 'context_on_site_rtb',
        template          => '<ya-units-grid rows=${grid_rows} cols=${grid_columns}>',
        is_custom_bk_data => 1,
    },
    '09_natural_deleted' => {
        model    => 'context_on_site_natural',
        template => '<ya-unit-image/>',
        deleted  => 1,
    },
    '10_natural_updated' => {
        model    => 'context_on_site_natural',
        template => '<ya-unit-image disable-video/>',
    },
    '11_natural_updated_gm' => {
        model             => 'context_on_site_natural',
        template          => '<ya-unit-image disable-video/>',
        is_custom_bk_data => 1,
    },
    '12_rtb_updated' => {
        model    => 'context_on_site_rtb',
        template => '<ya-unit-image ratio="1" disable-video/>',
    },
    '13_rtb_updated_gm' => {
        model             => 'context_on_site_rtb',
        template          => '<ya-unit-image disable-video/>',
        is_custom_bk_data => 1,
    },

);

my @custom_bk_data = grep {delete $fixtures{$_}{is_custom_bk_data}} sort keys %fixtures;
my @deleted        = grep {delete $fixtures{$_}{deleted}} sort keys %fixtures;

run_tests(
    sub {
        my ($app) = @_;
        mock_curdate('2020-09-10 10:10:10');

        my @blocks;
        my @list = ('N-A-100-100');    # для проверки отсутстующего блока
        for my $name (sort keys %fixtures) {
            my $block = get_fixture($name);
            push @blocks, $block;
            my $accessor = $fixtures{$name}{model};
            push @list, $app->$accessor->public_id_prefix . "$block->{campaign_id}-$block->{id}",;
        }

        my $data = {before => get_templates($app, \@blocks)};

        for my $name (@custom_bk_data) {
            my $accessor = $fixtures{$name}{model};
            my $block    = get_fixture($name);
            my $bk_data  = get_block_bk_data($app, $accessor, $block);
            if ($accessor eq 'context_on_site_rtb') {
                $data->{bk_data}{$name}{before} = $bk_data->{RtbDesign};
            } else {
                $data->{bk_data}{$name}{before} = $bk_data->{CustomBlockData}{WidgetInfo}{Template};
            }
            $app->$accessor->do_action($block, "edit", is_custom_bk_data => 1);
        }

        for my $name (@deleted) {
            my $accessor = $fixtures{$name}{model};
            my $block    = get_fixture($name);
            $app->$accessor->do_action($block, "delete");
        }

        my $stdout_1 = capture_stdout {
            Utils::Oneshots::UpdateContentTemplate::update_content_template($app, {blocks => \@list});
        };

        $data->{after} = get_templates($app, \@blocks);

        for my $name (@custom_bk_data) {
            my $accessor = $fixtures{$name}{model};
            my $block    = get_fixture($name);
            my $bk_data  = get_block_bk_data($app, $accessor, $block);
            if ($accessor eq 'context_on_site_rtb') {
                $data->{bk_data}{$name}{after} = $bk_data->{RtbDesign};
            } else {
                $data->{bk_data}{$name}{after} = $bk_data->{CustomBlockData}{WidgetInfo}{Template};
            }
        }

        if ($IS_PRECISE) {
            $stdout_1 =~ s/(NOTGM.*?"design_id":)"(\d+)"/$1$2/;
            for my $k (keys %$data) {
                next unless ref $data->{$k} eq 'ARRAY';
                for my $e (@{$data->{$k}}) {
                    for my $f (qw(id campaign_id block_id page_id)) {
                        $e->{$f} += 0 if exists $e->{$f};
                    }
                }
            }
        }

        my $expected_1 = get_test_data_and_update_if_needed("stdout_1.dump", $stdout_1, raw => TRUE);
        eq_or_diff($stdout_1, $expected_1, "check stdout short");

        my $has = get_test_data_and_update_if_needed("data.json", $data);
        eq_or_diff($data, $has, "check data");
    },
    do_not_die_on_fail => 1,
    fill_databases     => 0,
    fixtures           => ['user_cron', 'tns_dict_article', 'picategories_dict'],
    extra_fixtures     => make_fixtures(),
    mock_features      => {'mocked-yan-partner' => ['design_auction_native', 'design_auction_native_turbo']}
);

sub make_fixtures {

    my @result;
    for my $name (sort keys %fixtures) {
        if ($fixtures{$name}{model} eq 'context_on_site_rtb') {
            push @result, make_fixture_rtb($name, $fixtures{$name});
        } else {
            push @result, make_fixture_content($name, $fixtures{$name});
        }
    }

    return +{@result};
}

sub make_fixture_content {
    my ($name, $data) = @_;

    return Test::Partner2::Fixture::DB::Block::context_natural_block_fixtures(
        fixture_name      => $name,
        page_fixture_name => 'partner_context_page',
        opts              => {
            grid_rows    => 3,
            grid_columns => 3,
            strategy     => '1',
            css          => 'some custom css',
            template     => $data->{template},
        },
    );
}

sub make_fixture_rtb {
    my ($name, $data) = @_;

    return Test::Partner2::Fixture::DB::Block::context_rtb_block_fixtures(
        fixture_name      => $name,
        page_fixture_name => 'partner_context_page',
        site_version      => 'turbo',
        design_templates  => [
            {
                caption         => 'native ' . $name,
                type            => 'native',
                design_settings => {
                    grid_rows    => 1,
                    grid_columns => 1,
                    template     => $data->{template},
                    css          => 'some custom css'
                },
                is_custom_format_direct => 0,
                filter_tags             => ["adaptive", "horizontal"]
            },
            {
                caption         => 'tga ' . $name,
                type            => 'tga',
                design_settings => {
                    name  => "adaptive0418",
                    limit => 1,
                },
                is_custom_format_direct => 0,
                filter_tags             => ["adaptive", "horizontal"]
            },
        ],
    );
}

sub get_templates {
    my ($app, $blocks) = @_;

    my @filter;
    for my $block (@$blocks) {
        push @filter, [AND => [[campaign_id => '=' => \$block->{campaign_id}], [id => '=' => \$block->{id}],]];
    }
    my $list = $app->partner_db->context_on_site_natural->get_all(
        fields => [qw(campaign_id id opts)],
        filter => [OR => \@filter],
    );
    for my $row (@$list) {
        $row->{template} = from_json(delete $row->{opts})->{template};
    }

    my @filter2;
    for my $block (@$blocks) {
        push @filter2, [AND => [[page_id => '=' => \$block->{campaign_id}], [block_id => '=' => \$block->{id}],]];
    }
    my $list2 = $app->partner_db->design_templates->get_all(
        fields => [qw(id caption page_id block_id opts)],
        filter => [AND => [[OR => \@filter2], [type => '=' => \'native']]],
    );
    for my $row (@$list2) {
        $row->{template} = from_json(delete $row->{opts})->{design_settings}{template};
    }

    return [@$list, @$list2];
}

sub get_block_bk_data {
    my ($app, $accessor, $block) = @_;

    my $model = $app->$accessor;
    my $bk_data;
    eval {$bk_data = get_bk_data($model, $block)};
    if ($@) {
        $bk_data = 'error in get_bk_data ' . $accessor;
        warn $accessor . "\n" . $@;
    }
    return $bk_data->{$block->{id}};
}
