use strict;
use warnings FATAL => 'all';
use utf8;

use Test::More tests => 80;
use Test::Deep;

use Utils::Order qw(get_sort_function);

my $data = [[2, 2], [2, 1], [10, 20], [1, 2], [1, 1], [1, 3], [20, 10], [undef, undef]];

my $column_names = ['field', 'field2'];

my $TESTS = [
    #one field
    {
        'expected' => [[undef, undef], [1, 2], [1, 1], [1, 3], [2, 2], [2, 1], [10, 20], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[20, 10], [10, 20], [2, 2], [2, 1], [1, 2], [1, 1], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 2], [1, 1], [1, 3], [10, 20], [2, 2], [2, 1], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[20, 10], [2, 2], [2, 1], [10, 20], [1, 2], [1, 1], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [2, 2], [1, 2], [1, 3], [20, 10], [10, 20]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[10, 20], [20, 10], [1, 3], [2, 2], [1, 2], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [20, 10], [2, 2], [1, 2], [10, 20], [1, 3]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[1, 3], [10, 20], [2, 2], [1, 2], [20, 10], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    #two fields (type: number)
    {
        'expected' => [[undef, undef], [1, 1], [1, 2], [1, 3], [2, 1], [2, 2], [10, 20], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 3], [1, 2], [1, 1], [2, 2], [2, 1], [10, 20], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[20, 10], [10, 20], [2, 1], [2, 2], [1, 1], [1, 2], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[20, 10], [10, 20], [2, 2], [2, 1], [1, 3], [1, 2], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 1], [2, 1], [1, 2], [2, 2], [1, 3], [20, 10], [10, 20]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [2, 2], [1, 2], [1, 3], [20, 10], [10, 20]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[10, 20], [20, 10], [1, 3], [1, 2], [2, 2], [1, 1], [2, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[10, 20], [20, 10], [1, 3], [2, 2], [1, 2], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    #two fields (type: string)
    {
        'expected' => [[undef, undef], [1, 1], [1, 2], [1, 3], [10, 20], [2, 1], [2, 2], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 3], [1, 2], [1, 1], [10, 20], [2, 2], [2, 1], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[20, 10], [2, 1], [2, 2], [10, 20], [1, 1], [1, 2], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[20, 10], [2, 2], [2, 1], [10, 20], [1, 3], [1, 2], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 1], [2, 1], [20, 10], [1, 2], [2, 2], [10, 20], [1, 3]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [20, 10], [2, 2], [1, 2], [10, 20], [1, 3]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[1, 3], [10, 20], [1, 2], [2, 2], [20, 10], [1, 1], [2, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[1, 3], [10, 20], [2, 2], [1, 2], [20, 10], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    #different types
    {
        'expected' => [[undef, undef], [1, 1], [1, 2], [1, 3], [2, 1], [2, 2], [10, 20], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 3], [1, 2], [1, 1], [2, 2], [2, 1], [10, 20], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[20, 10], [10, 20], [2, 1], [2, 2], [1, 1], [1, 2], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[20, 10], [10, 20], [2, 2], [2, 1], [1, 3], [1, 2], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 1], [1, 2], [1, 3], [10, 20], [2, 1], [2, 2], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 3], [1, 2], [1, 1], [10, 20], [2, 2], [2, 1], [20, 10]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[20, 10], [2, 1], [2, 2], [10, 20], [1, 1], [1, 2], [1, 3], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[20, 10], [2, 2], [2, 1], [10, 20], [1, 3], [1, 2], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 1], [2, 1], [1, 2], [2, 2], [1, 3], [20, 10], [10, 20]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [2, 2], [1, 2], [1, 3], [20, 10], [10, 20]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[10, 20], [20, 10], [1, 3], [1, 2], [2, 2], [1, 1], [2, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[10, 20], [20, 10], [1, 3], [2, 2], [1, 2], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'number'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'string'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [1, 1], [2, 1], [20, 10], [1, 2], [2, 2], [10, 20], [1, 3]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[undef, undef], [2, 1], [1, 1], [20, 10], [2, 2], [1, 2], [10, 20], [1, 3]],
        'order_by' => [
            {
                'sort_order' => 'asc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[1, 3], [10, 20], [1, 2], [2, 2], [20, 10], [1, 1], [2, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'asc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    },
    {
        'expected' => [[1, 3], [10, 20], [2, 2], [1, 2], [20, 10], [2, 1], [1, 1], [undef, undef]],
        'order_by' => [
            {
                'sort_order' => 'desc',
                'name'       => 'field2',
                'type'       => 'string'
            },
            {
                'sort_order' => 'desc',
                'name'       => 'field',
                'type'       => 'number'
            }
        ]
    }
];

foreach my $test (@$TESTS) {
    my $order = get_sort_function($test->{'order_by'}, row_is_hash => 0, column_names => $column_names);

    my $test_name_postfix =
      join(' ', map {"$_->{name} (type: $_->{type}, sort_order: $_->{sort_order})"} @{$test->{'order_by'}});

    cmp_deeply([sort {$order->($a, $b)} @$data], $test->{'expected'}, 'ARRAY: ' . $test_name_postfix);

    my $hash_data     = _get_hash_data($data);
    my $hash_expected = _get_hash_data($test->{'expected'});

    my $hash_order = get_sort_function($test->{'order_by'}, row_is_hash => 1);

    cmp_deeply([sort {$hash_order->($a, $b)} @$hash_data], $hash_expected, 'HASH: ' . $test_name_postfix);
}

sub _get_hash_data {
    my ($data) = @_;

    my $index = 0;
    my %map_fields = map {$_ => $index++} @$column_names;

    my @hash_data = ();
    foreach my $row (@$data) {
        push(@hash_data, {map {$_ => $row->[$map_fields{$_}]} @$column_names});
    }

    return \@hash_data;
}
