use strict;

use Test::More;
use Test::Deep;

use Utils::Stream::Writer::Buffer;
use Utils::Stream::DataSource::Array;
use Utils::Stream::Serializer::JSON;

use qbit;

my $tests = [
    {
        to_serialize => sub {return Utils::Stream::DataSource::Array->new(source => []);},
        bulk_size    => 1,
        compact      => '[]',
        pretty       => "[\n]\n",
    },
    {
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 1,
        compact   => '[{"key":1,"value":1},{"key":2,"value":2},{"key":0,"value":100}]',
        pretty    => qq([
   {
      "key" : 1,
      "value" : 1
   },
   {
      "key" : 2,
      "value" : 2
   },
   {
      "key" : 0,
      "value" : 100
   }
]
),
    },
    {
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 4,
        compact   => '[{"key":1,"value":1},{"key":2,"value":2},{"key":0,"value":100}]',
        pretty    => qq([
   {
      "key" : 1,
      "value" : 1
   },
   {
      "key" : 2,
      "value" : 2
   },
   {
      "key" : 0,
      "value" : 100
   }
]
),
    },
    {
        to_serialize => sub {
            return {
                key => Utils::Stream::DataSource::Array->new(
                    source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],
                ),
            };
        },
        bulk_size => 4,
        compact   => '{"key":[{"key":1,"value":1},{"key":2,"value":2},{"key":0,"value":100}]}',
        pretty    => qq({
   "key" : [
      {
         "key" : 1,
         "value" : 1
      },
      {
         "key" : 2,
         "value" : 2
      },
      {
         "key" : 0,
         "value" : 100
      }
   ]
}
),
    },
    {
        to_serialize => sub {
            return [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100}];
        },
        bulk_size => 4,
        compact   => '[{"key":"1","value":"1"},{"key":"2","value":"2"},{"key":"0","value":"100"}]',
        pretty    => qq([
   {
      "key" : "1",
      "value" : "1"
   },
   {
      "key" : "2",
      "value" : "2"
   },
   {
      "key" : "0",
      "value" : "100"
   }
]
),
    },
];

for my $test (@$tests) {
    for my $mode (qw(compact pretty)) {
        my $to_serialize = $test->{to_serialize}->();
        my $bulk_size    = $test->{bulk_size};

        my $writer     = Utils::Stream::Writer::Buffer->new();
        my $serializer = Utils::Stream::Serializer::JSON->new(
            data   => $to_serialize,
            writer => $writer,
            ($mode eq 'pretty' ? (pretty => 1) : ()),
        );

        $serializer->serialize_full($test->{bulk_size});

        my $got      = $writer->content();
        my $expected = $test->{$mode};
        is($got, $expected, "Correct serializing in $mode mode");
    }
}

done_testing();
