use strict;

use Test::More;
use Test::Deep;

use Utils::Stream::Writer::Buffer;
use Utils::Stream::DataSource::Array;
use Utils::Stream::DataSource::Map;
use Utils::Stream::Serializer::TSV;

use qbit;

my $tests = [
    {
        to_serialize => sub {return Utils::Stream::DataSource::Array->new(source => []);},
        bulk_size    => 1,
        headers      => '',
        body         => '',
    },
    {
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 1,
        headers   => "key\tvalue\n",
        body      => "1\t1\n2\t2\n0\t100\n",
    },
    {
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 4,
        headers   => "key\tvalue\n",
        body      => "1\t1\n2\t2\n0\t100\n",
    },
    {
        to_serialize => sub {
            return {
                key => Utils::Stream::DataSource::Array->new(
                    source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],
                ),
            };
        },
        bulk_size => 4,
        headers   => "key\tvalue\n",
        body      => "1\t1\n2\t2\n0\t100\n",
    },
    {
        to_serialize => sub {
            return [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},];
        },
        bulk_size => 4,
        headers   => "key\tvalue\n",
        body      => "1\t1\n2\t2\n0\t100\n",
    },
    {
        to_serialize => sub {
            return Utils::Stream::DataSource::Map->new(
                source => Utils::Stream::DataSource::Array->new(
                    source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],
                ),
                map => sub {
                    my ($item) = @_;
                    $item->{value} += 111;
                    return $item;
                },
            );
        },
        bulk_size => 4,
        headers   => "key\tvalue\n",
        body      => "1\t112\n2\t113\n0\t211\n",
    },
];

for my $test (@$tests) {
    for my $mode (qw(no_end_marker no_headers)) {
        my $to_serialize = $test->{to_serialize}->();
        my $bulk_size    = $test->{bulk_size};

        my $writer     = Utils::Stream::Writer::Buffer->new();
        my $serializer = Utils::Stream::Serializer::TSV->new(
            data   => $to_serialize,
            writer => $writer,
            ($mode eq 'no_end_marker' ? (end_marker => FALSE) : ()),
            ($mode eq 'no_headers'    ? (no_headers => TRUE)  : ()),
        );

        $serializer->serialize_full($test->{bulk_size});

        my $got      = $writer->content();
        my $expected = $test->{body};

        $expected = $test->{headers} . $expected if $mode ne 'no_headers';
        $expected .= "#END\n" if $mode ne 'no_end_marker';

        is($got, $expected, "Correct serializing in $mode mode");
    }
}

done_testing();
