use strict;

use Test::More;
use Test::Deep;

use Utils::Stream::Writer::Buffer;
use Utils::Stream::DataSource::Array;
use Utils::Stream::Serializer::XML;

use qbit;

my $tests = [
    {
        name         => 'empty array',
        to_serialize => sub {return Utils::Stream::DataSource::Array->new(source => []);},
        bulk_size    => 1,
        compact      => '<?xml version="1.0" encoding="utf-8" ?><response><arrayref></arrayref></response>',
        pretty       => qq(<?xml version="1.0" encoding="utf-8" ?>
<response>
    <arrayref>
    </arrayref>
</response>
),
    },
    {
        name         => 'full array with size=1',
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 1,
        compact =>
'<?xml version="1.0" encoding="utf-8" ?><response><arrayref><item key="0"><hashref><item key="key">1</item><item key="value">1</item></hashref></item><item key="1"><hashref><item key="key">2</item><item key="value">2</item></hashref></item><item key="2"><hashref><item key="key">0</item><item key="value">100</item></hashref></item></arrayref></response>',
        pretty => qq(<?xml version="1.0" encoding="utf-8" ?>
<response>
    <arrayref>
        <item key="0">
            <hashref>
                <item key="key">1</item>
                <item key="value">1</item>
            </hashref>
        </item>
        <item key="1">
            <hashref>
                <item key="key">2</item>
                <item key="value">2</item>
            </hashref>
        </item>
        <item key="2">
            <hashref>
                <item key="key">0</item>
                <item key="value">100</item>
            </hashref>
        </item>
    </arrayref>
</response>
),
    },
    {
        name         => 'full array with size=4',
        to_serialize => sub {
            return Utils::Stream::DataSource::Array->new(
                source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],);
        },
        bulk_size => 4,
        compact =>
'<?xml version="1.0" encoding="utf-8" ?><response><arrayref><item key="0"><hashref><item key="key">1</item><item key="value">1</item></hashref></item><item key="1"><hashref><item key="key">2</item><item key="value">2</item></hashref></item><item key="2"><hashref><item key="key">0</item><item key="value">100</item></hashref></item></arrayref></response>',
        pretty => qq(<?xml version="1.0" encoding="utf-8" ?>
<response>
    <arrayref>
        <item key="0">
            <hashref>
                <item key="key">1</item>
                <item key="value">1</item>
            </hashref>
        </item>
        <item key="1">
            <hashref>
                <item key="key">2</item>
                <item key="value">2</item>
            </hashref>
        </item>
        <item key="2">
            <hashref>
                <item key="key">0</item>
                <item key="value">100</item>
            </hashref>
        </item>
    </arrayref>
</response>
),
    },
    {
        name         => 'array in hash',
        to_serialize => sub {
            return {
                key => Utils::Stream::DataSource::Array->new(
                    source => [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},],
                ),
            };
        },
        bulk_size => 4,
        compact =>
'<?xml version="1.0" encoding="utf-8" ?><response><hashref><item key="key"><arrayref><item key="0"><hashref><item key="key">1</item><item key="value">1</item></hashref></item><item key="1"><hashref><item key="key">2</item><item key="value">2</item></hashref></item><item key="2"><hashref><item key="key">0</item><item key="value">100</item></hashref></item></arrayref></item></hashref></response>',
        pretty => qq(<?xml version="1.0" encoding="utf-8" ?>
<response>
    <hashref>
        <item key="key">
            <arrayref>
                <item key="0">
                    <hashref>
                        <item key="key">1</item>
                        <item key="value">1</item>
                    </hashref>
                </item>
                <item key="1">
                    <hashref>
                        <item key="key">2</item>
                        <item key="value">2</item>
                    </hashref>
                </item>
                <item key="2">
                    <hashref>
                        <item key="key">0</item>
                        <item key="value">100</item>
                    </hashref>
                </item>
            </arrayref>
        </item>
    </hashref>
</response>
),
    },
    {
        name         => 'real array',
        to_serialize => sub {
            return [{key => 1, value => 1}, {key => 2, value => 2}, {key => 0, value => 100},];
        },
        bulk_size => 4,
        compact =>
'<?xml version="1.0" encoding="utf-8" ?><response><arrayref><item key="0"><hashref><item key="key">1</item><item key="value">1</item></hashref></item><item key="1"><hashref><item key="key">2</item><item key="value">2</item></hashref></item><item key="2"><hashref><item key="key">0</item><item key="value">100</item></hashref></item></arrayref></response>',
        pretty => qq(<?xml version="1.0" encoding="utf-8" ?>
<response>
    <arrayref>
        <item key="0">
            <hashref>
                <item key="key">1</item>
                <item key="value">1</item>
            </hashref>
        </item>
        <item key="1">
            <hashref>
                <item key="key">2</item>
                <item key="value">2</item>
            </hashref>
        </item>
        <item key="2">
            <hashref>
                <item key="key">0</item>
                <item key="value">100</item>
            </hashref>
        </item>
    </arrayref>
</response>
),
    }
];

for my $test (@$tests) {
    for my $mode (qw(compact pretty)) {
        my $to_serialize = $test->{to_serialize}->();
        my $bulk_size    = $test->{bulk_size};

        my $writer     = Utils::Stream::Writer::Buffer->new();
        my $serializer = Utils::Stream::Serializer::XML->new(
            data   => $to_serialize,
            writer => $writer,
            ($mode eq 'pretty' ? (pretty => 1) : ()),
        );

        $serializer->serialize_full($test->{bulk_size});

        my $got      = $writer->content();
        my $expected = $test->{$mode};
        is($got, $expected, "Correct serializing $test->{name}  with '$mode'");
    }
}

done_testing();
