#!/usr/bin/perl

use Test::More;
use Test::Deep;

use qbit;
use PiConstants qw($IS_PRECISE);

use Utils::XLS;
use Spreadsheet::ParseExcel;

plan tests => 1;

my $fields = [
    {
        'field'       => 'date',
        'field_order' => 0,
        'name'        => "Период",
        'sort_as'     => 'text',
        'type'        => 'select'
    },
    {
        'field'       => 'dsp_caption',
        'field_order' => 5,
        'name'        => "DSP",
        'sort_as'     => 'string',
        'type'        => 'string'
    },
    {
        'field'       => 'mobile_mediation_block_percent',
        'field_order' => 7,
        'name'        => "Процент",
        'sort_as'     => 'number',
        'type'        => 'percent'
    },
    {
        'field'       => 'mobile_mediation_block_hits',
        'field_order' => 12,
        'name'        => "Запросы рекламы",
        'sort_as'     => 'number',
        'type'        => 'number'
    },
];

my $data = [
    {
        date                           => '2018-10-26',
        dsp_caption                    => 'Директ',
        mobile_mediation_block_percent => 30.5,
        mobile_mediation_block_hits    => 150,
    },
    {
        date                           => '2018-W10',
        dsp_caption                    => 'Директ',
        mobile_mediation_block_percent => 46.5,
        mobile_mediation_block_hits    => 70,
    },
    {
        date                           => '2018-10',
        dsp_caption                    => 'Директ',
        mobile_mediation_block_percent => 50,
        mobile_mediation_block_hits    => 193,
    },
    {
        date                           => '2018',
        dsp_caption                    => 'Директ',
        mobile_mediation_block_percent => 97.36,
        mobile_mediation_block_hits    => 234,
    },
    {
        date                           => '2018',
        dsp_caption                    => undef,
        mobile_mediation_block_percent => undef,
        mobile_mediation_block_hits    => undef,
    },
    {
        date                           => '2018',
        dsp_caption                    => '-',
        mobile_mediation_block_percent => '-',
        mobile_mediation_block_hits    => '-',
    },
];

my $xls = xls_with_fields_and_names($data, $fields, default => '-');

my $parser   = Spreadsheet::ParseExcel->new();
my $workbook = $parser->parse(\$xls);

my $result = [];
for my $worksheet ($workbook->worksheets()) {
    my ($row_min, $row_max) = $worksheet->row_range();
    my ($col_min, $col_max) = $worksheet->col_range();

    for my $row ($row_min .. $row_max) {
        for my $col ($col_min .. $col_max) {

            my $cell = $worksheet->get_cell($row, $col);
            next unless $cell;

            $result->[$row][$col] = $cell->value();
        }
    }
}

my $v = $IS_PRECISE ? 'October2018 2018' : 'October, 2018';

cmp_deeply(
    $result,
    [
        ["Период", 'DSP',          "Процент", "Запросы рекламы"],
        ['26.10.2018',   "Директ", '31%',            '150'],
        ['05.03.2018',   "Директ", '47%',            '70'],
        [$v,             "Директ", '50%',            '193'],
        ['2018',         "Директ", '97%',            '234'],
        ['2018',         '-',            '-',              '-'],
        ['2018',         '-',            '-',              '-'],
        undef,
        undef,
        ['payment_disclaimer', '', '', ''],
    ],
    'check xls'
);
