#!/usr/bin/env perl

use Test::More tests => 19;
use Test::Deep qw(cmp_deeply);
use Mojo::Cookie::Request;
use RestApi;

my @test_cases = (
    # normal request cookie (RFC 2965)
    '$Version=1; foo=bar; $Path="/test"',
    # undef tests
    '',
    'foo',
    'foo=',
    ';',
    # request cookies from multiple header values (RFC 2965)
    '$Version=1; foo=bar; $Path="/test", $Version=0; baz=yada; $Path="/tset"',
    # request cookie (Netscape)
    'CUSTOMER=WILE_E_COYOTE',
    # multiple request cookies (Netscape)
    'CUSTOMER=WILE_E_COYOTE; PART_NUMBER=ROCKET_LAUNCHER_0001',
    # multiple request cookies from multiple header values (Netscape)
    'CUSTOMER=WILE_E_COYOTE, PART_NUMBER=ROCKET_LAUNCHER_0001',
    # request cookie without value (RFC 2965)
    '$Version=1; foo=; $Path="/test"',
    '$Version=1; foo=""; $Path="/test"',
    # quoted request cookie (RFC 2965)
    '$Version=1; foo="b ,a\" r\"\\\\"; $Path="/test"',
    # quoted request cookie roundtrip (RFC 2965)
    '$Version=1; foo="b ,a\";= r\"\\\\"; $Path="/test"',
    # quoted request cookie roundtrip (RFC 2965, alternative)
    '$Version=1; foo="b ,a\" r\"\\\\"; $Path="/test"',
    # quoted request cookie roundtrip (RFC 2965, another alternative)
    '$Version=1; foo="b ;a\" r\"\\\\"; $Path="/test"',
    # quoted request cookie roundtrip (RFC 2965, yet another alternative)
    '$Version=1; foo="\"b a\" r\""; $Path="/test"',
    # multiple cookie request (RFC 2965)
    '$Version=1; foo=bar; $Path=/test; baz="la la"; $Path=/tset'
);

foreach my $cookie (@test_cases) {
    cmp_deeply(RestApi->_parse_and_unescape_cookies($cookie), Mojo::Cookie::Request->parse($cookie), "testing $cookie");
}

my $cookie = 'fakelogin=info%40slash-cash-ru';
is(RestApi->_parse_and_unescape_cookies($cookie)->[0]->value, 'info@slash-cash-ru',   "unescaped cookie");
is(Mojo::Cookie::Request->parse($cookie)->[0]->value,         'info%40slash-cash-ru', "escaped cookie");
