#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use FindBin qw($Bin);
use lib "$Bin/../lib";

use Test::Most;
use Test::More;
use Test::Deep;
use Test::MockObject::Extends;

use qbit;

use Rosetta;
use QBit::Validator;

my $rosetta = Rosetta->new();
$rosetta->pre_run;

$rosetta->context_on_site_rtb;
$rosetta->{'context_on_site_rtb'} = Test::MockObject::Extends->new($rosetta->{'context_on_site_rtb'});

$rosetta->context_on_site_rtb->mock('get_all',
    sub {[{campaign_id => '123', id => '2'}, {campaign_id => '123', id => '1'}]});

$rosetta->context_on_site_rtb->mock(
    'add',
    sub {
        my ($self, %opts) = @_;

        QBit::Validator->new(
            data     => \%opts,
            template => {
                type   => 'hash',
                fields => {
                    campaign_id => {in      => ['123', '321']},
                    caption     => {len_max => 7}
                }
            },
            throw => TRUE,
        );
    }
);

$rosetta->context_on_site_rtb->mock(
    'do_action',
    sub {
        my ($self, $id, $action, %opts) = @_;

        if ($action eq 'start') {
            throw Exception::Denied gettext('Access denied');
        } elsif ($action eq 'edit') {
            throw Exception::Validation::BadArguments gettext('You cannot edit caption');
        } else {
            throw gettext('Unknown action "%s"', $action);
        }
    }
);

cmp_deeply(
    $rosetta->call(model => 'context_on_site_rtb', method => 'get_all', args => []),
    {
        'data' => [
            [
                {
                    'id'          => '2',
                    'campaign_id' => '123'
                },
                {
                    'id'          => '1',
                    'campaign_id' => '123'
                }
            ]
        ],
        'result' => 'ok'
    },
    'get data'
);

cmp_deeply(
    $rosetta->call(
        model  => 'context_on_site_rtb',
        method => 'add',
        args   => [campaign_id => 213, caption => 'new block']
    ),
    {
        'fields' => [
            {
                'messages' => [gettext('Got value "%s" not in array: %s', 213, join(', ', qw(123 321)))],
                'name'     => ['campaign_id']
            },
            {
                'messages' => [gettext('Length "%s" more than "%s"', 'new block', 7)],
                'name'     => ['caption']
            },
        ],
        'type'   => 'validation',
        'result' => 'error'
    },
    'type validation'
);

cmp_deeply(
    $rosetta->call(
        model  => 'context_on_site_rtb',
        method => 'do_action',
        args   => ['R-A-123-1', 'start']
    ),
    {
        'description' => gettext('Access denied'),
        'type'        => 'denied',
        'result'      => 'error'
    },
    'type denied'
);

cmp_deeply(
    $rosetta->call(
        model  => 'context_on_site_rtb',
        method => 'do_action',
        args   => ['R-A-123-1', 'edit', caption => 'Rename block']
    ),
    {
        'fields' => [{name => [], messages => [gettext('You cannot edit caption')]}],
        'type'   => 'validation',
        'result' => 'error'
    },
    'type validation (Exception::Validation::BadArguments)'
);

cmp_deeply(
    $rosetta->call(
        model  => 'context_on_site_rtb',
        method => 'do_action',
        args   => ['R-A-123-1', 'blablabla']
    ),
    {
        'description'         => gettext('Internal Error'),
        'private_description' => gettext('Unknown action "%s"', 'blablabla'),
        'type'                => 'internal',
        'result'              => 'error',
    },
    'type internal'
);

done_testing();
exit;
