#!/usr/bin/perl -w

use Test::More;
use Test::Deep;
use Test::MockObject::Extends;

use qbit;

use FindBin qw($Bin);
use lib "$Bin/lib";
use lib "$Bin/../lib";

use WebInterface;

{
    no strict 'refs';
    no warnings 'redefine';
    *{'WebInterface::init'} = \&QBit::Application::init;
}

$ENV{'REQUEST_METHOD'} = 'GET';
subtest(
    _mark_active_path_in_menu => sub {
        my $web_interface = WebInterface->new();
        my ($default_path, $default_cmd) = $web_interface->default_cmd();

        {
            $web_interface->request(QBit::WebInterface::FastCGI::ShortLinkRequest->new(params => '', uri => '/',));
            $web_interface->set_option('cur_cmdpath' => $default_path);
            $web_interface->set_option('cur_cmd'     => $default_cmd);

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/".'
                );
            }
        }
        {
            $web_interface->request(QBit::WebInterface::FastCGI::ShortLinkRequest->new(params => '', uri => '/the/',));
            $web_interface->set_option('cur_cmdpath' => 'the');
            $web_interface->set_option('cur_cmd'     => 'url');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/the/url',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/the/url',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/the/url".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/the/url/',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/the/url/',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/the/url/".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/the/',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/the/',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/the/".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/the',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/the',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/the".'
                );
            }
            {
                my $items = [
                    {
                        name  => "Продукты",
                        type  => "group",
                        url   => "/context_on_site/list",
                        items => [
                            {
                                type => "item",
                                url  => '/the/url',
                            }
                        ],
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            name  => "Продукты",
                            type  => "group",
                            url   => "/context_on_site/list",
                            items => [
                                {
                                    type   => "item",
                                    url    => '/the/url',
                                    active => TRUE,
                                }
                            ],
                            active => TRUE,
                        }
                    ],
                    'One item in one group.'
                );
            }
            {
                my $items = [
                    {
                        name  => "Продукты",
                        type  => "group",
                        url   => "/other/url",
                        items => [
                            {
                                type => "item",
                                url  => "/other/url",
                            }
                        ],
                    },
                    {
                        name  => "Продукты",
                        type  => "group",
                        url   => "/context_on_site/list",
                        items => [
                            {
                                type => "item",
                                url  => '/the/url',
                            }
                        ],
                    },
                    {
                        name  => "Продукты",
                        type  => "group",
                        url   => "/other/url",
                        items => [
                            {
                                type => "item",
                                url  => '/the/url',
                            }
                        ],
                    },
                    {
                        type => "item",
                        url  => '/the/url',
                    },
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            name  => "Продукты",
                            type  => "group",
                            url   => "/other/url",
                            items => [
                                {
                                    type => "item",
                                    url  => "/other/url",
                                }
                            ],
                        },
                        {
                            name  => "Продукты",
                            type  => "group",
                            url   => "/context_on_site/list",
                            items => [
                                {
                                    type   => "item",
                                    url    => '/the/url',
                                    active => TRUE,
                                }
                            ],
                            active => TRUE,
                        },
                        {
                            name  => "Продукты",
                            type  => "group",
                            url   => "/other/url",
                            items => [
                                {
                                    type => "item",
                                    url  => '/the/url',
                                }
                            ],
                        },
                        {
                            type => "item",
                            url  => '/the/url',
                        },
                    ],
                    'Several active items in several groups. The first one must be chosen.'
                );
            }
            {
                my $items = [
                    {
                        name  => "Продукты",
                        type  => "group",
                        url   => '/the/url',
                        items => [
                            {
                                type => "item",
                                url  => '/the/url',
                            }
                        ],
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            name  => "Продукты",
                            type  => "group",
                            url   => '/the/url',
                            items => [
                                {
                                    type   => "item",
                                    url    => '/the/url',
                                    active => TRUE,
                                }
                            ],
                            active => TRUE,
                        }
                    ],
                    'One item in one group. The same urls. Activation from top to bottom.'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(
                    params => 'deleted=1',
                    uri    => '/distribution_set/?deleted=1',
                )
            );
            $web_interface->set_option('cur_cmdpath' => 'distribution_set');
            $web_interface->set_option('cur_cmd'     => 'list');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set/list',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set/list?deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set/list',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set/list?deleted=1',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set/list?deleted=1".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set/list/',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set/list/?deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set/list/',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set/list/?deleted=1',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set/list/?deleted=1".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set/',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set/?deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set/',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set/?deleted=1',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set/?deleted=1".'
                );
            }
            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set?deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set?deleted=1',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set?deleted=1".'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(
                    params => 'deleted=1',
                    uri    => '/distribution_set/',
                )
            );
            $web_interface->set_option('cur_cmdpath' => 'distribution_set');
            $web_interface->set_option('cur_cmd'     => 'list');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set/',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set/?deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set/',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set/?deleted=1',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set/?deleted=1".'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(
                    params => '',
                    uri    => '/distribution_set/add',
                )
            );
            $web_interface->set_option('cur_cmdpath' => 'distribution_set');
            $web_interface->set_option('cur_cmd'     => 'add');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/distribution_set/',
                    },
                    {
                        type => "item",
                        url  => '/distribution_set/add',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/distribution_set/',
                        },
                        {
                            type   => "item",
                            url    => '/distribution_set/add',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and deleted "/distribution_set/add".'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(
                    params => 'request=very_long_request',
                    uri    => '/statistics/?request=very_long_request',
                )
            );
            $web_interface->set_option('cur_cmdpath' => 'statistics');
            $web_interface->set_option('cur_cmd'     => 'constructor');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/statistics/',
                    },
                    {
                        type => "item",
                        url  => '/statistics/?request=very_long_request',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/statistics/',
                        },
                        {
                            type   => "item",
                            url    => '/statistics/?request=very_long_request',
                            active => TRUE,
                        }
                    ],
                    'Two items in menu. Full path and long param string "/statistics/?request=very_long_request".'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(
                    params => 'request=very_long_request&deleted=1',
                    uri    => '/statistics/?request=very_long_request&deleted=1',
                )
            );
            $web_interface->set_option('cur_cmdpath' => 'statistics');
            $web_interface->set_option('cur_cmd'     => 'constructor');

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/statistics/',
                    },
                    {
                        type => "item",
                        url  => '/statistics/?request=very_long_request&deleted=1',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type => "item",
                            url  => '/statistics/',
                        },
                        {
                            type   => "item",
                            url    => '/statistics/?request=very_long_request&deleted=1',
                            active => TRUE,
                        }
                    ],
'Two items in menu. Full path and long param string "/statistics/?request=very_long_request&deleted=1".'
                );
            }
        }
        {
            $web_interface->request(
                QBit::WebInterface::FastCGI::ShortLinkRequest->new(params => '', uri => '/the/url',));
            $web_interface->set_option('cur_cmdpath' => 'the');
            $web_interface->set_option('cur_cmd'     => 'url');

            $web_interface = Test::MockObject::Extends->new($web_interface);
            $web_interface->set_always(get_cmds => {the => {__DEFAULT__ => {name => 'url'}}});

            {
                my $items = [
                    {
                        type => "item",
                        url  => '/the/',
                    }
                ];
                $web_interface->_mark_active_path_in_menu($items);
                cmp_deeply(
                    $items,
                    [
                        {
                            type   => "item",
                            url    => '/the/',
                            active => TRUE,
                        }
                    ],
                    'Just one item in menu "/the/" - default for current uri.'
                );
            }
        }
    }
);

done_testing();
exit;
