use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Test::More;
use Net::INET6Glue::INET_is_INET6;
use Yandex::Tanker;
use QBit::Gettext::Extract;
use Config::IniFiles;
use File::Find;

use lib::abs qw(
  ../lib
  );
use qbit;

my $TANKER_SEPARATOR = '+++MSGCTXT+++:';

sub extract_msgids_from_code {

    # Большая часть этого кода скопирована из скрипта tanker-sync

    my $config = Config::IniFiles->new(-file => '.tanker', -fallback => 'GENERAL', -nocase => TRUE)
      // die("File \".tanker\" is invalid:\n" . join("\n", @Config::IniFiles::errors));

    my @file_ignore_re = $config->val('GENERAL', 'file_ignore_re');

    my @invalid_dirs = grep {!-d $_} $config->val('GENERAL', 'dir');
    die(
        ngettext(
            'Directory "%s" defined, but does not exists.',
            "Directories defined, but do not exists:\n%s",
            scalar(@invalid_dirs),
            join("\n", @invalid_dirs)
        )
       ) if @invalid_dirs;

    my @invalid_files = grep {!-f $_} $config->val('GENERAL', 'file');
    die(
        ngettext(
            'File "%s" defined, but does not exists.',
            "Files defined, but do not exists:\n%s",
            scalar(@invalid_files),
            join("\n", @invalid_files)
        )
       ) if @invalid_files;

    my @files;
    find(
        {
            wanted => sub {
                if (-f $File::Find::name) {
                    foreach my $ignore_re (@file_ignore_re) {
                        if ($File::Find::name =~ /$ignore_re/) {
                            return;
                        }
                    }
                    push(@files, $File::Find::name);
                }
            },
            no_chdir => TRUE
        },
        $config->val('GENERAL', 'dir'),
        $config->val('GENERAL', 'file')
    );

    my $extractor = QBit::Gettext::Extract->new();

    foreach my $file (@files) {
        $extractor->extract_from_file($file);
    }

    my @msgids;

    foreach my $message (values(%{$extractor->po->{'__MESSAGES__'}})) {

        my $msgid = $message->{'message'};

        if (defined($message->{'context'})) {
            $msgid = $msgid . $TANKER_SEPARATOR . $message->{'context'};
        }

        push(@msgids, tanker_quote($msgid));
    }

    return sort @{array_uniq(@msgids)};
}

sub get_msgids_with_ru_translation_from_tanker {
    my (%opts) = @_;

    my $branch     = delete($opts{'branch'});
    my $project_id = delete($opts{'project_id'});

    my $tanker = Yandex::Tanker->new(
        {
            project => $project_id,
            url     => "https://tanker-api.yandex-team.ru",
        }
    );

    my $tjson = $tanker->get_project_tjson(
        status => 'approved',
        branch => $branch,
    );

    my $tanker_data = from_json(Encode::decode_utf8($tjson));

    my @msgids;

    foreach my $keyset_name (keys(%{$tanker_data->{keysets}})) {
        foreach my $msgid (keys(%{$tanker_data->{keysets}{$keyset_name}{keys}})) {
            my $status = $tanker_data->{keysets}{$keyset_name}{keys}{$msgid}{translations}{ru}{status};
            if ($status eq 'approved') {
                push(@msgids, $msgid);
            }
        }
    }

    return sort @msgids;
}

sub tanker_quote {
    my ($str) = @_;

    for ($str) {
        s/\r//g;
        s/\n/\\n/g;
        s/"/\\"/g;
    }

    return $str;
}

sub main_in_test {

    pass('Loaded ok');

    my @msgids_from_code = extract_msgids_from_code();

    my @msgids_with_ru_translation_from_tanker = get_msgids_with_ru_translation_from_tanker(
        project_id => 'pi2',
        branch     => 'master',
    );

    my @diff = @{arrays_difference(\@msgids_from_code, \@msgids_with_ru_translation_from_tanker)};

    if (@diff == 0) {
        pass('All msgids from code has ru translations in tanker');
    } else {
        fail('All msgids from code has ru translations in tanker');

        note('');
        note('msgids that has no ru translation in tanker:');
        note(join("\n", map {" * $_"} @diff));
        note('');
    }

    done_testing();
}
main_in_test();
