#!/usr/bin/perl

=encoding UTF-8
=cut

=head1 DESCRIPTION

=cut

# common modules
use strict;
use warnings FATAL => 'all';
use feature 'say';
use utf8;
use open qw(:std :utf8);

use Carp;
use JSON::PP;
use HTTP::Tiny;
use Data::Dumper;
use Net::INET6Glue::INET_is_INET6;
use Test::More;
use lib::abs qw(
  ../lib
  );

use PiConstants qw(
  $MOL_REPORT_TYPE_MAIN
  $MOL_REPORT_TYPE_DSP
  $MOL_REPORT_TYPE_MM
  );

require Application::Model::BKStatistics;

use QBit::Application;

# global vars
my $MOL_URL;
my $RESULTS;
my $EXPECTING_FAIL   = 1;
my $PAGE_ID_FILTER   = {eq => [1038]};
my $COUNTABLE_FIELDS = {
    $MOL_REPORT_TYPE_MAIN => ['DirectBannerShows'],
    $MOL_REPORT_TYPE_MM   => ['HitsMM'],
};
my %DEFAULT_OPTS = (
    date_from => '20160701',
    date_to   => '20160701',
    currency  => 'RUB',
    limits    => {offset => 0, limit => 1},
);

# subs
sub get_mol_url {
    my $config_in_lib = lib::abs::path('../lib/Application.json');
    my $dev_config    = lib::abs::path('../configs/dev/Application.json');

    my $config_file = -e $config_in_lib ? $config_in_lib : $dev_config;

    my $config = QBit::Application::read_config($config_file);

    my $url = $config->{'api_http_mol'}{url};

    die "Can't find out MOL url" unless $url;

    return $url;
}

sub check_mol_return_succes {
    my ($query, $name, $expecting_fail) = @_;

    $expecting_fail = $expecting_fail || 0;
    my $success = 0;

    # PI-17869 Запросы без фильтра по page_id очень тяжелые и вычитывают все данные за период
    # тесты ходят в предпрод, который ходит в продакшн кликхаус. Может неприятно сказаться на работе прода
    $query->{filters_pre}->{'PageID'} //= $PAGE_ID_FILTER;

    my $content = encode_json($query);
    my $url = sprintf('%s/export/master-report-pi.cgi', $MOL_URL);

    my $response = HTTP::Tiny->new->request(
        'POST', $url,
        {
            headers => {'Content-Type' => 'application/json',},
            content => $content,
        }
    );

    my $parsed_answer;

    if ($response->{status} eq '200') {
        eval {
            $parsed_answer = decode_json($response->{content});
            $success       = $parsed_answer->{status} == 0;
        };
    }

    if ($success xor $expecting_fail) {
        pass($name);
    } else {
        fail($name);
        diag '';
        diag 'Request:';
        diag ' ' x 4 . "QUERY='$content'";
        diag ' ' x 4 . "curl -H 'Content-Type: application/json' '$url' --data \"\$QUERY\"";
        diag '';

        diag 'Response:';
        diag Dumper $response;
        diag '';
    }

    return $success;

}

sub _grep_stat_type {
    my ($field, $stat_type) = @_;
    my $res = grep {$_ eq $stat_type} @{$field->{'report_types'} // [$MOL_REPORT_TYPE_MAIN]};
    return $res ? 1 : 0;
}

sub _get_stat_type_defaults {
    my ($stat_type) = @_;
    return {%DEFAULT_OPTS, countable_fields => $COUNTABLE_FIELDS->{$stat_type}};
}

sub check_select_fields {

    foreach my $stat_type (keys %$COUNTABLE_FIELDS) {
        my @mol_select_fields =
          map {$_->{'mol_id'}}
          grep {_grep_stat_type($_, $stat_type)}
          grep {$_->{'select'}} @Application::Model::BKStatistics::FIELDS;

        my $stat_type_default_opts = _get_stat_type_defaults($stat_type);
        foreach my $field (@mol_select_fields) {
            my $result =
              check_mol_return_succes({%$stat_type_default_opts, countable_fields => [$field],}, "select '$field'",);
            $RESULTS->{$stat_type}->{'select'}->{$field} = $result ? 'ok' : 'fail';
        }
    }
}

sub check_group_by_fields {

    foreach my $stat_type (keys %$COUNTABLE_FIELDS) {
        my @mol_group_by_fields =
          map {$_->{'mol_id'}}
          grep {_grep_stat_type($_, $stat_type)}
          grep {$_->{'group_by'}} @Application::Model::BKStatistics::FIELDS;

        my $stat_type_default_opts = _get_stat_type_defaults($stat_type);
        foreach my $field (@mol_group_by_fields) {
            my $result =
              check_mol_return_succes({%$stat_type_default_opts, group_by => [$field],}, "group_by '$field'",);
            $RESULTS->{'group_by'}->{$field} = $result ? 'ok' : 'fail';
        }

        my $result = check_mol_return_succes(
            {%DEFAULT_OPTS, group_by => \@mol_group_by_fields,},
            sprintf('group_by %s', join(',', @mol_group_by_fields)),
            $EXPECTING_FAIL
        );
        $RESULTS->{$stat_type}->{'group_by'}->{join(',', @mol_group_by_fields)} = $result ? 'faill' : 'ok';
    }
}

sub check_where_fields {

    foreach my $stat_type (keys %$COUNTABLE_FIELDS) {
        my %mol_where_fields =
          map {$_->{'mol_id'} => {get_tree2_type => $_->{'get_tree2_type'}}}
          grep {_grep_stat_type($_, $stat_type)}
          grep {$_->{'where'}} @Application::Model::BKStatistics::FIELDS;

        my $stat_type_default_opts = _get_stat_type_defaults($stat_type);
        foreach my $field (keys(%mol_where_fields)) {
            next if $field eq 'NoSuchFieldInMolPublicID';

            my $query = {
                %$stat_type_default_opts,
                filters_pre => _get_filters_pre(
                    field => $field,
                    type  => $mol_where_fields{$field}{'get_tree2_type'},
                ),
            };

            my $result = check_mol_return_succes($query, "where '$field'");
            $RESULTS->{$stat_type}->{'where'}->{$field} = $result ? 'ok' : 'fail';
        }
    }
}

sub check_compare_periods {

    foreach my $stat_type (keys %$COUNTABLE_FIELDS) {
        my $stat_type_default_opts = _get_stat_type_defaults($stat_type);
        delete $stat_type_default_opts->{date_from};
        delete $stat_type_default_opts->{date_to};
        my $query = {
            %$stat_type_default_opts,
            "date_from_a"    => "20210102",
            "date_from_b"    => "20210101",
            "date_to_a"      => "20210102",
            "date_to_b"      => "20210101",
            countable_fields => $COUNTABLE_FIELDS->{$stat_type}
        };

        my $result = check_mol_return_succes($query, "period_cmp with stat_type [$stat_type]",);
        $RESULTS->{$stat_type}->{'period_cmp'} = $result ? 'ok' : 'fail';
    }
}

sub output_results {
    my %all_fields;

    foreach my $stat_type (keys %$COUNTABLE_FIELDS) {
        foreach my $type (qw(select group_by where)) {
            foreach my $field (keys(%{$RESULTS->{$stat_type}->{$type}})) {
                $all_fields{$field} = 1;
            }
        }

        my $format = '%20s | %20s %20s %20s';
        note sprintf $format, '', 'select', 'where', 'group_by';
        foreach my $field (sort(keys(%all_fields))) {
            note sprintf $format,
              $field,
              $RESULTS->{'select'}{$field}   // '',
              $RESULTS->{'where'}{$field}    // '',
              $RESULTS->{'group_by'}{$field} // '',
              ;
        }
    }
}

sub _get_filters_pre {
    my (%opts) = @_;

    my $field = delete($opts{'field'});
    my $type  = delete($opts{'type'});

    if ($type eq 'text') {
        return {$field => {contains => ['ASDF']}};
    } elsif ($type eq 'publicid') {
        return {$field => {eq => [1]}};
    } elsif ($type eq 'login') {
        return {$field => {contains => ['ASDF']}};
    } elsif ($type eq 'geo') {
        return {$field => {eq => ["225"]}};
    } elsif ($type eq 'multistate') {
        return {$field => {eq => ["ASDF"]}};
    } elsif ($type eq 'number') {
        return {$field => {eq => [1]}};
    } elsif ($type eq 'boolean') {
        return {$field => {eq => [1]}};
    } elsif ($type eq 'tree') {
        return {$field => {eq => [1]}};
    } else {
        die "Uknown type $type";
    }
}

sub main_in_test {
    pass('Loaded ok');

    $MOL_URL = get_mol_url();

    check_select_fields();
    check_where_fields();
    check_group_by_fields();
    check_compare_periods();

    output_results;

    done_testing();
}
main_in_test();
