package Test::Partner2::Database;

=encoding UTF-8

=head1 DESCRIPTION

Модуль для создания отдельных базы данных ПИ2 для тестов.

Использование. В коде теста:

    create_mocked_databases(app => $app);

При старте теста это выполнит следующие действия:

 * создаст базы данных:
    * mocked_partner_db_${pid}_${login}
    * mocked_partner_logs_db_${pid}_${login}
 * создаст таблицы в этих базах на основании данных из
   Application::Model::PartnerDB::*
 * наполнит эти таблицы данными из файлов mocked_databases/*/*

В момент END созданные базы будут удалены.

В сабу create_mocked_databases() можно передать следующие опциональные
параметры:

 * keep_databases - булево значнеие, по дефолту false, если указать true, то
   тест не удалит созданные базы данных, и в выоде теста будет информация
   о базе данных
 * reuse_database - Переиспользует ранее созданную базу (имеет смысл только для READONLY тестов)
 * reuse_db_suffix - cуффикс в названии BD ( в случае Jenkins это $ENV{EXECUTOR_NUMBER}, для TeamCity это $ENV{BUILD_NUMBER} )
 * fill_databases - булево значнеие, по дефолту true, если указать false, то
   база данных не будет наполнена данными из файлов mocked_databases/*/*
 * use_cached_sql - boolean (default TRUE). Использовать промежуточный sql
   (хранится в файловой системе)

=cut

use strict;
use warnings FATAL => 'all';

use Exporter;
use Test::More qw();
use Carp;
use Coro;
use qbit;
use File::Glob qw();
use Test::More;
use Fcntl qw(:flock);

use Test::Partner2::Database::MySQLCreator;
use Test::Partner2::Database::ClickHouseCreator;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(create_mocked_databases save_databases);
our @EXPORT    = @EXPORT_OK;

my %test_databases = ();
my $app;
my $keep_databases;

our @DATABASES = qw(partner_db partner_logs_db kladr);
our $DB_SUITS  = {
    partner_db => {
        # suite       db_tables
        users => [qw( user_role  users  roles )],
    }
};

sub create_mocked_databases {
    my (@opts) = @_;
    # (
    #     app             => $app,
    #     fill_databases  => TRUE,
    #     keep_databases => FALSE,
    #     reuse_database => FALSE,
    #     reuse_db_suffix => '',
    #     use_cached_sql => FALSE,
    #     db_suites      => {
    #         partner_db => [qw( users )],
    #         kladr      => ['*'],
    #     },
    #     reconnect        => TRUE,
    #     mocked_databases => 'mocked_databases_restapi'
    #     create_clickhouse_db => TRUE,
    #);

    croak "create_mocked_databases() must be used with %opts" if @opts % 2;
    my %opts = @opts;

    $app = delete($opts{'app'});
    croak "No app" unless $app;

    my $reuse_database  = delete $opts{'reuse_database'}  // FALSE;
    my $reuse_db_suffix = delete $opts{'reuse_db_suffix'} // 'READONLY';
    $keep_databases = delete($opts{'keep_databases'}) // FALSE;
    $keep_databases ||= $reuse_database;

    my $fill_databases       = delete($opts{'fill_databases'})       // TRUE;
    my $use_cached_sql       = delete($opts{'use_cached_sql'})       // TRUE;
    my $db_suites            = delete $opts{'db_suites'};
    my $reconnect            = delete($opts{'reconnect'})            // FALSE;
    my $mocked_databases     = delete($opts{'mocked_databases'})     // 'mocked_databases';
    my $create_clickhouse_db = delete($opts{'create_clickhouse_db'}) // FALSE;

    croak "Unknown opts: ", join(', ', keys(%opts)) if %opts;

    my $host = $ENV{PARTNER2_TEST_DB_HOST} // 'localhost';
    my $port = $ENV{PARTNER2_TEST_DB_PORT} // '3306';

    my $db_tables = {};

    my @databases = (@DATABASES, $create_clickhouse_db ? 'clickhouse_db' : ());

    foreach my $db_name (@databases) {
        if ($db_suites) {
            my $db_suite_names = $db_suites->{$db_name};
            if ($db_suite_names && @$db_suite_names) {
                if (grep {$_ eq '*'} @$db_suite_names) {
                    $db_tables->{$db_name} = ['*'];
                } elsif ($DB_SUITS->{$db_name}) {
                    foreach my $suit_name (@$db_suite_names) {
                        my $suit_tables = $DB_SUITS->{$db_name}->{$suit_name} // [];
                        my $ar = $db_tables->{$db_name} //= [];
                        push @$ar, @$suit_tables;
                    }
                }
            }
        } else {
            $db_tables->{$db_name} = ['*'];
        }
    }

    my $db_suffix = $reuse_database ? $reuse_db_suffix : $$;
    $db_suffix = substr($db_suffix, -32) if length($db_suffix) > 32;
    $db_suffix =~ s|[/-]|_|g;

    my $dir = $app->get_option('ApplicationPath') . $mocked_databases;

    foreach my $database (@databases) {
        next unless $db_tables->{$database};

        my $login = $ENV{'USER'} // 'unknown';
        $login =~ s/\W/_/g;
        my $db_name = sprintf("mocked_%s_%s_%s", ${database}, $db_suffix, $login);

        my $is_clickhouse_db = $app->$database->isa('QBit::Application::Model::DB::clickhouse');

        my %init_opts = (
            app      => $app,
            database => $database,
            db_name  => $db_name,
            dir      => $dir,
            (
                $is_clickhouse_db
                ? (
                    host => 'localhost',
                    port => 8123,
                  )
                : (
                    host => $host,
                    port => $port,
                  )
            )
        );

        my $db_creator =
          $is_clickhouse_db
          ? Test::Partner2::Database::ClickHouseCreator->new(%init_opts)
          : Test::Partner2::Database::MySQLCreator->new(%init_opts);

        if ($reconnect) {
            $db_creator->reconnect();

            next;
        }

        my $cache_db_file_path = "$dir/$database.sql";
        if (-e $cache_db_file_path) {
            # refresh bunch of all mocked sql (data and struct)
            `make -C $dir -f clean_cache_sql.mk $database.sql`;
        }

        my $fh_db_lock;
        if ($reuse_database) {
            open($fh_db_lock, '>', $app->get_option('ApplicationPath') . ".db_lock");
            flock($fh_db_lock, LOCK_EX);

            $db_creator->check_readonly_db();
        }

        unless ($reuse_database && $db_creator->readonly_db_already_exists) {

            if ($keep_databases) {
                Test::More::note("Creating database '$db_name' at '$host:$port'");
            }

            $db_creator->create_db();

            if ($use_cached_sql) {
              CHECK_FILE: if (-f $cache_db_file_path) {
                    open(my $fh, '<', $cache_db_file_path);

                    flock($fh, LOCK_SH);

                    $db_creator->create_tables($fill_databases);

                    close($fh);
                } else {
                    open(my $fh, '>', $cache_db_file_path);

                    if (flock($fh, LOCK_EX | LOCK_NB)) {
                        $db_creator->create_mocked_database(
                            tables        => ['*'],
                            fill_database => TRUE,
                            cached        => TRUE,
                        );

                        close($fh);

                        goto CHECK_FILE unless $fill_databases;
                    } else {
                        close($fh);

                        goto CHECK_FILE;
                    }
                }
            } else {
                Test::More::note('Start init "' . $database . '" database');

                $db_creator->create_mocked_database(
                    tables        => $db_tables->{$database},
                    fill_database => $fill_databases,
                    cached        => FALSE,
                );
            }
        }

        close($fh_db_lock) if $reuse_database;

        $db_creator->reconnect();

        $test_databases{$database} = $db_name;

        Test::More::note("\t\""
              . $database
              . '" database '
              . ($reuse_database && $db_creator->readonly_db_already_exists ? 'reused' : 'created'));
    }

    if ($reuse_database) {

        foreach my $database (@DATABASES) {
            # лочим на запись, если хотим переиспользовать
            #eval {$app->$database->_do("FLUSH TABLES WITH READ LOCK")} if $reuse_database;

            eval {$app->$database->_do(q[CREATE USER IF NOT EXISTS 'readuser'@'%' IDENTIFIED BY 'pa$$word'])};
            eval {$app->$database->_do(q[GRANT SELECT, SHOW VIEW, PROCESS ON *.* TO 'readuser'@'%'])};
            $app->$database->_do(q[FLUSH PRIVILEGES]);

            $app->$database->close_dbh();
            $app->$database->set_option('user',     'readuser');
            $app->$database->set_option('password', 'pa$$word');
            $app->$database->_connect();
        }
    }
}

sub save_databases {
    my ($app, $folder) = @_;

    my $path = $app->get_option('ApplicationPath') . $folder;

    foreach my $db (@DATABASES) {
        my $mysql_dumper = sprintf(
            '/usr/bin/mysqldump -n \
            --host %s \
            --port %s \
            --default-character-set utf8 \
            -uroot \
            --max_allowed_packet=64K \
            --compact \
            --no-create-info \
            --complete-insert \
            %s',
            $app->$db->get_option('host'),
            $app->$db->get_option('port'),
            $app->$db->get_option('database'),
        );

        my @table_names = keys(%{$app->$db->get_all_meta()->{'tables'}});

        foreach my $table (@table_names) {
            my $table_name = $app->$db->$table->name();

            my $sql = `$mysql_dumper --tables $table_name`;
            chomp($sql);

            next unless $sql;

            $sql =~ s/^([^(]+)\(/$1\n    (/;
            $sql =~ s/\)\s+VALUES\s+/)\nVALUES\n    /;
            $sql =~ s/\),\(/),\n    (/g;

            writefile("$path/$db/$table.sql", "$sql\n");
        }
    }
}

sub _drop_databases {
    foreach my $db (keys(%test_databases)) {
        $app->$db->_do("DROP DATABASE $test_databases{$db}");
    }
}

END {
    unless ($keep_databases) {
        _drop_databases();
    }
}

TRUE;
