package Test::Partner2::Database::ClickHouseCreator;

use qbit;

use base qw(QBit::Class);

use Test::More;

__PACKAGE__->mk_ro_accessors(qw(app database db_name host port dir));

__PACKAGE__->mk_accessors(qw(database_settings readonly_db_already_exists));

my $CH_CLIENT_PORT = 9000;

sub init {
    my ($self) = @_;

    my $database = $self->database;

    my $meta = $self->app->$database->get_all_meta();

    #в тестах используем только SummingMergeTree
    foreach my $table (keys(%{$meta->{'tables'}})) {
        my $engine = $self->app->$database->$table->engine();

        if ($engine->{'ReplicatedSummingMergeTree'}) {
            $engine->{'SummingMergeTree'} = [splice(@{delete($engine->{'ReplicatedSummingMergeTree'})}, 2)];

            $self->app->$database->$table->{'engine'} = $engine;
        } elsif ($engine->{'Dictionary'}) {
            $self->app->$database->$table->{'engine'} = {StripeLog => []};
        }
        delete $self->app->$database->$table->{'partition_by'};
    }

    $self->database_settings(
        {
            'output_format_json_quote_64bit_integers' =>
              $self->app->$database->get_option('output_format_json_quote_64bit_integers'),
            'user'     => 'default',
            'password' => '',
            'host'     => $self->host,
            'port'     => $self->port,
        }
    );

    $self->app->set_option($database => $self->database_settings);

    $self->app->$database->_connect();
}

sub reconnect {
    my ($self) = @_;

    my $database = $self->database;

    $self->app->set_option($database => {%{$self->database_settings}, database => $self->db_name});

    my $cur_dbh_key = $self->app->$database->get_key();

    delete($self->app->$database->{'__DBH__'}{$cur_dbh_key});

    $self->app->$database->_connect();
}

sub check_readonly_db {
    my ($self) = @_;

    my $dir      = $self->dir;
    my $database = $self->database;
    my $db_name  = $self->db_name;

    $self->readonly_db_already_exists(FALSE);

    my $res = $self->app->$database->_get_all(sprintf q[show databases format JSONCompact]);

    foreach my $row (@$res) {
        if ($row->{'name'} eq $db_name) {
            if (-e "$dir/$database.sql") {
                $self->readonly_db_already_exists(TRUE);
            } else {
                $self->app->$database->_do("drop database $db_name");
            }

            last;
        }
    }
}

sub create_db {
    my ($self) = @_;

    my $database = $self->database;
    my $db_name  = $self->db_name;

    $self->app->$database->_do("CREATE DATABASE IF NOT EXISTS $db_name");
    $self->app->$database->set_option('database', $db_name);

    $self->reconnect();
}

sub get_db_connect_cmd {
    my ($self) = @_;

    my $db_name = $self->db_name;

    return "clickhouse-client -m --port=$CH_CLIENT_PORT --user=default -d $db_name";
}

sub create_tables {
    my ($self, $fill_databases) = @_;

    my $database = $self->database;
    my $dir      = $self->dir;

    my $db_connect_cmd = $self->get_db_connect_cmd();

    my $meta = $self->app->$database->get_all_meta();

    foreach my $table (keys(%{$meta->{'tables'}})) {
        my $file_name = "$dir/${database}_struct__$table.sql";

        `cat $file_name | $db_connect_cmd`;

        $file_name = "$dir/${database}_data__$table.native";

        if ($fill_databases && -s $file_name) {
            `$db_connect_cmd --query="INSERT INTO $table FORMAT Native" < $file_name`;
        }
    }
}

sub create_mocked_database {
    my ($self, %opts) = @_;

    my $dir      = $self->dir;
    my $db_name  = $self->db_name;
    my $database = $self->database;

    my $app = $self->app;

    my $tables        = $opts{'tables'};
    my $cached        = $opts{'cached'};
    my $fill_database = $opts{'fill_database'};

    my $need_all_data = $tables->[0] eq '*';

    Test::More::note('Start init "' . $database . '" database');
    my $tables_list =
        $tables && @$tables && !$need_all_data
      ? $tables
      : undef;

    $app->$database->init_db($tables_list);

    my $clickhouse_cmd = sprintf(
        'clickhouse-client -m --host=%s --port=%s --user=%s -d %s',
        $app->$database->get_option('host'),
        9000, $app->$database->get_option('user'), $db_name
    );

    if ($cached) {
        my $meta = $app->$database->get_all_meta();

        foreach my $table (keys(%{$meta->{'tables'}})) {
            writefile("$dir/${database}_struct__$table.sql", $app->$database->$table->create_sql);
        }
    }

    if ($fill_database) {
        my $path = "$dir/$database";
        if (-d $path) {
            my @table_names = ();
            if ($need_all_data) {
                @table_names = map {chomp; $_ =~ s/\.sql\z//; $_} `cd $path; ls -1 *`;
            } else {
                @table_names = @$tables;
            }

            Test::More::note("\tStart fill \"" . scalar(@table_names) . '" tables: ',
                substr(join(", ", @table_names), 0, 100));

            # Keep files order
            my @sort_tables = $app->$database->_sorted_tables(@table_names);
            foreach my $table (@sort_tables) {
                my $cmd = sprintf 'cd %s; cat %s.sql | %s', $path, $table, $clickhouse_cmd;

                # Fill tables
                `$cmd`;
                throw qq[SQL error in "$path/$table.sql"] if $? != 0;
            }
        }

        if ($cached) {
            my $meta = $app->$database->get_all_meta();

            foreach my $table (keys(%{$meta->{'tables'}})) {
                `$clickhouse_cmd --query="SELECT * FROM $table FORMAT Native" > $dir/${database}_data__$table.native`;
            }

            writefile("$dir/$database.sql", 'FAKE SQL');
        }
    }
}

TRUE;
