package Test::Partner2::Database::MySQLCreator;

use qbit;

use base qw(QBit::Class);

use Test::More;

__PACKAGE__->mk_ro_accessors(qw(app database db_name host port dir));

__PACKAGE__->mk_accessors(qw(database_settings readonly_db_already_exists));

sub init {
    my ($self) = @_;

    $self->database_settings(
        {
            'user'     => 'root',
            'password' => '',
            'host'     => $self->host,
            'port'     => $self->port,
        }
    );

    my $database = $self->database;

    $self->app->set_option($database => $self->database_settings);

    $self->app->$database->_connect();
}

sub reconnect {
    my ($self) = @_;

    $self->app->{__ORIG_OPTIONS__}{$self->database} = {%{$self->database_settings}, database => $self->db_name};
    $self->app->set_option($self->database => {%{$self->database_settings}, database => $self->db_name});
}

sub check_readonly_db {
    my ($self) = @_;

    my $dir      = $self->dir;
    my $database = $self->database;
    my $db_name  = $self->db_name;

    $self->readonly_db_already_exists(FALSE);

    my $res = $self->app->$database->_get_all(sprintf q[show databases like '%s'], $db_name);
    if ($res && @$res) {
        $self->app->$database->_do("USE $db_name");

        if (-e "$dir/$database.sql") {
            $self->readonly_db_already_exists(TRUE);
        } else {
            $self->app->$database->_do("drop database $db_name");
        }
    }
}

sub create_db {
    my ($self) = @_;

    my $database = $self->database;
    my $db_name  = $self->db_name;

    $self->app->$database->_do("CREATE DATABASE IF NOT EXISTS $db_name");
    $self->app->$database->_do("USE $db_name");
}

sub get_db_connect_cmd {
    my ($self) = @_;

    my $host    = $self->host;
    my $port    = $self->port;
    my $db_name = $self->db_name;

    return "mysql --host $host --port $port --default-character-set utf8 -uroot $db_name";
}

sub create_tables {
    my ($self, $fill_databases) = @_;

    my $dir      = $self->dir;
    my $database = $self->database;

    my $file_name = $fill_databases ? "$dir/$database.sql" : "$dir/${database}_struct.sql";

    my $db_connect_cmd = $self->get_db_connect_cmd();

    local $ENV{LC_ALL} = 'C';
    `cat $file_name | $db_connect_cmd`;
}

sub create_mocked_database {
    my ($self, %opts) = @_;

    my $dir      = $self->dir;
    my $db_name  = $self->db_name;
    my $database = $self->database;

    my $app = $self->app;

    my $tables        = $opts{'tables'};
    my $cached        = $opts{'cached'};
    my $fill_database = $opts{'fill_database'};

    my $need_all_data = $tables->[0] eq '*';

    Test::More::note('Start init "' . $database . '" database');
    my $tables_list =
        $tables && @$tables && !$need_all_data
      ? $tables
      : undef;

    $app->$database->init_db($tables_list);

    my $mysql_cmd = sprintf(
        'mysql --host %s --port %s --default-character-set utf8 -u%s %s',
        $app->$database->get_option('host'),
        $app->$database->get_option('port'),
        $app->$database->get_option('user'), $db_name
    );

    my $mysql_dumper = sprintf(
'/usr/bin/mysqldump -n --host %s --port %s --default-character-set utf8 -uroot --extended-insert --add-locks --triggers',
        $app->$database->get_option('host'),
        $app->$database->get_option('port')
    );

    `cat $dir/../MySQL/${database}/*.sql | $mysql_cmd`;

    if ($cached) {
        `$mysql_dumper --no-data $db_name > $dir/${database}_struct.sql`;
    }

    if ($fill_database) {
        $app->$database->_do("SET GLOBAL sql_mode='NO_AUTO_VALUE_ON_ZERO';");

        my $path = "$dir/$database";
        if (-d $path) {
            my @table_names = ();
            if ($need_all_data) {
                @table_names = map {chomp; $_ =~ s/\.sql\z//; $_} `cd $path; ls -1 *`;
            } else {
                @table_names = @$tables;
            }

            Test::More::note("\tStart fill \"" . scalar(@table_names) . '" tables: ',
                substr(join(", ", @table_names), 0, 100));

            # Keep files order
            my @sort_tables = $app->$database->_sorted_tables(@table_names);
            foreach my $table (@sort_tables) {
                my $cmd = sprintf 'cd %s; cat %s.sql | %s', $path, $table, $mysql_cmd;

                # Fill tables
                `$cmd`;
                throw qq[SQL error in "$path/$table.sql"] if $? != 0;
            }
        }

        if ($cached) {
            `$mysql_dumper $db_name > $dir/$database.sql`;
        }
    }
}

TRUE;
