package Test::Partner2::Mock;

use strict;
use warnings FATAL => 'all';

BEGIN {
    warn "Test::MockTime load after qbit: mock_time will be broken" if $INC{'qbit.pm'} && !$INC{'Test/MockTime.pm'};
}
use Test::MockTime qw();    #keep this before including qbit
use Test::MockObject::Extends::Easy;
use Scalar::Util qw(refaddr);
use qbit;

use lib::abs;

use Exception;
use Exception::BlackBox::NeedAuth;
use Exception::BlackBox::NeedRealLogin;
use Exception::BlackBox::NeedResign;
use Exception::API::AdFoxGraphQL;
use Exception::API::FormatSystem;
use Exception::MoneyMap;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  apply_mocks
  get_default_mock_options
  mock_adfox
  mock_adfox_graphql
  mock_adfox_graphql_error
  mock_agreement_checker
  mock_api_java_bk_data
  mock_api_rmp
  mock_balance
  mock_bk
  mock_bk_data_validator
  mock_blackbox
  mock_check_statistics_by_blocks
  mock_edit_cooperation_form
  mock_curdate
  mock_geobase
  mock_get_lock
  mock_http
  mock_juggler
  mock_lang_detect
  mock_mailer
  mock_maps
  mock_mds_avatars
  mock_mol
  mock_moneymap
  mock_moneymap_error
  mock_news
  mock_oebsapi
  mock_selfservice
  mock_send_to_graphite
  mock_sentry
  mock_sort_sql
  mock_subs
  mock_text_template
  mock_time
  mock_tvm
  mock_selfcheck
  mock_utils_partner2
  mock_zora
  mock_statistics_is_available
  mock_format_system
  mock_yt_partner
  mock_yt_picategory_mapping
  restore_subs
  restore_time
  mock_memcached
  mock_for_config
  mock_yt_requests
  mock_java_jsonapi
  mock_page_validator
  mock_model_methods
  mock_features
  reset_features
  mock_host
  );
our @EXPORT = @EXPORT_OK;

our $PAGE_ID_START = 1000;

our $RUN_IN_EDIT_PAGE = sub {
    my ($data) = @_;

    return 1;
};

our $RUN_IN_CREATE_OR_UPDATE_PLACE = sub {
    my ($operator_uid, $opts) = @_;

    return 1;
};

sub get_default_mock_options {
    my ($is_tjson) = @_;

    my @default_mocks = ();
    if ($is_tjson) {
        @default_mocks = qw(
          mock_blackbox
          mock_format_system
          mock_host
          mock_mailer
          mock_maps
          mock_moneymap
          mock_oebsapi
          mock_yt_requests
          );
    } else {
        @default_mocks = qw(
          mock_api_rmp
          mock_juggler
          mock_maps
          mock_moneymap
          mock_page_validator
          mock_push_client_add_objects
          mock_selfservice
          mock_send_to_graphite
          mock_yt_partner
          );
    }

    my $default_options = {};
    foreach my $key (@default_mocks) {
        $default_options->{$key} //= TRUE;
    }

    return $default_options;
}

sub apply_mocks {
    my ($app, $mock_options_precedence, $is_tjson) = @_;

    my $common_options = {map {%{$_ // {}}} @$mock_options_precedence};

    no strict 'refs';
    foreach my $key (keys %$common_options) {
        if ($key =~ /^mock_/ && $common_options->{$key}) {
            my $sub_name = sprintf 'Test::Partner2::Mock::%s', $key;
            if (exists &{$sub_name}) {
                $sub_name->($app, $common_options->{$key});
            }
        }
    }

    return 1;
}

sub mock_time {
    my ($time_str, $format_str) = @_[-2, -1];

    $time_str //= curdate(oformat => 'db_time');
    $format_str //= '%Y-%m-%d %H:%M:%S';

    Test::MockTime::set_fixed_time($time_str, $format_str);

    return 1;
}

sub restore_time {
    Test::MockTime::restore_time();
    return 1;
}

sub mock_check_statistics_by_blocks {
    my ($app) = @_;

    $app->statistics_advnet_context_on_site_rtb;

    mock_subs({'Application::Model::Statistics::Product::check_statistics_by_blocks' => sub { },});
}

sub mock_curdate {
    my ($curdate) = $_[-1];

    croak("Curdate for mocking isn't gotten.") unless $curdate;

    no strict 'refs';
    no warnings 'redefine';

    my $sub_curdate = sub {
        my (%opts) = @_;
        return trdate('db_time', $opts{oformat} || 'norm', $curdate);
    };

    my @packages_having_curdate = grep {defined(&{$_ . '::curdate'})} (map {s'\.pm''; s'/'::'g; $_} keys(%INC)), 'main';
    *{$_ . '::curdate'} = $sub_curdate foreach @packages_having_curdate;

    mock_time($curdate, '%Y-%m-%d %H:%M:%S');
    mock_gettimeofday(trdate(db_time => sec => $curdate));
}

sub mock_gettimeofday {
    my ($tm) = $_[-1];

    my $addr = refaddr(\&Time::HiRes::gettimeofday);
    my @packages_having_gettimeofday =
      grep {defined(&{$_ . '::gettimeofday'}) && refaddr(\&{$_ . '::gettimeofday'}) == $addr}
      (map {s'\.pm''; s'/'::'g; $_} keys(%INC)), 'main';
    my $sub = sub() {return wantarray ? ($tm, 0) : $tm;};
    {
        no strict 'refs';
        no warnings 'redefine';
        *{$_ . '::gettimeofday'} = $sub foreach @packages_having_gettimeofday;
    }
}

sub mock_agreement_checker {
    my ($app) = @_;

    $app->agreement_checker;
    $app->{'agreement_checker'} = Test::MockObject::Extends::Easy->new($app->{'agreement_checker'});

    $app->agreement_checker->mock(
        'get_data_for_client_id',
        sub {
            my $all  = $app->agreement_checker->get_all_products();
            my $data = {
                today    => $all,
                tomorrow => $all,
            };
        }
    );
}

sub mock_balance {
    my ($app) = @_;

    $app->documents;
    $app->{'documents'} = Test::MockObject::Extends->new($app->{'documents'});
    $app->{'documents'}->mock('_is_live', sub {TRUE});

    $app->api_balance;
    $app->{'api_balance'} = Test::MockObject::Extends->new($app->{'api_balance'});

    $app->{'api_balance'}->mock(
        'call',
        sub {
            my ($self, $method, $operator_uid, $opts) = @_;

            if ($method eq 'Balance.GetPartnerContracts') {
                return [
                    [
                        {
                            Contract => {
                                person_id     => 1,
                                type          => 'PARTNERS',
                                contract_type => 1,
                                contract2_id  => 123456,
                                dt            => '',
                                end_dt        => '',
                                is_signed     => 1,
                                firm          => 9,
                                nds           => 1,
                                currency      => 643,
                            },
                            Person      => {id => "666", bik => '12345678'},
                            Collaterals => [],
                        }
                    ]
                ];
            }
            if ($method eq 'Balance.FindClient') {
                return [0];
            }
            if ($method eq 'Balance.CreateClient') {
                return [0, 'SUCCESS', "1$operator_uid"];
            }
            if ($method eq 'Balance2.CreateUserClientAssociation') {
                return [[0, 'SUCCESS', "{}"]];
            }
            if ($method eq 'Balance.GetClientPersons') {
                return [[{id => "666"},]];
            }
            if ($method eq 'Balance2.CreatePerson') {
                return [[]];
            }
            if ($method eq 'Balance2.CreateOffer') {
                return [{'EXTERNAL_ID' => '702491/19', 'ID' => '111666'}];
            }
            if ($method eq 'Balance2.GetBank') {
                return [{name => 'bank_name', hidden => '0'}]
                  if $operator_uid->{Bik};

                return [{name => 'bank_name', hidden_dt => ''}]
                  if $operator_uid->{Swift};
            }
            if ($method eq 'Balance2.CreateCommonContract') {
                return [{'EXTERNAL_ID' => '702491/19', 'ID' => '111666'}];
            }
            if ($method eq 'Balance.CreateCollateral') {
                return [
                    {
                        COLLATERAL_NUM       => '01',
                        CONTRACT_EXTERNAL_ID => 'РС-149159-10/20',
                        CONTRACT_ID          => '2217433',
                    }
                ];
            }
            if ($method eq 'Balance.CreateOrUpdatePlace') {
                $RUN_IN_CREATE_OR_UPDATE_PLACE->($operator_uid, $opts);
                return [0];
            }

            throw "This mock not works with $method";
        }
    );
}

sub mock_geobase {
    my ($app) = @_;

    $app->api_geobase;
    $app->{'api_geobase'} = Test::MockObject::Extends->new($app->{'api_geobase'});

    $app->api_geobase->mock('get_region_parents_by_ip', sub {return [10000];});    #регион Земля
    $app->api_geobase->mock('get_region_parents_by_id', sub {return [10000];});    #регион Земля
}

sub mock_utils_partner2 {
    my ($app) = @_;

    $app->api_utils_partner2;
    $app->{'api_utils_partner2'} = Test::MockObject::Extends->new($app->{'api_utils_partner2'});

    $app->api_utils_partner2->mock(
        'get_next_page_id',
        sub {
            return $PAGE_ID_START++;
        }
    );
}

sub mock_bk {
    my ($app) = @_;

    $app->api_bk;
    $app->{'api_bk'} = Test::MockObject::Extends->new($app->{'api_bk'});

    $app->api_bk->mock(
        '_set_page_id_if_need_and_call',
        sub {
            my ($self, $method, $data, @opts) = @_;

            my @numbers = keys %$data;
            throw 'Incorrect number of campaigns send to EditPage' if @numbers != 1;

            my $page_id = $data->{$numbers[0]}->{PageID};

            unless ($page_id) {
                $page_id = $PAGE_ID_START;
                $PAGE_ID_START++;
            }

            $RUN_IN_EDIT_PAGE->($data, @opts);

            return [
                {
                    $numbers[0] => {
                        'PageID'   => $page_id,
                        'Error'    => 0,
                        'ErrorStr' => '',
                    }
                }
            ];
        }
    );

    $app->api_http_bk;
    $app->{'api_http_bk'} = Test::MockObject::Extends->new($app->{'api_http_bk'});

    $app->api_http_bk->mock('import_cookie_match_settings', sub { });
    $app->api_http_bk->mock(stop_dsp  => sub { });
    $app->api_http_bk->mock(edit_dsp  => sub { });
    $app->api_http_bk->mock(start_dsp => sub { });
    $app->api_http_bk->mock(dsp_lb    => sub { });
}

sub mock_bk_data_validator {
    use QBit::Validator::Type::bk_data::block;

    my ($app) = @_;

    {
        no strict 'refs';
        no warnings 'redefine';

        *{'QBit::Validator::Type::bk_data::block::validate'} = sub { };
    }
}

sub mock_oebsapi {
    my ($app) = @_;

    $app->api_oebs;
    $app->{'api_oebs'} = Test::MockObject::Extends->new($app->{'api_oebs'});

    $app->api_oebs->mock(
        'get_partner_act_headers',
        sub {
            my ($self, $client_id) = @_;

            throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;

            return [
                {
                    "doc_size"             => 35110,
                    "receive_akt_status"   => "N",
                    "receive_schet_status" => "N",
                    "external_id"          => "РС-28111-01/15",
                    "send_status"          => "Y",
                    "period_end_date"      => "2018-12-31T00:00:00",
                    "period_start_date"    => "2015-01-01T00:00:00",
                    "multistate_name"      => "Please send the original",
                    "contract_id"          => "123456",
                    "receive_status"       => "N",
                    "receive_sf_status"    => "N",
                    "receive_doc_status"   => "N",
                    "seen_status"          => "N",
                    "send_original_status" => "N",
                    "id"                   => 969305,
                    "paid_status"          => undef
                }
            ];
        }
    );
    $app->api_oebs->mock(
        'get_partner_act_contents',
        sub {
            my ($self, $client_id, $doc_id, %opts) = @_;

            throw Exception::Validation::BadArguments gettext("'%s': is missing", 'client_id') unless $client_id;
            throw Exception::Validation::BadArguments gettext("'%s': is missing", 'doc_id')    unless $doc_id;
            return [
                {
                    "doc_format" => "XML Excel",
                    "doc_body" =>
"PD94bWwgdmVyc2lvbj0iMS4wIiBlbmNvZGluZz0iVVRGLTgiPz48P21zby1hcHBs\r\naWNhdGlvbiBwcm9naWQ9IkV4Y2VsLlNoZWV0Ij8+PFdvcmtib29rIHhtbG5zPSJ1\r\ncm46c2NoZW1hcy1taWNyb3NvZnQtY29tOm9mZmljZTpzcHJlYWRzaGVldCIgeG1s\r\nbnM6bz0idXJuOnNjaGVtYXMtbWljcm9zb2Z0LWNvbTpvZmZpY2U6b2ZmaWNlIiB4\r\nbWxuczp4PSJ1cm46c2NoZW1hcy1taWNyb3NvZnQtY29tOm9mZmljZTpleGNlbCIg\r\neG1sbnM6c3M9InVybjpzY2hlbWFzLW1pY3Jvc29mdC1jb206b2ZmaWNlOnNwcmVh\r\nZHNoZWV0IiB4bWxuczpodG1sPSJodHRwOi8vd3d3LnczLm9yZyPFBhZ2VTZXR1cD4gICAgPFBh\r\nZ2VNYXJnaW5zIHg6Qm90dG9tPSIwLjM1NDMzMDcwODY2MTQxNzM2IiB4OkxlZnQ9\r\nIjAuNDMzMDcwODY2MTQxNzMyMjkiIHg6UmlnaHQ9IjAuMjM2MjIwNDcyNDQwOTQ0\r\nOTEiIHg6VG9wPSIwLjM1NDMzMDcwODY2MTQxNzM2Ii8+ICAgPC9QYWdlU2V0dXA+\r\nICAgPFByaW50PiAgICA8VmFsaWRQcmludGVySW5mby8+ICAgIDxTY2FsZT44MDwv\r\nU2NhbGU+ICAgPC9QcmludD4gICA8VGFiQ29sb3JJbmRleD41MTwvVGFiQ29sb3JJ\r\nbmRleD4gICA8Wm9vbT4xMDA8L1pvb20+ICAgPERvTm90RGlzcGxheUdyaWRsaW5l\r\ncy8+ICAgPFByb3RlY3RPYmplY3RzPkZhbHNlPC9Qcm90ZWN0T2JqZWN0cz4gICA8\r\nUHJvdGVjdFNjZW5hcmlvcz5GYWxzZTwvUHJvdGVjdFNjZW5hcmlvcz4gIDwvV29y\r\na3NoZWV0T3B0aW9ucz4gPC9Xb3Jrc2hlZXQ+PC9Xb3JrYm9vaz4="
                }
            ];

        }
    );

}

sub mock_lang_detect {
    my ($app, $lang) = @_;

    $app->lang_detect;
    $app->{'lang_detect'} = Test::MockObject::Extends->new($app->{'lang_detect'});

    $app->lang_detect->mock('get_user_lang', sub {return $lang;});
}

sub mock_zora {
    my ($app) = @_;

    $app->api_http_gozora;
    $app->{'api_http_gozora'} = Test::MockObject::Extends->new($app->{'api_http_gozora'});

    $app->api_http_gozora->mock('get_page_inf', sub {return {body_length_bytes => 1}});
}

sub mock_mailer {
    my ($app) = @_;

    $app->mailer;
    $app->{'mailer'} = Test::MockObject::Extends->new($app->{'mailer'});

    $app->mailer->mock(
        'send',
        sub {
            TRUE;
        }
    );
}

sub mock_mol {
    my ($app, $data) = @_;

    $app->api_http_mol;
    $app->{'api_http_mol'} = Test::MockObject::Extends->new($app->{'api_http_mol'});

    $data = {
        header     => [],
        data       => [],
        error_text => "",
        status     => 0,
        total_rows => 0,
        totals     => {},
      }
      unless ref $data eq 'HASH';

    my $cdata = clone($data);
    $app->api_http_mol->mock(
        'get_data',
        sub {
            return $cdata;
        }
    );
}

sub mock_maps {
    my ($app, $sub) = @_;

    $app->api_http_maps;
    $app->{'api_http_maps'} = Test::MockObject::Extends->new($app->{'api_http_maps'});

    $app->api_http_maps->mock(
        'check_business_oid',
        sub {
            return {
                result  => 1,
                message => ""
            };
        }
    );
    $app->api_http_maps->mock(
        'get_companies_by_ll',
        sub {
            return [
                {
                    business_oid => 1132721398,
                    name         => 'mocked organization',
                    categories   => ['mocked category'],
                },
                {
                    business_oid => 123,
                    name         => 'another mocked organization',
                    categories   => ['another mocked category'],
                },
            ];
        }
    );
}

sub mock_news {
    my ($app) = @_;

    $app->api_http_news;
    $app->{'api_http_news'} = Test::MockObject::Extends->new($app->{'api_http_news'});

    $app->api_http_news->mock(
        'check_sources',
        sub {
            return {
                result  => 1,
                message => ""
            };
        }
    );
}

sub mock_model_methods {
    my ($app, $data) = @_;

    foreach my $accessor (keys %$data) {
        $app->$accessor;
        $app->{$accessor} = Test::MockObject::Extends->new($app->{$accessor});

        foreach my $method (keys %{$data->{$accessor}}) {
            $app->$accessor->mock(
                $method,
                sub {
                    return $data->{$accessor}->{$method} // FALSE;
                }
            );
        }
    }
}

sub mock_tvm {
    my ($app, $data) = @_;

    $app->api_tvm;
    $app->{api_tvm} = Test::MockObject::Extends->new($app->{api_tvm});
    $app->api_tvm->mock(
        'check_ticket',
        sub {
            return $data;
        }
    );
}

sub mock_adfox {
    my ($app, $sub) = @_;

    $app->api_adfox;
    $app->{'api_adfox'} = Test::MockObject::Extends->new($app->{'api_adfox'});

    $app->api_adfox->mock(
        'get_adfox_user_info',
        sub {
            return {
                firstName => "",
                lastName  => "",
                login     => "sdelkino.com"
            };
        }
    );

    $app->api_adfox_graphql;
    $app->{'api_adfox_graphql'} = Test::MockObject::Extends->new($app->{'api_adfox_graphql'});

    $app->api_adfox_graphql->mock(
        'bind_user',
        sub {
            return {
                id    => 111,
                login => "sdelkino.com"
            };
        }
    );

    $app->api_adfox_graphql->mock(
        'set_billing_data',
        sub {
            return {
                id    => 111,
                login => "sdelkino.com"
            };
        }
    );

    for my $sub_name (qw/update_page_name update_block_name delete_block restore_block start_stop_page/) {
        $app->api_adfox_graphql->mock(
            $sub_name => sub {
                shift;
                if (ref($sub) eq 'CODE') {
                    return $sub->($sub_name, @_);
                } else {
                    return $sub;
                }
            }
        );
    }
}

sub mock_adfox_graphql {
    my ($app) = @_;

    $app->api_adfox_graphql;
    $app->{'api_adfox_graphql'} = Test::MockObject::Extends->new($app->{'api_adfox_graphql'});

    $app->api_adfox_graphql->mock(
        'create_user',
        sub {
            {
                id    => 123000,
                login => 'mocked_adfox_login',
            };
        }
    );
}

sub mock_adfox_graphql_error {
    my ($app) = @_;

    $app->api_adfox_graphql;
    $app->{'api_adfox_graphql'} = Test::MockObject::Extends->new($app->{'api_adfox_graphql'});

    $app->api_adfox_graphql->mock(
        'create_user',
        sub {
            throw Exception::API::AdFoxGraphQL('Incorrect answer');
        }
    );
}

sub mock_selfcheck {
    require Application::Model::CheckAlive;
    mock_subs(
        {
            'Application::Model::CheckAlive::_is_alive' => sub {
                return (0, []);
              }
        }
    );
}

sub mock_juggler {
    mock_subs(
        {
            'Partner2::Juggler::API::send' => sub { }
        }
    );
}

sub mock_sentry {
    my ($app, $sub) = @_;

    $sub //= sub {TRUE};

    $app->sentry;
    $app->{'sentry'} = Test::MockObject::Extends->new($app->{'sentry'});

    $app->sentry->mock('send_exception', $sub);
}

sub mock_blackbox {
    my ($app) = @_;

    $app->api_blackbox;
    $app->{'api_blackbox'} = Test::MockObject::Extends->new($app->{'api_blackbox'});

    $app->api_blackbox->mock(
        'sessionid',
        sub {
            my ($self, %opts) = @_;

            my %pass_data = (url_auth => "https://passport.yandex.ru/passport?mode=auth");

            throw Exception::BlackBox::NeedAuth 'No session', %pass_data
              if !$opts{session_id};

            if ($opts{session_id} eq 'need_auth') {
                throw Exception::BlackBox::NeedAuth 'Exception from mocked blackbox';
            } elsif ($opts{session_id} eq 'need_resign') {
                throw Exception::BlackBox::NeedResign 'Exception from mocked blackbox';
            } elsif ($opts{session_id} eq 'need_real_login') {
                throw Exception::BlackBox::NeedRealLogin 'Exception from mocked blackbox';
            } elsif ($opts{session_id} eq 'exception') {
                throw Exception 'Exception from mocked blackbox';
            } else {
                return {
                    'auth'    => {'secure' => {'content' => '1'},},
                    'dbfield' => {
                        'accounts.login.uid'   => 'yndx-developer',
                        'account_info.fio.uid' => 'Pupkin Vasily',
                        'subscription.suid.85' => undef
                    },
                    'uid'          => {'content' => '731583645',},
                    'login'        => {'content' => 'yndx.developer'},
                    'address-list' => {'address' => {'content' => 'yndx.developer@yandex.ru',}}
                };
            }
        }
    );

    $app->api_blackbox->mock(
        'get_user_info',
        sub {
            my ($self, $user_id) = @_;

            throw Exception 'Exception from mocked blackbox'
              unless defined $user_id;
            return {'language' => 'ru'};
        }
    );

    $app->api_blackbox->mock(
        'get_users_avatar_and_lang',
        sub {
            my ($self, @ids) = @_;

            return +{map {$_ => {avatar => '0/0-0', 'lang' => 'ru',}} @ids};
        }
    );
}

sub mock_get_lock {
    no strict 'refs';
    no warnings 'redefine';
    *{'QBit::Cron::get_lock'} = sub {1};
}

my $format_240x400        = from_json(readfile(lib::abs::path('./Mock/api_format_system_formats_240x400.json')));
my $format_240x400_mobile = from_json(readfile(lib::abs::path('./Mock/api_format_system_formats_240x400_mobile.json')));
my $format_adaptive       = from_json(readfile(lib::abs::path('./Mock/api_format_system_formats_adaptive.json')));
my $format_adaptive0418_mobile =
  from_json(readfile(lib::abs::path('./Mock/api_format_system_formats_adaptive0418_mobile.json')));
my $format_list = from_json(readfile(lib::abs::path('./Mock/api_format_system/format_list.json')));

sub mock_format_system {
    my ($app) = @_;

    $app->api_format_system;
    $app->{'api_format_system'} = Test::MockObject::Extends->new($app->{'api_format_system'});

    $app->api_format_system->mock(
        'validate',
        sub {
            my ($self, %opts) = @_;

            my $design = $opts{design};
            my $items;

            if ($design->{name} eq 'zen') {
                throw Exception::API::FormatSystem "Формат не найден" unless $opts{role} eq 'manager';
            }

            foreach ('name', 'limit',
                (($opts{role} eq 'partner' && $design->{name} eq 'adaptive') ? ('width', 'height') : ()))
            {
                $items->{$_} = {
                    valid    => 0,
                    messages => [
                        {
                            type => 'ERR',
                            text => 'Mocked error message for missed mandatory field',
                        }
                    ],
                  }
                  unless exists $design->{$_};
            }
            return {
                valid    => 0,
                items    => $items,
                messages => [],
            } if $items;
            if (($design->{limit} // 0) > 10) {
                return {
                    valid => 0,
                    items => {
                        limit => {
                            valid    => 0,
                            messages => [
                                {
                                    type => 'ERR',
                                    text => 'Mocked error message for limit out of bounds',
                                }
                            ],
                        },
                    },
                    messages => [],
                };
            } else {
                return {
                    valid    => 1,
                    items    => {},
                    messages => [],
                };
            }
        }
    );

    $app->api_format_system->mock(
        'formats',
        sub {
            my ($self, %opts) = @_;

            if (defined $opts{format}) {
                return $format_adaptive            if $opts{format} eq 'adaptive';
                return $format_240x400_mobile      if $opts{format} eq '240x400' && $opts{site} eq 'mobile';
                return $format_adaptive0418_mobile if $opts{format} eq 'adaptive0418' && $opts{site} eq 'mobile';
                return $format_240x400;
            }
            my $form_factor = $opts{form_factor} // 'vertical';
            my $site        = $opts{site}        // 'desktop';
            return $format_list->{$site}->{$form_factor};
        }
    );

    $app->api_format_system->mock(
        'get_design_limits',
        sub {
            my ($self, %opts) = @_;
            return {};
        }
    );
}

=head2
    mock requests to YT [//home/partner/**]
=cut

sub mock_yt_partner {
    my ($app) = @_;

    $app->api_yt_partner_target_tags;
    $app->{'api_yt_partner_target_tags'} = Test::MockObject::Extends->new($app->{'api_yt_partner_target_tags'});

    $app->api_yt_partner_target_tags->mock(
        'get_target_tags',
        sub {
            return [
                {"id" => 1, "name" => "name1", "descr" => "descr1"},
                {"id" => 2, "name" => "name2", "descr" => "descr2"},
                {"id" => 3, "name" => "name3", "descr" => "descr3"},
                {"id" => 4, "name" => "name4", "descr" => "descr4"},
                {"id" => 5, "name" => "name5", "descr" => "descr5"},
                {"id" => 6, "name" => "name6", "descr" => "descr6"},
            ];
        }
    );
}

sub mock_yt_picategory_mapping {
    my ($app, $map) = @_;

    unless ($map and ref $map eq 'HASH') {
        $map = {
            TNS_PI => {
                40 => [qw(1 2 3)],
                41 => [qw(52 53 54 55)],
            },
            PI_TNS => {
                1  => [40],
                2  => [40],
                3  => [40],
                52 => [41],
                53 => [41],
                54 => [41],
                54 => [41],
            }
        };
    } elsif (!(ref $map->{TNS_PI} eq 'HASH' and ref $map->{PI_TNS} eq 'HASH')) {
        throw Exception "Bad format mapping.\n";
    }

    $app->api_yt_partner_picategory_mapping;
    $app->{'api_yt_partner_picategory_mapping'} =
      Test::MockObject::Extends->new($app->{'api_yt_partner_picategory_mapping'});

    $app->api_yt_partner_picategory_mapping->mock(
        'get_picategory_articles_mapping',
        sub {
            return $map;
        }
    );
}

sub mock_yt_requests {
    my ($app) = @_;

    $app->api_yt;
    $app->{'api_yt'} = Test::MockObject::Extends->new($app->{'api_yt'});

    $app->api_yt->mock(
        'insert_rows',
        sub {
            my ($app, %opts) = @_;
            return \%opts;
        }
    );
}

sub mock_http {
    use SOAP::Lite;
    use XMLRPC::Lite;
    use MIME::Lite;
    use LWP::UserAgent;

    mock_subs(
        [
            qw(
              HTTP::Tiny::request
              LWP::UserAgent::get
              MIME::Lite::send_by_sendmail
              MIME::Lite::send_by_smtp
              SOAP::Lite::call
              XMLRPC::Lite::call
              )
        ]
    );
    mock_lwp_sub();
    return 1;
}

sub mock_mds_avatars {
    my ($app) = @_;

    my $MDS_AVATARS_IMG_GROUP_ID = 666111;

    $app->api_tvm;
    $app->{'api_tvm'} = Test::MockObject::Extends->new($app->{'api_tvm'});
    $app->api_tvm->mock(
        'get_service_ticket',
        sub {
            my ($tvm, $tvm_dst_alias) = @_;
            return 'mocked-service-ticket-' . $tvm_dst_alias // '';
        }
    );

    $app->api_media_storage_avatars;
    $app->{'api_media_storage_avatars'} = Test::MockObject::Extends->new($app->{'api_media_storage_avatars'});

    $app->api_media_storage_avatars->mock(
        'upload_file',
        sub {
            my ($self, $file_name) = @_;
            return _mocked_mds_avatars_response($MDS_AVATARS_IMG_GROUP_ID, $file_name);
        }
    );

    $app->api_media_storage_avatars->mock(
        'upload_link',
        sub {
            my ($self, $file_name) = @_;
            return _mocked_mds_avatars_response($MDS_AVATARS_IMG_GROUP_ID, $file_name);
        }
    );
}

sub _mocked_mds_avatars_response {
    my ($group_id, $file_name) = @_;
    return {
        'extra'     => {'svg' => {'path' => "/get-partner/65861/$file_name/svg"}},
        'group-id'  => $group_id,
        'imagename' => $file_name,
        'meta'      => {
            'crc64'             => '8E26F2EFC7269C63',
            'md5'               => 'e38174c4a5118d7d4c64f4ab01d07ad6',
            'modification-time' => 1549985590,
            'orig-animated'     => FALSE,
            'orig-format'       => 'JPEG',
            'orig-orientation'  => '',
            'orig-size'         => {
                'x' => 1178,
                'y' => 640
            },
            'orig-size-bytes'                          => 349141,
            'processed_by_computer_vision'             => FALSE,
            'processed_by_computer_vision_description' => 'computer vision is disabled',
            'processing'                               => 'finished'
        },
        'sizes' => {
            'big' => {
                'height' => 435,
                'path'   => "/get-partner/$group_id/$file_name/big",
                'width'  => 800
            },
            'card' => {
                'height' => 150,
                'path'   => "/get-partner/$group_id/$file_name/card",
                'width'  => 210
            },
            'card_retina' => {
                'height' => 300,
                'path'   => "/get-partner/$group_id/$file_name/card_retina",
                'width'  => 420
            },
            'list' => {
                'height' => 50,
                'path'   => "/get-partner/$group_id/$file_name/list",
                'width'  => 70
            },
            'list_retina' => {
                'height' => 100,
                'path'   => "/get-partner/$group_id/$file_name/list_retina",
                'width'  => 140
            },
            'orig' => {
                'height' => 640,
                'path'   => "/get-partner/$group_id/$file_name/orig",
                'width'  => 1178
            }
        }
    };
}

sub mock_statistics_is_available {
    my ($app) = @_;

    # must init any statistics accessor before mocking ::is_available
    # or mock would be redefined on init
    $app->statistics_dsp;
    mock_subs({'Application::Model::Statistics::Product::is_available' => sub {TRUE},});
    $app->bk_statistics;
    mock_subs({'Application::Model::BKStatistics::is_available' => sub {TRUE},});
}

our $original_subs = {};

sub mock_subs {
    my ($subs, %opts) = @_;

    $subs = {map {$_ => undef} @$subs} if ref($subs) eq 'ARRAY';

    no warnings 'redefine';
    no strict 'refs';

    foreach my $sub_name (keys %$subs) {
        my $already_mocked = exists($original_subs->{$sub_name});

        unless ($already_mocked) {
            $original_subs->{$sub_name} = \&{$sub_name};
        }

        if (!$already_mocked || $opts{'force'}) {
            *{$sub_name} = $subs->{$sub_name} // sub {
                throw Exception "Original $sub_name is called! Mock it for the test.\n";
            };
        }
    }
}

sub restore_subs {
    my ($sub_full_names) = @_;

    no warnings 'redefine';
    no strict 'refs';

    foreach my $sub_name (@$sub_full_names) {
        if ($original_subs->{$sub_name}) {
            *{$sub_name} = delete $original_subs->{$sub_name};
        }
    }
}

sub mock_lwp_sub {
    my (@uri_prefixes) = grep {length} map {$ENV{$_}} qw/JAVA_MOCKED_JSONAPI_URL DEPLOY_TVM_TOOL_URL/;
    my $sub_name = 'LWP::UserAgent::request';
    mock_subs(
        {
            $sub_name => sub {
                my ($ua, $req) = @_;
                my $uri = $req->uri;
                foreach my $pref (@uri_prefixes) {
                    if (substr($req->uri, 0, length($pref)) eq $pref) {
                        my $res = $original_subs->{$sub_name}->(@_);
                        return $res;
                    }
                }
                throw Exception "Original $sub_name " . ($req->uri) . " is called! Mock it for the test.\n";
              }
        }
    );
}

my %memcached_value;

sub mock_memcached {
    my ($app) = @_;

    $app->memcached;
    $app->{'memcached'} = Test::MockObject::Extends->new($app->{'memcached'});

    $app->memcached->mock(
        'incr',
        sub {
            my ($self, $prefix, $key, $value) = @_;
            $memcached_value{$key} //= 0;
            return ++$memcached_value{$key};
        }
    );
}

sub mock_for_config {
    my ($app, $option_config) = @_;

    my $orig_pre_run = \&QBit::Application::pre_run;
    mock_subs(
        {
            'QBit::Application::pre_run' => sub {
                $orig_pre_run->(@_);

                my $app_config = $app->get_option("api_configs");
                _apply_config_recursively($app_config, $option_config->{"api_configs"});
                $app->set_option("api_configs", $app_config);

              }
        }
    );
}

sub _apply_config_recursively {
    my ($dst, $src) = @_;

    for (keys %$src) {
        if (exists $dst->{$_} && length ref $dst->{$_} > 0) {
            #throw "type of key '$_' not matched in configs" if ref($src->{$_}) ne ref($dst->{$_});
            _apply_config_recursively($dst->{$_}, $src->{$_});
        } else {
            $dst->{$_} = $src->{$_};
        }
    }
}

sub mock_send_to_graphite {
    my @subs_send_to_graphite = grep {defined(&{"$_"})} (map {s'\.pm''; s'/'::'g; "${_}::send_to_graphite"} keys(%INC)),
      'main';

    mock_subs(
        {
            map {
                $_ => sub { }
              } @subs_send_to_graphite
        }
    );
}

sub mock_push_client_add_objects {
    mock_subs({'Utils::PushClient::add_objects' => sub { },});
}

sub mock_selfservice {
    mock_subs({'Application::Model::API::Yandex::SelfService::call' => sub { },});
}

sub mock_moneymap {
    my ($app) = @_;
    $app->api_moneymap;
    $app->{'api_moneymap'} = Test::MockObject::Extends->new($app->{'api_moneymap'});
    $app->{'api_moneymap'}->mock(
        get_block => sub {
            {
                "page_id"          => $_[1],
                "block_id"         => $_[2],
                "abc_id"           => 123,
                "abc_oebs_id"      => 1234,
                "os"               => "",
                "page_type"        => "internal",
                "platform"         => "desktop",
                "sub_traffic_type" => "native",
                "traffic_type"     => "web"
            };
        }
    );
    $app->{'api_moneymap'}->mock(
        get_page => sub {
            {
                "page_id"          => $_[1],
                "abc_id"           => 123,
                "abc_oebs_id"      => 1234,
                "os"               => "",
                "page_type"        => "internal",
                "platform"         => "desktop",
                "sub_traffic_type" => "native",
                "traffic_type"     => "web"
            };
        },
    );
    $app->{'api_moneymap'}->mock(
        blocks => sub {
            my ($self, %opts) = @_;

            my $response = {"blocks" => []};

            my $block_ids = $opts{'block[eq]'} // from_json($opts{':content'})->{'filter'}->{'block[eq]'} // [];
            foreach my $id (@$block_ids) {
                my ($page_id, $block_id) = ($id =~ /(\d+)[-:](\d+)$/);
                push(
                    @{$response->{'blocks'}},
                    {
                        "page_id"          => $page_id,
                        "block_id"         => $block_id,
                        "abc_id"           => 123,
                        "abc_oebs_id"      => 1234,
                        "os"               => "",
                        "page_type"        => "internal",
                        "platform"         => "desktop",
                        "sub_traffic_type" => "native",
                        "traffic_type"     => "web"
                    }
                );
            }

            return $response;
        }
    );
    $app->{'api_moneymap'}->mock(
        pages => sub {
            my ($self, %opts) = @_;

            my $response = {"pages" => []};

            my $page_ids = $opts{'page_id[eq]'} // from_json($opts{':content'})->{'filter'}->{'page_id[eq]'} // [];
            foreach my $page_id (@$page_ids) {
                push(
                    @{$response->{'pages'}},
                    {
                        "page_id"          => $page_id,
                        "abc_id"           => 123,
                        "abc_oebs_id"      => 1234,
                        "os"               => "",
                        "page_type"        => "internal",
                        "platform"         => "desktop",
                        "sub_traffic_type" => "native",
                        "traffic_type"     => "web"
                    }
                );
            }

            return $response;
        }
    );
    $app->{'api_moneymap'}->mock('add_page',    sub {{}});
    $app->{'api_moneymap'}->mock('add_block',   sub {{}});
    $app->{'api_moneymap'}->mock('edit_page',   sub {{}});
    $app->{'api_moneymap'}->mock('edit_block',  sub {{}});
    $app->{'api_moneymap'}->mock('delete_page', sub {{}});
    $app->{'api_moneymap'}->mock(
        'get_fields',
        sub {
            my ($self, $type) = @_;
            my ($res) = grep {$_->{traffic_type} eq $type} (
                {
                    "traffic_type"     => "web",
                    "traffic_sub_type" => ["context", "content", "native", "search", "adblock"],
                    "platform" => ["mobile", "desktop"],
                    "os"       => [],
                },
                {
                    "traffic_type"     => "sdk",
                    "traffic_sub_type" => [
                        "instream", "interstitial", "rewarded", "banner", "adaptive_banner", "native", "search", "other"
                    ],
                    "platform" => ["mobile"],
                    "os"       => ["ios", "android"],
                },
                {
                    "traffic_type"     => "videoresources",
                    "traffic_sub_type" => ["instream", "outstream", "inpage", "interstitial", "other"],
                    "platform" => ["mobile", "desktop", ""],
                    "os"       => [],
                },
                {
                    "traffic_type"     => "search",
                    "traffic_sub_type" => ["premium", "other"],
                    "platform"         => ["mobile", "desktop"],
                    "os"               => [],
                },
                {
                    "traffic_type"     => "ssp",
                    "traffic_sub_type" => ["web", "app"],
                    "platform"         => ["mobile", "desktop"],
                    "os"               => [],
                }

            );
            return $res;
        },
    );
    $app->{'api_moneymap'}->mock(
        'get_internal_owners',
        sub {
            return [
                qw(
                  adinside
                  mocked-yan-partner
                  mocked-video-partner
                  )
            ];
        }
    );
}

sub mock_moneymap_error {
    my ($app) = @_;
    $app->api_moneymap;
    $app->{'api_moneymap'} = Test::MockObject::Extends->new($app->{'api_moneymap'});
    $app->{'api_moneymap'}->mock('add_page',            sub {throw Exception::MoneyMap 'test moneymap error'});
    $app->{'api_moneymap'}->mock('add_block',           sub {throw Exception::MoneyMap 'test moneymap error'});
    $app->{'api_moneymap'}->mock('edit_page',           sub {throw Exception::MoneyMap 'test moneymap error'});
    $app->{'api_moneymap'}->mock('edit_block',          sub {throw Exception::MoneyMap 'test moneymap error'});
    $app->{'api_moneymap'}->mock('get_internal_owners', sub {[qw(adinside mocked-yan-partner)]});
    $app->{'api_moneymap'}->mock('delete_page',         sub {{}});
    $app->{'api_moneymap'}->mock('get_blocks_by_ids',   sub {throw Exception::MoneyMap 'test moneymap error'});
}

sub mock_sort_sql {
    use QBit::Application::Model::DB::Query;
    no strict 'refs';
    no warnings 'redefine';
    my $orig_sub_sql = \&QBit::Application::Model::DB::Query::get_sql_with_data;
    *{'QBit::Application::Model::DB::Query::get_sql_with_data'} = sub {
        return $orig_sub_sql->(@_, canonical => 1);
    };
    my $orig_sub_field = \&QBit::Application::Model::DB::Query::_field_to_sql;
    *{'QBit::Application::Model::DB::Query::_field_to_sql'} = sub {
        return $orig_sub_field->(@_, canonical => 1);
    };
}

sub mock_api_rmp {
    my ($app) = @_;

    $app->api_rmp;
    $app->{'api_rmp'} = Test::MockObject::Extends->new($app->{'api_rmp'});

    $app->api_rmp->mock(
        'get_app_info',
        sub {
            (undef, my %opts) = @_;
            my ($store_id_from_url) = $opts{store_url} =~ /id=([a-zA-Z\.]+)/;
            return {
                'type' => $opts{type} // 1,
                'store_id' => $opts{store_id} // $store_id_from_url // 'ru.mocked.app',
                'icon'           => 'mocked icon',
                'developer'      => 'mocked developer',
                'apple_store_id' => $opts{type} eq '2' ? 'id111000' : undef,
                'store_url'      => $opts{store_url},
            };
        }
    );
}

sub mock_api_java_bk_data {
    my ($app) = @_;

    $app->api_java_bk_data;

    mock_subs(
        {
            'Application::Model::API::Yandex::JavaBKData::enrich_page' => sub {
                (undef, my %opts) = @_;

                $opts{page_bk_data} //= {};
                foreach my $block_id (keys %{$opts{page_bk_data}->{rtb_blocks}}, "1") {
                    $opts{page_bk_data}->{RtbBlocks}{$block_id} =
                      exists $opts{page_bk_data}->{rtb_blocks}{$block_id}
                      ? $opts{page_bk_data}->{rtb_blocks}{$block_id}
                      : {
                        "Model"                          => "context_on_site_rtb",
                        "MockedJavaGeneratedBlockBKData" => 1,
                      };
                }
                return $opts{page_bk_data};
              }
        }
    );
}

sub mock_edit_cooperation_form {
    my ($app) = @_;

    $app->users->{__MODEL_FIELDS__}->{cooperation_form}->{api_can_edit} = TRUE;
    $app->users->{__LAST_FIELDS__}->{cooperation_form}->{api_can_edit}  = TRUE;
}

sub mock_java_jsonapi {
    my ($app, $mode) = @_;

    my $mock_url = $ENV{JAVA_MOCKED_JSONAPI_URL} // die 'ENV JAVA_MOCKED_JSONAPI_URL not set';
    if ($mode eq 'RestApi') {
        my $sub_name = 'Application::pre_run';
        mock_subs(
            {
                $sub_name => sub {
                    $original_subs->{$sub_name}->(@_);
                    $app->api_java_jsonapi->set_option(url => $mock_url);
                    $app->api_java_bk_data->set_option(url => "${mock_url}api/bkdata/");
                  }
            }
        );
    } else {
        $app->api_java_jsonapi->set_option(url => $mock_url);
        $app->api_java_bk_data->set_option(url => "${mock_url}api/bkdata/");
    }
}

sub mock_text_template {
    my ($app) = @_;

    $app->text_template;
    $app->inviter;
    mock_subs(
        {
            'Application::Model::TextTemplate::get_content' => sub {
                return 'Test invitation content with [% invite_link %]';
            },
            'Application::Model::TextTemplate::process_content' => sub {
                return 'Test processed content';
            },
            'Application::Model::Inviter::_generate_id' => sub {
                return '1d92030d9b1a34fb2804b2f2eda83627';
            },
        }
    );
}

sub mock_host {
    my ($host) = $_[-1];

    no warnings 'redefine';
    no strict 'refs';
    *{'Mojo::Message::Request::env'} = sub {
        return {
            HTTP_X_REAL_IP     => '127.0.0.1',
            HTTP_HOST          => $host // 'test.com',
            HTTP_X_REAL_SCHEME => 'https',
            HTTP_X_REAL_PORT   => '80',
        };
    };
}

my $FEATURES = {};

sub mock_features {
    my ($app, $features) = @_;

    my %users = map {$_->{'login'} => $_->{'id'}} @{
        $app->partner_db->users->get_all(
            fields => [qw(id login)],
            filter => [login => 'IN' => \[keys(%$features)]],
        )
      };

    unless (%$FEATURES) {
        map {$FEATURES->{$_->{user_id}}{$_->{feature}} = 1} @{
            $app->partner_db->user_features->get_all(
                fields => [qw(user_id feature)],
                filter => [user_id => 'IN' => \[values(%users)]]
            )
          };
    }

    $app->set_cur_user({id => 0});
    while (my ($login, $list) = each %$features) {
        for my $feature (@$list) {
            unless ($FEATURES->{$users{$login}}{$feature}++) {
                $app->user_features->add($users{$login}, $feature);
            }
        }
    }
}

sub reset_features {
    my ($app, $features) = @_;

    my %users = map {$_->{'login'} => $_->{'id'}} @{
        $app->partner_db->users->get_all(
            fields => [qw(id login)],
            filter => [login => 'IN' => \[keys(%$features)]],
        )
      };

    while (my ($login, $list) = each %$features) {
        my @need_delete = ();
        for my $feature (@$list) {
            unless (--$FEATURES->{$users{$login}}{$feature}) {
                push(@need_delete, $feature);
            }
        }

        $app->user_features->delete($users{$login}, @need_delete) if @need_delete;
    }
}

sub mock_page_validator {
    require QBit::Validator::Type::bk_data::page;
    @QBit::Validator::Type::bk_data::page::NOT_INTERNAL_PAGES = (1 .. 1000);
}

1;
