package Test::Partner2::RosettaServer;

use strict;
use warnings FATAL => 'all';

use qbit;

use Test::More;
use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_blackbox);

use RosettaProtocol qw(
  write_request
  read_response
  interact
  );

use IO::Handle;
use IO::Socket::IP;
use Encode;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  authorization_request
  run_rosetta_tests
  );
our @EXPORT = @EXPORT_OK;

sub authorization_request {
    {
        "rosetta_type"     => "authorization",
        "request_id"       => "1490711915159-23-578",
        "nginx_request_id" => "485e7c8f29843578661da03320888e39",
        "session_id"       => "$_[0]",
        "session_id2"      => "mocked-sessionid2",
        "server_name"      => "partner-creator01f.yandex.ru",
        "user_ip"          => "::ffff:172.17.0.59",
        "headers"          => {
            "host"          => "partner-creator01f.yandex.ru",
            "x-real-port"   => "443",
            "x-real-scheme" => "https",
            "x-request-id"  => "485e7c8f29843578661da03320888e39",
            "connection"    => "close",
            "accept"        => "text/html,application/xhtml+xml,application/xml;q=0.9,*/*;q=0.8",
            "cookie"        => "mocked-cookie",
            "user-agent" =>
"Mozilla/5.0 (Macintosh; Intel Mac OS X 10_12_3) AppleWebKit/602.4.8 (KHTML, like Gecko) Version/10.0.3 Safari/602.4.8",
            "accept-language" => "ru",
            "referer"         => "https://partner-creator01f.yandex.ru:8510/v2/dashboard/",
            "accept-encoding" => "gzip, deflate"
        }
    };
}

sub response_ok {
    my ($response, $expected) = @_;

    is_deeply([sort keys %$response], [sort keys %$expected], 'response keys');

    for my $key (keys %$expected) {
        is((ref($response->{$key}) || $response->{$key}), $expected->{$key}, "response->{$key}");
    }
}

sub interact_ok {
    my ($rosetta_port, $requests, $expected_responses, $exception, $test_name) = @_;

    my $client = IO::Socket::IP->new(
        Proto    => 'tcp',
        PeerAddr => "localhost:$rosetta_port",
        Type     => SOCK_STREAM,
        Family   => AF_INET6,
    ) or die "Failed to open client socket: $@";

    my @got_responses;

    subtest $test_name => sub {
        plan tests => scalar(@$expected_responses) + (0 + defined $exception);

        try {
            while (my $request = shift @$requests) {
                my $type = delete $request->{rosetta_type};
                write_request($client, $type, $request);
                my $response = read_response($client) // last;
                push @got_responses, $response;
            }
        }
        catch {
            my ($caught) = @_;
            is_deeply($caught->message, $exception, 'exception');
        };

        for my $expected (@$expected_responses) {
            my $got = shift @got_responses;
            subtest $expected->{result} => sub {
                plan tests => scalar(keys %$expected) + 1;
                response_ok($got, $expected);
              }
        }
    };

    close($client) or die "Failed to close client socket: $@";
}

sub run_rosetta_tests {
    my ($app, $tests) = @_;

    my $rosetta_port;
    my $rosetta_pid;

    mock_blackbox($app);

    my $server = IO::Socket::IP->new(
        Proto     => 'tcp',
        LocalAddr => 'localhost:0',
        Type      => SOCK_STREAM,
        Family    => AF_INET6,
        ReuseAddr => 1,
        Listen    => 10,
        Timeout   => 5,
    ) or die "Failed to open listening socket: $@";
    $rosetta_port = $server->sockport;

    $rosetta_pid = fork() // die "Failed to fork: $@";
    if ($rosetta_pid) {
        close($server) or die "Failed to close listening socket in client process: $@";
    } else {
        my $expecting_connections = keys %$tests;
        $SIG{INT} = sub {
            warn 'Server process got SIGINT';
        };
        try {
            while ($expecting_connections > 0) {
                my $connection = $server->accept() or throw Exception "Failed to accept connection: $@";
                interact($connection, $app);
                close($connection) or throw Exception "Failed to close accepted connection: $@";
                $expecting_connections -= 1;
            }
        }
        catch {
            warn $_[0]->message;
        };
        exit;
    }

    for my $name (sort keys %$tests) {
        interact_ok(
            $rosetta_port,
            $tests->{$name}{requests},
            $tests->{$name}{responses},
            $tests->{$name}{exception}, $name,
        );
    }

    waitpid($rosetta_pid, 0);
}

1;
