package Test::Partner2::Simple;

=encoding UTF-8

=head1 DESCRIPTION

Пример теста с использованием этой библиотеки:
    use lib::abs qw(
        ../lib
        ../t_lib
    );
    use Test::Partner2::Simple;

    run_tests(
        sub {
            my ($app) = @_;
            ...
        },
        ########## опционально
        application_package   => 'Rosetta',   # Дефолтное значение – Application
        user                  => 'm-ru-text', # По умолчанию будет использовать пользователь у которого есть все права
        mock_http             => 1,           # По дефолту, тест будет падать при попытке сделать HTTP запросы (LWP, XMLRPC, SOAP, etc)
        mock_time             => 1,           # По дефолту время замораживается
        dont_create_database  => 1,           # Не создает базу данных
        fill_databases        => 0,           # Не наливает базу данных
        keep_databases        => 1,           # Не удаляет тестовую базу данных после прогона теста
        reuse_database        => 1,           # Переиспользует ранее созданную базу (имеет смысл только для READONLY тестов)
        reuse_db_suffix       => '',          # В случае Jenkins это $ENV{EXECUTOR_NUMBER}, для TeamCity это $ENV{BUILD_NUMBER}
        do_not_die_on_fail    => 1,           # Прогонять все тесты до конца
        db_suites             => {            # Наливает не все таблицы, а только из определенного набора
            partner_db => [qw( users ) ],
        },
        locale                => 'C',         # локаль: "en" | "ru" | "C",
        init                  => [qw(api_bk)] # создает объект перед прогоном тестов
        mocks                 => [qw(mock_check_statistics_by_blocks)]
    );

Код запускается со всеми правами.

Библиотека в конце сама делает done_testing().

=cut

use Test::Partner2::Mock;    #keep this first to run mock_time properly

use qbit qw();
use QBit::Exceptions;
use Carp;
use Application;
use Rosetta;
use API;
use IntAPI;
use Cron;
use Test::TestBuilder;
use Test::Partner2::AllRights;
use Test::Partner2::AnyKeys;
use Test::Partner2::Database;
use Test::Partner2::Fixture;
use Test::Partner2::Utils qw(set_locale_in_tests);
use SOAP::Lite;
use XMLRPC::Lite;
use MIME::Lite;
use LWP::UserAgent;
use Test::Mojo;
use Mojo::Util qw(monkey_patch);
use Mojo::URL;
use RestApi;
use HTTP::Tiny;
use Data::Dumper;

use PiConstants qw($AUTHORIZATION_USER_FIELDS);

use Exception;
use Exception::DB;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  change_cur_user
  need_self_update
  run_tests
  );
our @EXPORT = @EXPORT_OK;

our $DEFAULT_LOGIN = 'yndx-bessarabov';

BEGIN {
    Test::TestBuilder::init_builder();
    require Test::Most;

    no warnings 'redefine';
    Test::Most->import();

    my $orig_eq_or_diff = \&Test::Differences::eq_or_diff;
    my %eq_or_diff_params = (context => 3);
    *{'Test::Differences::eq_or_diff'} = sub {
        local $Test::Builder::Level = $Test::Builder::Level + 1;
        $orig_eq_or_diff->(
            $_[0], $_[1],
            $_[2] // '',
            $_[3] ? {exists $_[3]->{context} ? %{$_[3]} : (%{$_[3]}, %eq_or_diff_params)} : \%eq_or_diff_params
        );
    };
}

sub run_tests {
    my ($sub, %opts) = @_;

    $opts{locale} //= 'C';

    Test::Most::die_on_fail() unless ($opts{'do_not_die_on_fail'} || $ENV{'do_not_die_on_fail'});

    pass('Loaded ok');

    my $application_package = $opts{'application_package'} // 'Application';

    mock_time() if $opts{'mock_time'} // 1;

    $ENV{mock_http} = $opts{'mock_http'} // 1;
    mock_http() if $ENV{mock_http};

    if ($ENV{'LAZY_LOAD'}) {
        no warnings 'redefine';
        no strict 'refs';
        *{'Application::add_all_tmp_rights'} = sub {
            my %all_rights;
            tie %all_rights, 'Test::Partner2::AnyKeys';

            my $all_rights_obj =
              Test::Partner2::AllRights->new(app => $_[0], rights => $_[0]->{'__CURRENT_USER_RIGHTS__'});

            $_[0]->{'__CURRENT_USER_RIGHTS__'} = \%all_rights;

            return $all_rights_obj;
        };
    }

    my $app;
    my $app_with_db;

    my $post_run = \&QBit::Application::post_run;

    if ($application_package eq 'RestApi') {
        $app = Test::Mojo->new($application_package);

        my $old_listen = \&{*Mojo::IOLoop::Server::listen};
        monkey_patch 'Mojo::IOLoop::Server', listen => sub {
            my $self    = shift;
            my $args    = ref($_[0]) eq 'HASH' ? $_[0] : {@_};
            my $retries = 1;
            if ($args->{address} eq '127.0.0.1') {
                $args->{address} = 'localhost';
                $retries = 3;
            }
            my $result;
            while ($retries--) {
                try {
                    $result = $old_listen->($self, $args);
                    $retries = 0;
                }
                catch {
                    my ($e) = @_;
                    throw($e) unless $retries;
                    warn "about to retry Mojo::IOLoop::Server::listen\n";
                    sleep(1);
                };
            }
            return $result;
        };
        my $old_connect = \&{*Mojo::IOLoop::Client::connect};
        monkey_patch 'Mojo::IOLoop::Client', connect => sub {
            my $self    = shift;
            my $args    = ref($_[0]) eq 'HASH' ? $_[0] : {@_};
            my $retries = 1;

            for my $key (qw(socks_address address)) {
                if (exists($args->{$key}) && ($args->{$key} // '') eq '127.0.0.1') {
                    $args->{$key} = 'localhost';
                    $retries = 3;
                }
            }
            if ($retries > 1) {
                my @old_subscribers;
                @old_subscribers = @{$self->subscribers('error')};
                @{$self->subscribers('error')} = ();
                my $catch;
                $catch = sub {
                    $retries--;
                    if ($retries <= 1) {
                        @{$self->subscribers('error')} = @old_subscribers;
                    }
                    warn "about to retry Mojo::IOLoop::Client::connect\n";
                    Mojo::IOLoop->timer(1 => sub {$old_connect->($self, $args);});
                };
                $self->on(error => $catch);
            }
            return $old_connect->($self, $args);
        };
        $app_with_db = $app->app->models;
    } else {
        $app = $application_package->new();
        $app->pre_run();
        $app_with_db = $app;

        no warnings 'redefine';
        no strict 'refs';

        *{'QBit::Application::pre_run'}  = sub { };
        *{'QBit::Application::post_run'} = sub { };
    }

    if (my $map = ($opts{mock_yt_picategory_mapping} // 1)) {
        mock_yt_picategory_mapping($app_with_db, $map);
    }

    if ($opts{'create_clickhouse_db'}) {
        $app_with_db->clickhouse_db;

        my $sub = \&{'QBit::Application::Model::DB::clickhouse::st::execute'};

        mock_subs(
            {
                'QBit::Application::Model::DB::clickhouse::st::execute' => sub {
                    restore_subs([qw(LWP::UserAgent::request)]);
                    my $result = $sub->(@_);
                    mock_subs([qw(LWP::UserAgent::request)]);
                    return $result;
                },
            }
        );
    }

    mock_get_lock() if $application_package eq 'Cron';

    my $tmp_rights;

    if ($opts{'locale'} && $app->isa('Application')) {
        set_locale_in_tests($app, $opts{'locale'});
    }

    my $default_mock_options = get_default_mock_options();
    if ($opts{'mocks'} && ref($opts{'mocks'}) eq 'ARRAY') {
        $opts{'mocks'} = {map {$_ => 1} @{$opts{'mocks'}}};
    }
    apply_mocks($app_with_db, [$default_mock_options, $opts{'mocks'}]);

    unless ($opts{'dont_create_database'}) {

        subtest 'Create databases ' => sub {
            note("Start create mocked database");
            my $timestamp = time;
            create_mocked_databases(
                app => $app_with_db,
                (
                    map {$_ => $opts{$_}}
                      qw(
                      keep_databases
                      reuse_database
                      fill_databases
                      db_suites
                      use_cached_sql
                      reconnect
                      mocked_databases
                      create_clickhouse_db
                      )
                ),
                (reuse_db_suffix => $opts{'reuse_db_suffix'} // $ENV{'EXECUTOR_NUMBER'} // $ENV{'BUILD_NUMBER'}
                  )    # NOTE! эти переменные прилетают из Jenkins и TeamCity
            );
            pass('Mocked database created (' . (time - $timestamp) . 'sec)');
        };

        if ($opts{fixtures} or $opts{extra_fixtures}) {
            Test::Partner2::Fixture::set_app($app_with_db);

            if ($opts{extra_fixtures}) {
                push @{$opts{fixtures}}, Test::Partner2::Fixture::add_fixtures($opts{extra_fixtures});
            }

            for my $fixture ('user_yndx_bessarabov', @{$opts{fixtures}}) {
                Test::Partner2::Fixture::load_fixture($fixture);
            }
            my @models = keys(
                %{
                    {
                        map {$_ => 1}
                          (map {@{Test::Partner2::Fixture::validate_models_for_fixture($_)}} @{$opts{fixtures}})
                    }
                  }
            );
            my $all_models_are_ok = 1;
            for my $model (@models) {
                if ($app_with_db->$model->can('get_template')) {
                    no warnings 'redefine';
                    local *QBit::Application::check_rights = sub {1};
                    my ($is_ok, $details) =
                      $app_with_db->qbit_validator_checker->check_all_elements_in_model(accessor => $model);
                    unless ($is_ok) {
                        $all_models_are_ok = 0;
                        STDERR->print($model);
                        STDERR->print(Dumper($details));
                    }
                }
            }
            throw Exception 'Applied fixtures led to invalid db state'
              unless $all_models_are_ok;
        }

        apply_mocks($app_with_db, [\%opts]);

        if ($opts{'user'}) {
            throw Exception 'You can\'t use user with application_package IntAPI' if $application_package eq 'IntAPI';
            change_cur_user($app_with_db, $opts{'user'});
        } elsif ($application_package eq 'IntAPI' and !$opts{skip_init_user}) {
            change_cur_user($app_with_db, 'yndx-partner-intapi');
        } else {
            my $yndx_bessarabov = $app_with_db->partner_db->users->get_all(
                filter => [login => '=' => \$DEFAULT_LOGIN],
                fields => [qw(login name client_id id multistate business_unit)],
            )->[0];
            $yndx_bessarabov //= {
                'login'         => $DEFAULT_LOGIN,
                'name'          => '',
                'client_id'     => '2901607',
                'id'            => '155209804',
                'multistate'    => '1',
                'business_unit' => '0',
            };
            $app_with_db->set_cur_user($yndx_bessarabov);

            my %all_rights;
            tie %all_rights, 'Test::Partner2::AnyKeys';

            $app_with_db->{'__CURRENT_USER_RIGHTS__'} = \%all_rights;
        }
    } else {
        no strict 'refs';
        no warnings 'redefine';

        $app_with_db->partner_db;

        *{'QBit::Application::Model::DB::mysql::_connect'} = sub {
            my ($self) = @_;
            my $dbh = DBI->connect('DBI:Mock:', '', '');
            $self->{'__DBH__'}{$self->get_key} = $dbh;
            return 1;
        };
    }

    foreach (@{$opts{'init'} // []}) {
        $app_with_db->$_;
    }

    mock_subs({'Application::Model::PartnerDB::all_pages' => sub {$app_with_db->partner_db->all_pages_view},});
    mock_java_jsonapi($app_with_db, $application_package) unless $ENV{NO_JAVA_MOCK};
    eval {$sub->($app);};
    if ($@) {
        my $e = $@;

        # Catch DB deadlocks (#INFRASTRUCTUREPI-1654)
        if (($e->message // '') =~ /Deadlock found/) {
            my $ref           = $app->partner_db->_get_all('SHOW ENGINE INNODB STATUS');
            my $innodb_status = $ref->[0]->{'Status'};
            $e->{'text'} .= "\n\n##################### ENGINE INNODB STATUS #####################\n" . $innodb_status;
        }

        die $e;
    }

    if ($application_package ne 'RestApi') {
        $post_run->($app);
    }

    done_testing();
}

sub change_cur_user {
    my ($app, $login) = @_;

    my $cur_user = {
        id            => 0,
        client_id     => 0,
        login         => 'foobar',
        name          => 'foo',
        lastname      => 'bar',
        email         => 'devnull@foo.bar',
        business_unit => 0,
        multistate    => 1,
    };

    if ($login) {

        my $tmp   = $app->add_all_tmp_rights;
        my $users = $app->users->get_all(
            fields => $AUTHORIZATION_USER_FIELDS,
            filter => {login => $login}
        );

        croak sprintf("User with login '%s' not found", $login) unless @$users == 1;

        $cur_user = $users->[0];

        # Нужно для get_avalilable_block_model_names.t (#PI-7688)
        {
            my $user_adfox = [];
            eval {
                $user_adfox = $app->partner_db->user_adfox->get_all(
                    fields => ['adfox_id'],
                    filter => $app->partner_db->filter({user_id => $cur_user->{id}}),
                    limit  => 1
                );
            };
            $cur_user->{has_adfox} = (@$user_adfox && $user_adfox->[0]->{adfox_id}) ? 1 : 0;
        }

    }

    $app->{'__CURRENT_USER_RIGHTS__'} = {};

    $app->set_cur_user($cur_user);

    return $cur_user;
}

sub need_self_update {
    return $ENV{SELF_UPDATE} || grep {$_ eq '--self_update'} @ARGV;
}

1;
