package Test::Partner2::Simple::API;

use Exporter qw(import);
our @EXPORT = qw(
  test
  );
our @EXPORT_OK = @EXPORT;

use Test::Partner2::Mock;    #keep this first to run mock_time properly

use Test::Partner2::Simple;
use Test::MockObject::Extends;
use Data::Dumper;
use Test::Partner2::Database qw(save_databases);
use Test::Partner2::Fixture;
use Test::Partner::Utils qw(use_perl_do_actions use_perl_get_all);

use Test::More;
use Test::Deep;
use Test::Differences;

use qbit;

my $MOCKED_DATABASES_RESTAPI = 'mocked_databases_restapi';
my $BUILD_NUMBER = $ENV{'EXECUTOR_NUMBER'} // $ENV{'BUILD_NUMBER'}; # EXECUTOR_NUMBER - Jenkins; BUILD_NUMBER - TeamCity
my $READONLY_DATABASES_PREFIX = 'READONLY_API' . (defined($BUILD_NUMBER) ? "_$BUILD_NUMBER" : '');
my $database_created          = FALSE;

my $sub_dcnpgettext = \&Locale::gettext_pp::dcnpgettext;

sub set_fields_by_accessor {
    my ($fields, $accessor) = @_;

    return "fields[$accessor]=" . join(',', @{$fields->{$accessor} // []});
}

sub set_all_fields_by_accessor {
    my ($models, $accessor) = @_;

    my $model_fields = $models->$accessor->get_model_fields();

    return "fields[$accessor]="
      . join(',',
        sort grep {$model_fields->{$_}{'api'} && ($_ ne 'login' || $accessor eq 'users')} keys(%$model_fields));
}

sub test {
    my ($file_path_or_structure) = @_;

    my ($structure, $header);
    if (ref($file_path_or_structure)) {
        $structure = $file_path_or_structure;
    } else {
        my $content = readfile($file_path_or_structure);

        $content =~ s/^(.+?\n)//;
        $header = $1;

        $structure = from_json($content);
    }

    my $option_skip = $structure->{'options'}{'skip'};
    plan skip_all => $option_skip if $option_skip;

    my $default_mock_options = get_default_mock_options('tjson');

    my $option_readonly             = $structure->{'options'}{'read_only'}            // TRUE;
    my $option_dont_create_db       = $structure->{'options'}{'dont_create_database'} // FALSE;
    my $option_create_clickhouse_db = $structure->{'options'}{'create_clickhouse_db'} // FALSE;
    my $preload_accessors           = $structure->{'options'}{'preload_accessors'}    // FALSE;
    my $option_lang                 = $structure->{'options'}{'lang'}                 // $ENV{'MOCK_LANG_DETECT'};

    my $fields          = $structure->{'options'}{'fields'}   // {};
    my $option_fixtures = $structure->{'options'}{'fixtures'} // undef;
    my $option_init     = $structure->{'options'}{'init'}     // [];

    my $option_config       = $structure->{'options'}{'config'};
    my $use_perl_do_actions = $structure->{'options'}{'use_perl_do_actions'};
    my $use_perl_get_all    = $structure->{'options'}{'use_perl_get_all'};

    my $tests = $structure->{'tests'};
    $tests = [$tests] unless ref($tests) eq 'ARRAY';

    my $test_name_filter = $ENV{'TEST_NAME'};

    $ENV{'LAZY_LOAD'} = $preload_accessors ? 0 : 1;

    run_tests(
        sub {
            my ($app) = @_;

            my $models = $app->app->models;

            unless ($ENV{'LAZY_LOAD'}) {
                $models->$_ foreach keys(%{$models->get_models()});
            }

            $ENV{RESTAPI_VERSION} = '1.0';

            _mock_app($models);
            mock_memcached($models);
            mock_for_config($models, $option_config) if $option_config;
            use_perl_do_actions($models, $use_perl_do_actions) if defined($use_perl_do_actions);
            use_perl_get_all($models, $use_perl_get_all) if defined($use_perl_get_all);

            my @tests_data  = ();
            my $self_update = FALSE;
            my %TEST_VARS   = (
                LAST_CREATED_ID => 'NO',
                LAST_CHANGED_ID => 'NO',
                TOKEN           => 'NO',
                FIELDS          => $fields,
            );

            foreach my $test (@$tests) {
                my $token = $TEST_VARS{'TOKEN'} = $test->{'request'}->{'token'};

                my ($resource) = ($test->{'request'}{'url'} =~ m{^/v1/([^/]+)/});

                my $req = _process_variables($models, $resource, $test->{'request'}, \%TEST_VARS);

                my $name = $req->{'name'};
                my $url  = $req->{'url'};

                next if $test_name_filter && $name !~ /^${test_name_filter}/;

                throw 'name must be defined' unless defined($name);
                throw 'url must be defined'  unless defined($url);

                note($name);

                apply_mocks($models, [$default_mock_options, $structure->{'options'}, $req, $req->{options}], 'tjson');

                if ($req->{'lang'} || $option_lang) {
                    mock_lang_detect($models, $req->{'lang'} || $option_lang);
                    _unmock_dcnpgettext();
                } else {
                    _mock_dcnpgettext();
                }

                my $method       = $req->{'method'}       // 'get';
                my $accept       = $req->{'accept'}       // 'application/vnd.api+json';
                my $content_type = $req->{'content_type'} // 'application/vnd.api+json';
                my $cookies      = $req->{'cookies'};
                my $req_opts     = $req->{'options'}      // {};

                if (defined($cookies)) {
                    $app->ua->cookie_jar->add(
                        Mojo::Cookie::Response->new(
                            name   => $_,
                            value  => $cookies->{$_},
                            domain => '127.0.0.1',
                            path   => '/'
                        )
                    ) foreach keys(%$cookies);
                }

                if ($method eq 'get') {
                    $app->get_ok(
                        $url => {
                            Accept => $accept,
                            (defined($token) ? (Authorization => "token $token") : ()),
                        }
                    );
                } elsif ($method eq 'patch') {
                    $app->patch_ok(
                        $url => {
                            Accept         => $accept,
                            'Content-Type' => $content_type,
                            (defined($token) ? (Authorization => "token $token") : ()),
                        },
                        ($req->{'body'} ? (json => $req->{'body'}) : ()),
                    );
                } elsif ($method eq 'post') {
                    $app->post_ok(
                        $url => {
                            Accept         => $accept,
                            'Content-Type' => $content_type,
                            (defined($token) ? (Authorization => "token $token") : ()),
                        },
                        ($req->{'body'} ? (json => $req->{'body'}) : ()),
                    );
                } elsif ($method eq 'delete') {
                    $app->delete_ok(
                        $url => {
                            Accept         => $accept,
                            'Content-Type' => $content_type,
                            (defined($token) ? (Authorization => "token $token") : ()),
                        },
                        ($req->{'body'} ? (json => $req->{'body'}) : ()),
                    );
                }

                reset_features($models, $req->{'mock_features'}) if $req->{'mock_features'};

                my $body = from_json(fix_utf($app->tx->res->body()) || '""');

                if ($method eq 'post' && !exists($body->{'errors'})) {
                    my $last_object;
                    if (ref($body->{'data'}) eq 'HASH') {
                        $last_object = $body->{'data'};
                    } else {
                        $last_object = $body->{'data'}[-1];
                    }

                    if (ref($last_object) eq 'HASH') {
                        $TEST_VARS{'LAST_CHANGED_ID'} = $last_object->{'id'};
                    } else {
                        $TEST_VARS{'LAST_CHANGED_ID'} = undef;
                    }

                    if ($url !~ /action/) {
                        $TEST_VARS{'LAST_CREATED_ID'} = $TEST_VARS{'LAST_CHANGED_ID'};
                    }
                }

                diag("ANSWER: \n" . to_json($body, pretty => TRUE) . "\n") if $ENV{'VIEW_ANSWER'};

                if ($body) {
                    foreach my $key (qw(links meta)) {
                        delete $body->{$key} if $req_opts->{"no_$key"};
                    }
                    my $data = $body->{data};
                    if ($data) {
                        $data = [$data] if ref($data) eq 'HASH';
                        foreach my $key (qw(relationships links)) {
                            if ($req_opts->{"no_$key"}) {
                                delete $_->{$key} foreach @$data;
                            }
                        }
                    }
                }

                # Leave only the keys by specified json path
                # NOTE! only dot syntax for HASHes supported for now
                if ($req_opts->{"json_path"}) {
                    my $res = {};
                    foreach my $cur_path (@{$req_opts->{"json_path"}}) {
                        my $path = [split /\./, $cur_path];
                        shift @$path if $path->[0] eq '';
                        _copy_by_path($body, \$res, $path, 0);
                    }
                    $res->{errors} = $body->{errors} if $body->{errors};
                    $body = $res;
                }

                my $response = $test->{'response'};
                if (defined($response) && !$ENV{'SELF_UPDATE'}) {

                    $response = _process_variables($models, $resource, $response, \%TEST_VARS);

                    if (exists($response->{'status'})) {
                        $app->status_is($response->{'status'});
                    }

                    if (exists($response->{'content_type'})) {
                        $app->header_is('Content-type' => $response->{'content_type'});
                    }

                    my $test_name = join "\n\t", $name, map {sprintf '"%s" : "%s"', $_, $req->{$_}} qw(method url);

                    my $cmp_func = $ENV{'VIEW_ANSWER'} ? \&eq_or_diff : \&cmp_deeply;
                    $cmp_func->($body, $response->{'body'}, $test_name, {Sortkeys => 1, context => 2});
                } else {
                    $self_update = TRUE;

                    $response = {
                        status => $app->tx->res->code,
                        body   => $body,
                        $req_opts->{"no_content_type"}
                        ? ()
                        : (content_type => $app->tx->res->headers->content_type),
                    };
                }

                push(@tests_data, {request => $test->{'request'}, response => $response});
            }

            save_databases($models, $MOCKED_DATABASES_RESTAPI) if $ENV{'SAVE_DATABASES'} && !$option_dont_create_db;

            if ($self_update) {
                $structure->{'tests'} = \@tests_data;

                if (defined($header)) {
                    writefile($file_path_or_structure, $header . to_json($structure, pretty => TRUE));
                } else {
                    my @caller = caller(2);

                    my $file_path = $caller[1];
                    my $line      = $caller[2];

                    open(my $fh, '<', $file_path);

                    my ($before_test, $after_test) = ('', '');
                    my $cur_line = 0;
                    while (<$fh>) {
                        $cur_line++;

                        if ($cur_line < $line) {
                            $before_test .= $_;
                        } else {
                            $after_test .= $_;
                        }
                    }

                    local $Data::Dumper::Terse = 1;

                    my $test_dump = "test(\n" . Dumper($structure) . ");\n\n";

                    $after_test =~ s/^test\(\s*.*\s*\);\s+/$test_dump/msi;

                    writefile($file_path, $before_test . $after_test);
                }

                fail('Not found response');
            }
        },
        application_package  => 'RestApi',
        init                 => $option_init,
        create_clickhouse_db => $option_create_clickhouse_db,
        $option_dont_create_db ? (dont_create_database => 1)
        : (
            reconnect        => $database_created,
            mocked_databases => $MOCKED_DATABASES_RESTAPI,
            (
                $option_readonly
                ? (
                    reuse_database  => TRUE,
                    reuse_db_suffix => $READONLY_DATABASES_PREFIX,
                  )
                : (keep_databases => $ENV{KEEP_DATABASES})
            )
          ),
        $option_fixtures
        ? (
            fill_databases => 0,
            fixtures       => $option_fixtures,
          )
        : (),
    );

    $database_created = TRUE;
}

sub _copy_by_path {
    my ($src, $dst_ref, $path, $i) = @_;

    my $key = $path->[$i];
    if (ref($src) eq 'HASH') {
        $$dst_ref //= {};
        if (exists $src->{$key}) {
            if ($i == $#$path) {
                $$dst_ref->{$key} = $src->{$key};
            } else {
                _copy_by_path($src->{$key}, \$$dst_ref->{$key}, $path, $i + 1);
            }
        } else {
            unless ($key eq 'data' && $i == 0 && $src->{errors}) {
                $$dst_ref->{$key} = sprintf('KEY "%s" NOT FOUND', $key);
            }

        }
    } elsif (ref($src) eq 'ARRAY') {
        if ($key eq '[]') {
            $$dst_ref //= [];
            for my $j (0 .. $#$src) {
                _copy_by_path($src->[$j], \$$dst_ref->[$j], $path, $i + 1);
            }
        } else {
            $$dst_ref->{$key} = sprintf('BAD json_path STRUCTURE: unable fetch key "%s" from array', $key);
        }
    }
}

sub _process_variables {
    my ($models, $resource, $data, $vars) = @_;

    my ($last_created_id, $last_changed_id, $token, $fields) =
      @$vars{qw( LAST_CREATED_ID  LAST_CHANGED_ID  TOKEN  FIELDS )};

    my $json = to_json($data);

    $json =~ s/\$LAST_CREATED_ID/$last_created_id/g;
    $json =~ s/\$LAST_CHANGED_ID/$last_changed_id/g;
    $json =~ s/\$TOKEN/$token/g;
    $json =~ s/\$FIELDS\[(\w+?)\]/set_fields_by_accessor($fields, $1)/ge;
    $json =~ s/\$ALL_FIELDS\[(\w+?)\]/set_all_fields_by_accessor($models, $1)/ge;
    $json =~ s/\$FIXTURE_PUBLIC_ID\[([^\]]+?)\]/set_fixture_public_id_by_name($models, $resource, $1)/ge;

    $data = from_json($json);

    return $data;
}

sub set_fixture_public_id_by_name {
    my ($models, $resource, $params) = @_;

    my @params = split /\s*,\s*/, $params;

    my $fixture_name;
    if (@params == 2) {
        $resource     = $params[0];
        $fixture_name = $params[1];
    } elsif (@params == 1) {
        $fixture_name = $params[0];
    } else {
        throw "Wrong number of params for fixture id macro: $params";
    }

    return $models->$resource->public_id(get_fixture($fixture_name));
}

sub _mock_app {
    my ($app) = @_;

    my $api_cabinet = $app->api_cabinet;

    $api_cabinet = Test::MockObject::Extends->new($api_cabinet);
    $api_cabinet->mock(
        'check_key',
        sub {
            my ($api_method, %opts) = @_;

            my $login = $opts{'key'};

            my $user = $app->partner_db->users->get_all(
                fields => [qw(id login client_id)],
                filter => {login => $login}
            )->[0];

            if (defined($user)) {
                return {
                    key_info => {
                        dt     => "2015-08-31T13:29:55.611000",
                        hidden => 0,
                        id     => $login,
                        name   => "Key #$login",
                        user   => {
                            email             => "devnull\@yandex.ru",
                            login             => $user->{'login'},
                            name              => "Pupkin Vasily",
                            uid               => $user->{'id'},
                            roles             => ['user'],
                            balance_client_id => $user->{'client_id'},
                        },
                    },
                    result => "OK",
                };
            } elsif ($login eq "passport-only-user") {
                return {
                    key_info => {
                        dt     => "2015-08-31T13:29:55.611000",
                        hidden => 0,
                        id     => $login,
                        name   => "Key #$login",
                        user   => {
                            email             => "devnull\@yandex.ru",
                            login             => $login,
                            name              => "Pupkin Vasily",
                            uid               => 2**64 - 1,
                            roles             => ['user'],
                            balance_client_id => 2**64 - 1,
                        },
                    },
                    result => "OK",
                };
            } else {
                return {result => 'ERROR', error => 'NOT FOUND'};
            }
        }
    );

    mock_lang_detect($app, $ENV{'MOCK_LANG_DETECT'}) if defined($ENV{'MOCK_LANG_DETECT'});

    mock_utils_partner2($app);

    mock_geobase($app);
}

sub _mock_dcnpgettext {
    no warnings 'redefine';
    no strict 'refs';

    *{'Locale::gettext_pp::dcnpgettext'} = sub ($$$$$$) {return $_[2];}
}

sub _unmock_dcnpgettext {
    no warnings 'redefine';
    no strict 'refs';
    *{'Locale::gettext_pp::dcnpgettext'} = $sub_dcnpgettext;
}

TRUE;
