package Test::TestBuilder;

=pod

Слегка переработанная копия модуля из БК
(см. https://arc.yandex-team.ru/wsvn/arc/trunk/arcadia/yabs/utils/yabs-functest/Yabs/FuncTest/TestBuilder.pm )

=cut

use utf8;
use strict;
use warnings FATAL => 'all';

use Data::Dumper;
use Test::More;
use Test::Builder;
use Test::UTF8Builder;

use Term::ANSIColor;

use Scalar::Util qw/openhandle/;

our $c_ok          = '';
our $c_not_ok      = '';
our $c_reset       = '';
our $c_debug       = '';
our $c_debug_error = '';

# This is a copy-paste of Test::TestBuilder::ok subroutine with some custom modifications
sub custom_ok {
    my $self = shift;
    my $test = shift;
    my $name = shift;

    if ($self->{Child_Name} && not $self->{In_Destroy}) {
        $name = 'unnamed test' unless defined $name;
        $self->is_passing(0);
        $self->croak("Cannot run test ($name) with active children");
    }
    # $test might contain an object which we don't want to accidentally
    # store, so we turn it into a boolean.
    $test = $test ? 1 : 0;

    lock $self->{Curr_Test};
    $self->{Curr_Test}++;

    # In case $name is a string overloaded object, force it to stringify.
    $self->_unoverload_str(\$name);

    $self->diag(<<"ERR") if defined $name && $name =~ /^[\d\s]+$/;
    You named your test '$name'.  You shouldn't use numbers for your test names.
    Very confusing.
ERR

    # Capture the value of $TODO for the rest of this ok() call
    # so it can more easily be found by other routines.
    my $todo    = $self->todo();
    my $in_todo = $self->in_todo;
    local $self->{Todo} = $todo if $in_todo;

    $self->_unoverload_str(\$todo);

    my $out    = '';
    my $result = &Test::Builder::share({});

    if (!$test) {
        $out .= $c_not_ok . "not ok";
        $out .= " $self->{Curr_Test}" if $self->use_numbers;
        @$result{'ok', 'actual_ok'} = (($self->in_todo ? 1 : 0), 0);
    } else {
        $out .= $c_ok . "ok";
        $out .= " $self->{Curr_Test}" if $self->use_numbers;
        $out .= $c_reset;
        @$result{'ok', 'actual_ok'} = (1, $test);
    }

    if (defined $name) {
        $name =~ s/#/\\#/g;    # # in a name can confuse Test::Harness.
        $out .= " - $name";
        $result->{name} = $name;
    } else {
        $result->{name} = '';
    }

    $out .= $c_reset if !$test;

    if ($self->in_todo) {
        $out .= " # TODO $todo";
        $result->{reason} = $todo;
        $result->{type}   = 'todo';
    } else {
        $result->{reason} = '';
        $result->{type}   = '';
    }

    $self->{Test_Results}[$self->{Curr_Test} - 1] = $result;
    $out .= "\n";

    $self->_print($out);

    unless ($test) {
        my $msg = $self->in_todo ? "Failed (TODO)" : "Failed";
        $self->_print_to_fh($self->_diag_fh, "\n") if $ENV{HARNESS_ACTIVE};

        if (!_is_internal_test()) {
            my (undef, $file, $line) = $self->caller;
            if (defined $name) {
                $self->diag(qq[  $msg test '$name'\n]);
                $self->diag(qq[  at $file line $line.\n]);
            } else {
                $self->diag(qq[  $msg test at $file line $line.\n]);
            }
        } elsif ($ENV{HARNESS_ACTIVE}) {
            if ($name =~ /(.*?):\s+(.*)/) {
                $self->diag("  $msg test '$1'\n");
                $self->diag("  $2\n");
            } else {
                $self->diag("  $msg test '$name'");
            }
        }
    }

    $self->is_passing(0) unless $test || $self->in_todo;

    # Check that we haven't violated the plan
    $self->_check_is_passing_plan();

    return $test ? 1 : 0;
}

# Skip '1..N' plan messages for subtests
sub custom_output_plan {
    my $self      = shift;
    my $max       = shift;
    my $directive = shift;
    my $reason    = shift;

    $self->carp("The plan was already output") if $self->{Have_Output_Plan};

    my $plan = "1..$max";
    $plan .= " # $directive" if defined $directive;
    $plan .= " $reason"      if defined $reason;

    if ($self->_indent eq '') {
        $self->_print("$plan\n");
    }

    $self->{Have_Output_Plan} = 1;

    return;
}

# Is test called from Yabs::FuncTest namespace?
sub _is_internal_test {
    # Skip internal Test::Builder calls
    for my $i (3 .. 10) {
        my ($package) = caller($i);

        return 0 if !$package;
        return 1 if $package =~ /^Yabs::FuncTest::/;
        return 0 if $package !~ /^Test::/;
    }

    return 0;
}

# Is output is going directly to terminal?
sub is_interactive {
    my ($out_handle) = (@_, select);    # Default to default output handle

    # Not interactive if output is not to terminal...
    return 0 if not -t $out_handle;

    # If *ARGV is opened, we're interactive if...
    if (openhandle * ARGV) {
        # ...it's currently opened to the magic '-' file
        return -t *STDIN if defined $ARGV && $ARGV eq '-';

        # ...it's at end-of-file and the next file is the magic '-' file
        return @ARGV > 0 && $ARGV[0] eq '-' && -t *STDIN if eof *ARGV;

        # ...it's directly attached to the terminal
        return -t *ARGV;
    }

    # If *ARGV isn't opened, it will be interactive if *STDIN is attached
    # to a terminal.
    else {
        return -t *STDIN;
    }
}

sub init_builder {
    if (is_interactive()) {
        $c_ok          = Term::ANSIColor::color('green');
        $c_not_ok      = Term::ANSIColor::color('red');
        $c_reset       = Term::ANSIColor::color('reset');
        $c_debug       = Term::ANSIColor::color('cyan');
        $c_debug_error = Term::ANSIColor::color('bright_magenta');
    }

    if ($Test::Builder::VERSION eq '0.98') {
        no warnings 'redefine';
        *Test::Builder::ok           = \&custom_ok;
        *Test::Builder::_output_plan = \&custom_output_plan;
    }
}

INIT {
    init_builder();
}

1;
