package TestSubmit;

use strict;
use warnings FATAL => 'all';
use utf8;
use open qw(:std :utf8);

use Cpanel::JSON::XS;
use Data::Dumper;
use Encode;
use Clone 'clone';
use Test::More;
use Test::Differences;
use Test::Deep;
use Path::Tiny;

use Utils;
use Submit;
use Partner2;
use Branches;

use Exporter;

our @ISA       = qw(Exporter);
our @EXPORT_OK = qw(
  get_new_client_id
  mock_create_offer
  mock_update_contract
  mock_check_pi_adfox_contracts
  mock_get_partner2_user_data
  unmock_get_partner2_user_data
  mock_is_yamoney_account_identificated_for_russia
  mock_is_yamoney_account_identificated_for_ukraine
  is_person_ok
  is_partner2_user_data_ok
  is_submit_ok
  mock_check_bik_and_account
  );
our @EXPORT = @EXPORT_OK;

sub get_new_client_id {
    my ($user_id) = @_;

    my $current_client_id = get_client_id($user_id);

    remove_user_client_association(
        operator_user_id => $user_id,
        client_id        => $current_client_id,
        user_id          => $user_id,
    );

    my $new_client_id = get_client_id($user_id);

    return $new_client_id;
}

sub mock_create_offer {
    my ($data, $update) = @_;

    # К сожалению, в балансе нет ручки удалить договор. Поэтому пришлось замокать в тесте создание догвора.
    # Было бы лучше создавать в тесте договор, а потом удалять его.
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::create_offer'} = sub {
        my (%opts) = @_;

        foreach my $field (qw(client_id person_id service_start_dt start_dt)) {
            $opts{$field} = "mocked_$field" if defined $opts{$field};
        }
        $update->{create_offer_args} = \%opts;
        eq_or_diff(\%opts, $data->{create_offer_args}, 'create_offer args') unless $data->{create_offer_args_skip};

        return {
            EXTERNAL_ID => 'РС-64233-05/18',
            ID          => 611822,
        };
    };
}

sub mock_update_contract {
    # Works only for russia_ph_part2
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::update_contract'} = sub {
        my (%opts) = @_;
        is($opts{operator_uid}, 607369863, 'update_contract operator_uid');
        is($opts{contract_id},  611822,    'update_contract contract_id');
    };
}

sub mock_check_pi_adfox_contracts {
    no strict 'refs';
    no warnings 'redefine';
    *{'Type::adfox_account::check_pi_adfox_contracts'} = sub {
        my (%opts) = @_;

        return {contracts => 'ok',};
    };
}

my $orig_get_partner2_user_data = \&{$Submit::{get_partner2_user_data}};

sub mock_get_partner2_user_data {
    # Этот мок используется только для ветки russia_ph_part2
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::get_partner2_user_data'} = sub {
        return {
            user => {},
            db   => {
                form_data => {
                    data => Cpanel::JSON::XS->new->encode(
                        {
                            contract => {ID => '12345',},
                            fields   => {
                                "cyrillic_first_name"                        => "Имя",
                                "cyrillic_last_name"                         => "Фамилия",
                                "cyrillic_patronymic_name"                   => "Отчество",
                                "date_of_birth"                              => "1970-05-14",
                                "email"                                      => 'email@yandex.ru',
                                "phone"                                      => "+7 495 739-70-00",
                                "campaign_creation"                          => undef,
                                "newsletter"                                 => JSON::PP::true,
                                "allow_storage_and_processing_personal_data" => JSON::PP::true,
                                "oferta_agreement"                           => JSON::PP::true,
                                "gdpr_warning"                               => undef
                            },
                        }
                    ),
                },
            },
        };
    };
}

sub unmock_get_partner2_user_data {
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::get_partner2_user_data'} = $orig_get_partner2_user_data;
}

sub mock_is_yamoney_account_identificated_for_russia {
    no strict 'refs';
    no warnings 'redefine';
    *{'Type::bik_with_account_or_yandex_money::is_yamoney_account_identificated'} = sub {
        my (%opts) = @_;
        is_deeply(
            \%opts,
            {
                account    => "4100143322850",
                firstname  => "Имя",
                lastname   => "Фамилия",
                middlename => "Отчество",
                passport   => "6606033791",
            },
            'is_yamoney_account_identificated args'
        );
        return 1;
    };
}

sub mock_is_yamoney_account_identificated_for_ukraine {
    no strict 'refs';
    no warnings 'redefine';
    *{'Type::yandex_money_details::is_yamoney_account_identificated'} = sub {
        my (%opts) = @_;
        is_deeply(
            \%opts,
            {
                account    => "4100143322850",
                firstname  => "Мариана",
                lastname   => "Ткаченко",
                middlename => "Ивановна",
                passport   => "123456789",
            },
            'is_yamoney_account_identificated args'
        );
        return 1;
    };
}

my $orig_create_person = \&Submit::create_person;

sub mock_create_person {
    my ($data, $update) = @_;
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::create_person'} = sub {
        my (%opts) = @_;

        foreach my $field (qw(client_id)) {
            $opts{$field} = "mocked_$field" if defined $opts{$field};
        }
        $update->{create_person_args} = \%opts;
        eq_or_diff(\%opts, $data->{create_person_args}, 'create_person args');

        goto \&$orig_create_person;
    };
}

sub unmock_create_person {
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::create_person'} = $orig_create_person;
}

my $orig_create_or_update_partner = \&Submit::create_or_update_partner;

sub mock_create_or_update_partner {
    my ($data, $update) = @_;
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::create_or_update_partner'} = sub {
        my (%opts) = @_;
        $update->{create_or_update_partner_args} = \%opts;
        eq_or_diff(\%opts, $data->{create_or_update_partner_args}, 'create_or_update_partner args');
        goto \&$orig_create_or_update_partner;
    };
}

sub unmock_create_or_update_partner {
    no strict 'refs';
    no warnings 'redefine';
    *{'Submit::create_or_update_partner'} = $orig_create_or_update_partner;
}

sub is_partner2_user_data_ok {
    my ($partner2_user_data, $expected) = @_;

    delete $partner2_user_data->{db}->{user}->{create_date};
    delete $partner2_user_data->{db}->{user}->{domain_login};
    delete $partner2_user_data->{db}->{user}->{last_payout};
    delete $partner2_user_data->{db}->{user}->{is_adfox_partner};

    eq_or_diff($partner2_user_data->{db}->{user}, $expected, "table user has expected data",);

    return $partner2_user_data->{db}->{user};
}

sub is_person_ok {
    my ($person_id, $client_id, $expected_person_data) = @_;

    note('Person ID: ', $person_id);

    cmp_deeply($person_id, re(qr/^(\d+)\z/), 'got person_id',);

    my $got_person = get_person($client_id, $person_id,);

    delete $got_person->{id};
    delete $got_person->{dt};

    eq_or_diff($got_person, $expected_person_data, "balance has expected person data",);

    return $got_person;
}

sub is_submit_ok {
    my ($test_name, $user_id, $user_login, $test_data, $rm_user) = @_;

    my ($submit, $user, $person, $submit_pre) = @$test_data{qw(submit user person submit_pre)};

    my %update = (submit => clone($submit),);

    my $client_id = get_new_client_id($user_id);

    $user->{client_id} = $client_id;
    $person->{client_id} = $client_id if $person;

    my $branch_id = $submit->{branch_id};
    my $project   = delete $submit->{project};

    mock_create_offer($test_data, \%update);
    mock_create_person($test_data, \%update);
    if ($submit_pre) {
        my $submit_obj = Submit->new(
            {
                user_id       => $user_id,
                display_login => $user_login,
                %{$submit_pre},
            },
            'ru', $project,
        );

        my ($is_valid, $errors) = $submit_obj->submit();
        if (not $is_valid) {
            note Dumper $errors;
        }
    }

    if ($test_name =~ /russia_ph_part2_yandex_money$/) {
        mock_is_yamoney_account_identificated_for_russia();
    } elsif ($test_name =~ /ukraine_ph$/) {
        mock_is_yamoney_account_identificated_for_ukraine();
    }

    mock_create_or_update_partner($test_data, \%update);

    my $submit_obj = Submit->new(
        {
            user_id       => $user_id,
            display_login => $user_login,
            %{$submit},
        },
        'ru', $project,
    );

    my ($is_valid, $errors) = $submit_obj->submit();
    if (not $is_valid) {
        note Dumper $errors;
    }

    is($is_valid, 1, "Submited $test_name");

    my $partner2_user_data = get_partner2_user_data($user_id);

    if ($user) {
        $update{user} = is_partner2_user_data_ok($partner2_user_data, $user);
    }

    my $branches = Branches->new(project => $project);
    if ($person) {
        if ($branches->working_with_balance($branch_id)) {
            $update{person} = is_person_ok($partner2_user_data->{db}->{form_data}->{person_id},
                $partner2_user_data->{db}->{user}->{client_id}, $person,);
        } else {
            fail('branch does not work with balance');
        }
    } elsif ($branches->working_with_balance($branch_id)) {
        fail('branch should work with balance');
    }

    unmock_create_person();
    unmock_create_or_update_partner();
    rm_user_in_db($user_id) if $rm_user;

    delete $update{user}{client_id}   if $update{user};
    delete $update{person}{client_id} if $update{person};
    return \%update;
}

sub mock_check_bik_and_account {
    no strict 'refs';
    no warnings 'redefine';

    *Yandex::Balance::check_bik_and_account = sub {
        my ($bik, $account) = @_;
        if ($bik eq "123123123" && $account eq "40702810638050013199") {
            return 0;
        } else {
            return 1;
        }
    };
}

1;
