#!/usr/bin/perl -w
use Test::More;

use qbit;

use FindBin qw($Bin);
use lib "$Bin/lib";
use lib "$Bin/../lib";
use lib "$Bin/../utils/lib";

use Application;
use Test::Partner::DB::Mock;
use Test::Partner::Utils qw(mock_curdate);
use Test::MockObject::Extends;

my $app = Application->new();

$app->pre_run();

my $dbh = Test::Partner::DB::Mock->connect($app);
$dbh->mock($app);
$dbh->create_tables(partner_db => [qw(kv_store)]);

$app->statistics_internal_advnet_context_on_site_stripe;
$app->{'statistics_internal_advnet_context_on_site_stripe'} =
  Test::MockObject::Extends::Easy->new($app->{statistics_internal_advnet_context_on_site_stripe});
$app->statistics_internal_advnet_context_on_site_stripe->set_always('update_statistics', TRUE);
$app->statistics_internal_advnet_context_on_site_stripe->mock('__get_month_update_date',
    \&Application::Model::Statistics::Update::__get_month_update_date)
  ;    # Mock sub by itself to check whether it was called.

mock_curdate('2014-01-30 00:00:00');
my $update_for_month_min_date = get_update_for_month_min_date($app);

if ($update_for_month_min_date ne curdate(oformat => 'db_time')) {
    $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics();

    was_update_for_two_days_called($app);

    is_update_date_curdate($app);
    is_update_date_for_month($app, undef);

    $app->statistics_internal_advnet_context_on_site_stripe->clear();
}

{
    mock_curdate($update_for_month_min_date);
    $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics(for_month => TRUE);

    was_update_for_month_called($app);

    is_update_date_curdate($app);
    is_update_date_for_month_curdate($app);

    $app->statistics_internal_advnet_context_on_site_stripe->clear();
}

mock_curdate('2014-01-31 00:00:00');

my $last_update_for_month_date =
  $app->statistics_internal_advnet_context_on_site_stripe->kv_store->get(
    $app->statistics_internal_advnet_context_on_site_stripe->__get_kv_store_key_for_month(storage => 'mysql')),

  $update_for_month_min_date = get_update_for_month_min_date($app);

if ($update_for_month_min_date ne curdate(oformat => 'db_time')) {
    $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics();

    was_update_for_two_days_called($app);

    is_update_date_curdate($app);
    is_update_date_for_month($app, $last_update_for_month_date);

    $app->statistics_internal_advnet_context_on_site_stripe->clear();
}

mock_curdate(date_sub($update_for_month_min_date, second => 1, iformat => 'db_time', oformat => 'db_time'));

{
    $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics();

    was_update_for_two_days_called($app);

    is_update_date_curdate($app);
    is_update_date_for_month($app, $last_update_for_month_date);

    $app->statistics_internal_advnet_context_on_site_stripe->clear();
}

mock_curdate($update_for_month_min_date);

{
    $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics(for_month => TRUE);

    was_update_for_month_called($app);

    is_update_date_curdate($app);
    is_update_date_for_month_curdate($app);

    $app->statistics_internal_advnet_context_on_site_stripe->clear();
}

done_testing();
exit;

sub is_update_date_curdate {
    my ($app) = @_;

    is_update_date($app, curdate(oformat => 'db'));
}

sub is_update_date {
    my ($app, $date) = @_;

    is(
        $app->statistics_internal_advnet_context_on_site_stripe->kv_store->get(
            $app->statistics_internal_advnet_context_on_site_stripe->__get_kv_store_key(storage => 'mysql')
        ),
        $date,
        curdate(oformat => 'db_time') . ': Last update date is correct.'
      );
}

sub is_update_date_for_month_curdate {
    my ($app) = @_;

    is_update_date_for_month($app, curdate(oformat => 'db'));
}

sub is_update_date_for_month {
    my ($app, $date) = @_;

    is(
        $app->statistics_internal_advnet_context_on_site_stripe->kv_store->get(
            $app->statistics_internal_advnet_context_on_site_stripe->__get_kv_store_key_for_month(storage => 'mysql')
        ),
        $date,
        curdate(oformat => 'db_time') . ': Last update date for month is correct.'
      );
}

sub was_update_for_two_days_called {
    my ($app) = @_;

    ok(!$app->statistics_internal_advnet_context_on_site_stripe->called('__get_month_update_date'),
        curdate(oformat => 'db_time') . ': Update for two days was called');
}

sub was_update_for_month_called {
    my ($app) = @_;

    ok($app->statistics_internal_advnet_context_on_site_stripe->called('__get_month_update_date'),
        curdate(oformat => 'db_time') . ': Update for month was called');
}

sub get_update_for_month_min_date {
    my ($app) = @_;

    my $update_for_month_min_time = $app->get_option('statistics_update_for_month_min_time');
    my $update_for_month_min_date = [@{curdate()}[0 .. 2], @{$update_for_month_min_time}{'hour', 'minute', 'second'}];
    $update_for_month_min_date = trdate('norm', 'db_time', $update_for_month_min_date);

    return $update_for_month_min_date;
}
