#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More;

use Test::Partner2::Fixture;
use Test::Partner2::Mock qw(mock_subs restore_subs);
use Test::Partner2::Simple;

use qbit;

my $AVATARS = [
    {
        id      => 10001,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f018'
    },
    {
        id      => 10002,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3f9/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f019'
    }
];

run_tests(
    sub {
        my ($app) = @_;

        my $indoor_page_pk = get_fixtures_hash(('partner_indoor_page'))->{partner_indoor_page};
        my $indoor_page = $app->indoor->get($indoor_page_pk, fields => ['*']);

        my $indoor_block_public_key = $app->indoor_block->add(
            "page_id"       => $indoor_page->{page_id},
            "caption"       => "caption CBA",
            "min_duration"  => 5,
            "max_duration"  => 12,
            "resolution"    => "800x600",
            "zone_category" => 1,
            "mincpm"        => 1,
            "strategy"      => 0,
            "photo_id_list" => [],
            "aspect_ratio"  => "1:2",
            "sound"         => 1,
            "touch_screen"  => 0,
        );

        my $indoor_block_added =
          $app->indoor_block->get($indoor_block_public_key, fields => ['multistate', 'multistate_name', 'moderation']);

        ok(!$app->indoor_block->check_multistate_flag($indoor_block_added->{multistate}, 'need_approve'),
            'ADD indoor block is not in need_approve when no photos');

        ok($app->indoor_block->check_multistate_flag($indoor_block_added->{multistate}, 'approved'),
            'ADD indoor block is approved right away when no photos');

        ok($app->indoor_block->check_multistate_flag($indoor_block_added->{multistate}, 'working'),
            'ADD indoor block is working right away when no photos');

        is_deeply(
            $indoor_block_added->{moderation},
            {'photo_id_list' => {}},
            'ADD indoor block wo photos moderation ok'
        );

        mock_subs(
            {
                'Application::Model::MDSAvatars::get_all' => sub {
                    return $AVATARS;
                },
            }
        );

        my $indoor_block_public_key2 = $app->indoor_block->add(
            "page_id"       => $indoor_page->{page_id},
            "caption"       => "caption CBA",
            "min_duration"  => 5,
            "max_duration"  => 12,
            "resolution"    => "800x600",
            "zone_category" => 1,
            "mincpm"        => 1,
            "strategy"      => 0,
            "photo_id_list" => [10001, 10002],
            "aspect_ratio"  => "1:2",
            "sound"         => 1,
            "touch_screen"  => 0,
        );

        my $indoor_block_added2 =
          $app->indoor_block->get($indoor_block_public_key2, fields => ['multistate', 'multistate_name', 'moderation']);

        ok($app->indoor_block->check_multistate_flag($indoor_block_added2->{multistate}, 'need_approve'),
            'ADD indoor block is in need_approve when there are photos');

        ok(!$app->indoor_block->check_multistate_flag($indoor_block_added2->{multistate}, 'approved'),
            'ADD indoor block is not approved when there are photos');

        is_deeply(
            $indoor_block_added2->{moderation},
            {
                'photo_id_list' => {
                    '10001' => {
                        'request_id' => '',
                        'verdict'    => 0
                    },
                    '10002' => {
                        'request_id' => '',
                        'verdict'    => 0
                    }
                }
            },
            'ADD indoor block with photos moderation ok'
        );

        $AVATARS = [];
        $app->indoor_block->do_action($indoor_block_public_key2, "edit", "photo_id_list" => [],);
        my $indoor_block_edited =
          $app->indoor_block->get($indoor_block_public_key2, fields => ['multistate', 'multistate_name', 'moderation']);

        ok(!$app->indoor_block->check_multistate_flag($indoor_block_edited->{multistate}, 'need_approve'),
            'EDIT indoor block is not in need_approve when no photos');

        ok($app->indoor_block->check_multistate_flag($indoor_block_edited->{multistate}, 'approved'),
            'EDIT indoor block is approved right away when no photos');

        ok($app->indoor_block->check_multistate_flag($indoor_block_edited->{multistate}, 'working'),
            'EDIT indoor block is working right away when no photos');

        is_deeply($indoor_block_edited->{moderation}, {'photo_id_list' => {}}, 'EDIT indoor block moderation ok');

    },
    do_not_die_on_fail  => 0,
    fill_databases      => 0,
    fixtures            => ['partner_indoor_page', 'user_cron'],
    application_package => 'Cron',
    user                => 'cron',
);
