#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More;

use Test::Partner2::Fixture;
use Test::Partner2::Mock qw(mock_subs);
use Test::Partner2::Simple;

use qbit;

my $AVATARS = [
    {
        id      => 10001,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f018'
    },
    {
        id      => 10002,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3f9/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f019'
    }
];

my @fixture_name_list = qw(
  partner_outdoor_block_with_moderation_for_dup_need_approve
  partner_outdoor_block_with_moderation_for_dup_rejected
  partner_outdoor_block_with_moderation_for_dup_approved
  );

run_tests(
    sub {
        my ($app) = @_;

        mock_subs(
            {
                'Application::Model::MDSAvatars::get_all' => sub {
                    return $AVATARS;
                },
            }
        );

        my $blocks = get_fixtures_hash(@fixture_name_list);

        my $block4dup_pid = $blocks->{partner_outdoor_block_with_moderation_for_dup_need_approve};
        my $block4dup =
          $app->outdoor_block->get($block4dup_pid,
            fields => [qw(page_id id caption address gps photo_id_list moderation)]);

        my $next_block_id = _inc_outdoor_next_block_id($app, $block4dup->{page_id});

        my $block_duped_public_id = $app->outdoor_block->do_action_with_result($block4dup, 'duplicate');
        my $block_duped =
          $app->outdoor_block->get($block_duped_public_id->{public_id}, fields => [qw(moderation multistate id)]);

        _inc_outdoor_next_block_id($app, $block4dup->{page_id});

        is_deeply(
            $block_duped->{moderation},
            {
                'address' => {
                    'Mocked Outdoor Block address' => {
                        'request_id' => '',
                        'verdict'    => 0
                    }
                },
                'caption' => {
                    'Mocked Outdoor Block' => {
                        'request_id' => '111666',
                        'verdict'    => -1
                    }
                },
                'gps' => {
                    '-11.1,66.6' => {
                        'request_id' => '',
                        'verdict'    => 0
                    }
                },
                'photo_id_list' => {
                    '10001' => {
                        'request_id' => '111667',
                        'verdict'    => 1
                    },
                    '10002' => {
                        'request_id' => '',
                        'verdict'    => 0
                    }
                }
            },
            'ok moderation.request_id set on dup'
        );

        ok($app->outdoor_block->check_multistate_flag($block_duped->{multistate}, 'need_approve'),
            'ok duped outdoor block is in need_approve');

        ### dup rejected block
        $block4dup_pid = $blocks->{partner_outdoor_block_with_moderation_for_dup_rejected};
        $block4dup =
          $app->outdoor_block->get($block4dup_pid,
            fields => [qw(page_id id caption address gps photo_id_list moderation)]);

        _inc_outdoor_next_block_id($app, $block4dup->{page_id});

        $block_duped_public_id = $app->outdoor_block->do_action_with_result($block4dup, 'duplicate');
        $block_duped =
          $app->outdoor_block->get($block_duped_public_id->{public_id}, fields => [qw(moderation multistate id)]);

        is_deeply(
            $block_duped->{moderation},
            {
                'address' => {
                    'Mocked Outdoor Block address' => {
                        'request_id' => '111665',
                        'verdict'    => 1
                    }
                },
                'caption' => {
                    'Mocked Outdoor Block' => {
                        'request_id' => '111666',
                        'verdict'    => -1
                    }
                },
                'gps' => {
                    '-11.1,66.6' => {
                        'request_id' => '111665',
                        'verdict'    => 1
                    }
                },
                'photo_id_list' => {
                    '10001' => {
                        'request_id' => '111667',
                        'verdict'    => 1
                    },
                    '10002' => {
                        'request_id' => '111668',
                        'verdict'    => -1
                    }
                }
            },
            'ok moderation.request_id set on dup'
        );

        ok($app->outdoor_block->check_multistate_flag($block_duped->{multistate}, 'rejected'),
            'ok duped outdoor block is in rejected');

        ### dup approved block
        $block4dup_pid = $blocks->{partner_outdoor_block_with_moderation_for_dup_approved};
        $block4dup =
          $app->outdoor_block->get($block4dup_pid,
            fields => [qw(page_id id caption address gps photo_id_list moderation)]);
        _inc_outdoor_next_block_id($app, $block4dup->{page_id});

        $block_duped_public_id = $app->outdoor_block->do_action_with_result($block4dup, 'duplicate');
        $block_duped =
          $app->outdoor_block->get($block_duped_public_id->{public_id}, fields => [qw(moderation multistate)]);

        is_deeply(
            $block_duped->{moderation},
            {
                'address' => {
                    'Mocked Outdoor Block address' => {
                        'request_id' => '111665',
                        'verdict'    => 1
                    }
                },
                'caption' => {
                    'Mocked Outdoor Block' => {
                        'request_id' => '111666',
                        'verdict'    => 1
                    }
                },
                'gps' => {
                    '-11.1,66.6' => {
                        'request_id' => '111665',
                        'verdict'    => 1
                    }
                },
                'photo_id_list' => {
                    '10001' => {
                        'request_id' => '111667',
                        'verdict'    => 1
                    },
                    '10002' => {
                        'request_id' => '111668',
                        'verdict'    => 1
                    }
                }
            },
            'ok moderation.request_id set on dup'
        );

        ok($app->outdoor_block->check_multistate_flag($block_duped->{multistate}, 'approved'),
            'ok duped outdoor block is in approved');
    },
    do_not_die_on_fail => 0,
    fill_databases     => 0,
    fixtures           => \@fixture_name_list,
);

sub _inc_outdoor_next_block_id {
    my ($app, $page_id) = @_;

    my $next_block_id = 0;
    if (!(defined $app->partner_db->outdoor_block_seq->get({page_id => $page_id}))) {
        $next_block_id = 1 + scalar @{$app->partner_db->outdoor_block->get_all(filter => {page_id => $page_id})};

        $app->partner_db->outdoor_block_seq->add({page_id => $page_id, next_block_id => $next_block_id});
    } else {

        $next_block_id = 1 + ($app->partner_db->outdoor_block_seq->get({page_id => $page_id})->{id} // 0);
        $app->partner_db->outdoor_block_seq->edit(filter => {page_id => $page_id}, next_block_id => $next_block_id);
    }
    return $next_block_id;
}
