#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More;

use Test::Partner2::Fixture;
use Test::Partner2::Mock qw(mock_subs);
use Test::Partner2::Simple;

use qbit;

run_tests(
    sub {
        my ($app) = @_;

        my $indoor_page_pk =
          get_fixtures_hash(('partner_indoor_page_with_moderation'))->{partner_indoor_page_with_moderation};
        my $indoor_page = $app->indoor->get($indoor_page_pk, fields => ['*']);

        mock_subs(
            {
                'Utils::Logger::WARN' => sub {
                    is(
                        $_[1],
                        'There is no element to apply moderation verdict',
                        'on_verdict_received: WARN on request_id not found'
                      );
                },
                'Application::Model::Page::update_in_bk' => sub {
                    is($_[1]->{page_id}, $indoor_page_pk->{page_id});
                },
            }
        );

        lives_ok {
            $app->indoor->on_verdict_received(
                $indoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 123,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request_id not found';

        lives_ok {
            $app->indoor->on_verdict_received(
                $indoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 111_111,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request found, "yes" verdict for caption';

        my $indoor_page_obj =
          $app->indoor->get($indoor_page, fields => ['caption', 'caption_approved', 'moderation', 'multistate']);
        is($indoor_page_obj->{caption}, 'changed InDoor page caption', 'changed caption is the caption');
        is(
            $indoor_page_obj->{caption_approved},
            'changed InDoor page caption',
            'changed caption is the new caption_approved'
          );

        is_deeply(
            $indoor_page_obj->{moderation}->{caption},
            {
                'approved InDoor page caption' => {
                    'request_id' => 211111,
                    'verdict'    => 1
                },
                'changed InDoor page caption' => {
                    'request_id' => '111111',
                    'verdict'    => 1
                }
            },
            'ok moderation.caption after Yes verdict received'
        );
        ok($app->indoor->check_multistate_flag($indoor_page_obj->{multistate}, 'need_approve'),
            'page is still in need_approve as not all fields are moderated');

        lives_ok {
            $app->indoor->on_verdict_received(
                $indoor_page,
                {
                    'name'       => 'address',
                    'request_id' => 111112,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request found, "yes" verdict for address';

        $indoor_page_obj =
          $app->indoor->get($indoor_page, fields => ['address', 'address_approved', 'moderation', 'multistate']);
        is($indoor_page_obj->{address}, 'changed InDoor page address st., 97', 'changed address is the address');
        is(
            $indoor_page_obj->{address_approved},
            'changed InDoor page address st., 97',
            'changed address is the new address_approved'
          );

        is_deeply(
            $indoor_page_obj->{moderation}->{address},
            {
                'approved InDoor page address st., 95' => {
                    'request_id' => '211112',
                    'verdict'    => 1
                },
                'changed InDoor page address st., 97' => {
                    'request_id' => '111112',
                    'verdict'    => 1
                }
            },
            'ok moderation.address after "Yes" verdict received'
        );
        ok($app->indoor->check_multistate_flag($indoor_page_obj->{multistate}, 'need_approve'),
            'page is still in need_approve as not all fields are moderated');

        lives_ok {
            $app->indoor->on_verdict_received(
                $indoor_page,
                {
                    'name'       => 'gps',
                    'request_id' => 111113,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request found, "yes" verdict for gps';

        $indoor_page_obj =
          $app->indoor->get($indoor_page, fields => ['gps', 'gps_approved', 'moderation', 'multistate']);
        is($indoor_page_obj->{gps},          '-87,85', 'changed gps is the address');
        is($indoor_page_obj->{gps_approved}, '-87,85', 'changed gps is the new gps_approved');
        is_deeply(
            $indoor_page_obj->{moderation}->{gps},
            {
                '-85,87' => {
                    'request_id' => '211113',
                    'verdict'    => 1
                },
                '-87,85' => {
                    'request_id' => '111113',
                    'verdict'    => 1
                }
            },
            'ok moderation.gps after "Yes" verdict received'
        );

        ok(!$app->indoor->check_multistate_flag($indoor_page_obj->{multistate}, 'need_approve'),
            'page is not in need_approve as all fields are moderated');

        lives_ok {
            $app->indoor->on_verdict_received(
                $indoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 111_111,
                    'verdict'    => FALSE,
                }
            );
        }
        'on_verdict_received: request found, "no" verdict for caption after "yes" verdict';

        $indoor_page_obj =
          $app->indoor->get($indoor_page, fields => ['caption', 'caption_approved', 'moderation', 'multistate']);
        is($indoor_page_obj->{caption}, 'changed InDoor page caption', 'changed caption is the caption');
        is(
            $indoor_page_obj->{caption_approved},
            'changed InDoor page caption',
            'changed caption is the new caption_approved'
          );
        is_deeply(
            $indoor_page_obj->{moderation}->{caption},
            {
                'approved InDoor page caption' => {
                    'request_id' => 211111,
                    'verdict'    => 1
                },
                'changed InDoor page caption' => {
                    'request_id' => '111111',
                    'verdict'    => 1
                }
            },
            'ok moderation.caption after "No" verdict received'
        );
        ok(!$app->indoor->check_multistate_flag($indoor_page_obj->{multistate}, 'need_approve'),
            'page is not in need_approve');

        my $indoor_page1 = {
            %$indoor_page,
            address    => 'address asdsadsa',
            caption    => 'caption asdas',
            gps        => '1,1',
            moderation => {
                address => {'address asdsadsa' => {request_id => '', verdict => 0},},
                caption => {'caption asdas'    => {request_id => 2,  verdict => -1},},
                gps     => {'1,1'              => {request_id => '', verdict => 0},},
            }
        };
        lives_ok {
            $app->indoor->on_moderation_request_sent(
                [$indoor_page1],
                [
                    {
                        meta  => {request_id => 110},
                        extra => {
                            page_id  => 1,
                            id       => 1,
                            fields   => ['address', 'gps'],
                            position => 0,
                        },
                    },
                ]
            );
        }
        'on_moderation_request_sent: address-gps request has been sent';

        $indoor_page_obj = $app->indoor->get($indoor_page1, fields => ['moderation', 'multistate']);
        is(
            $indoor_page_obj->{moderation}->{address}->{[keys %{$indoor_page_obj->{moderation}->{address}}]->[0]}
              ->{request_id},
            110,
            'address request_id correct'
          );
        is(
            $indoor_page_obj->{moderation}->{gps}->{[keys %{$indoor_page_obj->{moderation}->{gps}}]->[0]}->{request_id},
            110,
            'gps request_id correct'
          );
        is(
            $indoor_page_obj->{moderation}->{caption}->{[keys %{$indoor_page_obj->{moderation}->{caption}}]->[0]}
              ->{request_id},
            2,
            'caption request_id correct'
          );

        ###
        my $indoor_page2 = {
            %$indoor_page,
            moderation => {
                address => {'changed InDoor page address st., 97' => {request_id => '1', verdict => 1},},
                caption => {
                    'changed InDoor page caption -1' => {request_id => '2', verdict => -1},
                    'changed InDoor page caption'    => {request_id => '3', verdict => 1},
                },
                gps => {
                    '-87,01' => {request_id => '11', verdict => 0},
                    '-87,85' => {request_id => '1',  verdict => 1},
                },
            }
        };
        lives_ok {
            # set moderation to have -1 & 1 in same field
            # and approved value for a field still waiting verdict
            $app->indoor->do_action($indoor_page2, 'edit', moderation => $indoor_page2->{moderation},);
            # edit to change field value back to prev approved value
            $app->indoor->do_action($indoor_page2, 'edit', 'gps' => '-87,85',);
        }
        'edit page moderation 2';
        # check that page is approved without on_verdict_received, since we already have a positive verdict
        my $indoor_page2_obj = $app->indoor->get($indoor_page2, fields => ['multistate']);

        ok(
            !$app->indoor->check_multistate_flag($indoor_page2_obj->{multistate}, 'need_approve'),
            'page is not in need_approve since we already have a positive verdict for a changed value'
          );
        ok($app->indoor->check_multistate_flag($indoor_page2_obj->{multistate}, 'approved'), 'page is approved');

        # checking reject
        lives_ok {
            $app->partner_db->indoor->edit(
                $indoor_page->{id},
                {
                    multistate => 640,    # check_statistics + need_approve (as newly added page)
                }
            );
            $app->indoor->do_action(
                $indoor_page,
                'edit',
                caption_approved => undef,
                moderation       => {
                    address => {'changed InDoor page address st., 97' => {request_id => '1', verdict => 1},},
                    caption => {
                        'changed InDoor page caption -1'      => {request_id => '2', verdict => -1},
                        'changed InDoor page caption weeeeed' => {request_id => '3', verdict => 0},
                    },
                    gps => {'-87,85' => {request_id => '1', verdict => 1},},
                },
            );
            # edit to change field value back to prev rejected value
            $app->indoor->do_action($indoor_page, 'edit', 'caption' => 'changed InDoor page caption -1',);
        }
        'edit page moderation 3';
        # check that page is approved without on_verdict_received, since we already have a positive verdict
        my $indoor_page3_obj = $app->indoor->get($indoor_page, fields => ['multistate']);
        ok(
            !$app->indoor->check_multistate_flag($indoor_page3_obj->{multistate}, 'need_approve'),
            'page is not in need_approve since we already have a positive verdict for a changed value'
          );
        ok($app->indoor->check_multistate_flag($indoor_page3_obj->{multistate}, 'rejected'), 'page is rejected');
        ###
    },
    do_not_die_on_fail  => 0,
    fill_databases      => 0,
    fixtures            => ['partner_indoor_page_with_moderation', 'user_cron'],
    application_package => 'Cron',
    user                => 'cron',
);
