#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Exception;
use Test::More;

use Test::Partner2::Fixture;
use Test::Partner2::Mock qw(mock_subs);
use Test::Partner2::Simple;

use qbit;

run_tests(
    sub {
        my ($app) = @_;

        my $outdoor_page_pk =
          get_fixtures_hash(('partner_outdoor_page_with_moderation'))->{partner_outdoor_page_with_moderation};
        my $outdoor_page = $app->outdoor->get($outdoor_page_pk, fields => ['*']);

        mock_subs(
            {
                'Utils::Logger::WARN' => sub {
                    is(
                        $_[1],
                        'There is no element to apply moderation verdict',
                        'on_verdict_received: WARN on request_id not found'
                      );
                },
                'Application::Model::Page::update_in_bk' => sub {
                    is($_[1]->{page_id}, $outdoor_page_pk->{page_id});
                },
            }
        );

        lives_ok {
            $app->outdoor->on_verdict_received(
                $outdoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 123,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request_id not found';

        lives_ok {
            $app->outdoor->on_verdict_received(
                $outdoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 111_111,
                    'verdict'    => TRUE,
                }
            );
        }
        'on_verdict_received: request found, "yes" verdict for caption';

        my $outdoor_page_obj =
          $app->outdoor->get($outdoor_page, fields => ['caption', 'caption_approved', 'moderation', 'multistate']);
        is($outdoor_page_obj->{caption}, 'changed OutDoor page caption', 'changed caption is the caption');
        is(
            $outdoor_page_obj->{caption_approved},
            'changed OutDoor page caption',
            'changed caption is the new caption_approved'
          );

        is_deeply(
            $outdoor_page_obj->{moderation}->{caption},
            {
                'approved OutDoor page caption' => {
                    'request_id' => 211111,
                    'verdict'    => 1
                },
                'changed OutDoor page caption' => {
                    'request_id' => '111111',
                    'verdict'    => 1
                }
            },
            'ok moderation.caption after Yes verdict received'
        );

        ok(!$app->outdoor->check_multistate_flag($outdoor_page_obj->{multistate}, 'need_approve'),
            'page is not in need_approve as all fields are moderated');

        lives_ok {
            $app->outdoor->on_verdict_received(
                $outdoor_page,
                {
                    'name'       => 'caption',
                    'request_id' => 111_111,
                    'verdict'    => FALSE,
                }
            );
        }
        'on_verdict_received: request found, "no" verdict for caption after "yes" verdict';

        $outdoor_page_obj =
          $app->outdoor->get($outdoor_page, fields => ['caption', 'caption_approved', 'moderation', 'multistate']);
        is($outdoor_page_obj->{caption}, 'changed OutDoor page caption', 'changed caption is the caption');
        is(
            $outdoor_page_obj->{caption_approved},
            'changed OutDoor page caption',
            'changed caption is the new caption_approved'
          );
        is_deeply(
            $outdoor_page_obj->{moderation}->{caption},
            {
                'approved OutDoor page caption' => {
                    'request_id' => 211111,
                    'verdict'    => 1
                },
                'changed OutDoor page caption' => {
                    'request_id' => '111111',
                    'verdict'    => 1
                }
            },
            'ok moderation.caption after "No" verdict received'
        );
        ok(!$app->outdoor->check_multistate_flag($outdoor_page_obj->{multistate}, 'need_approve'),
            'page is not in need_approve');

        my $outdoor_page1 = {
            %$outdoor_page,
            caption    => 'caption asdas',
            moderation => {caption => {'caption asdas' => {request_id => 2, verdict => -1},},}
        };
        is(
            $outdoor_page_obj->{moderation}->{caption}->{[keys %{$outdoor_page_obj->{moderation}->{caption}}]->[0]}
              ->{request_id},
            211111,
            'caption request_id correct'
          );

        ###
        my $outdoor_page2 = {
            %$outdoor_page,
            moderation => {
                caption => {
                    'changed OutDoor page caption -1' => {request_id => '2', verdict => -1},
                    'changed OutDoor page caption'    => {request_id => '3', verdict => 1},
                },
            }
        };
        lives_ok {
            # set moderation to have -1 & 1 in same field
            # and approved value for a field still waiting verdict
            $app->outdoor->do_action($outdoor_page2, 'edit', moderation => $outdoor_page2->{moderation},);
            # edit to change field value back to prev approved value
            $app->outdoor->do_action($outdoor_page2, 'edit', 'caption' => 'changed OutDoor page caption',);
        }
        'edit page moderation 2';
        # check that page is approved without on_verdict_received, since we already have a positive verdict
        my $outdoor_page2_obj = $app->outdoor->get($outdoor_page2, fields => ['multistate']);

        ok(
            !$app->outdoor->check_multistate_flag($outdoor_page2_obj->{multistate}, 'need_approve'),
            'page is not in need_approve since we already have a positive verdict for a changed value'
          );
        ok($app->outdoor->check_multistate_flag($outdoor_page2_obj->{multistate}, 'approved'), 'page is approved');

        # checking reject
        lives_ok {
            $app->partner_db->outdoor->edit(
                $outdoor_page->{id},
                {
                    multistate => 640,    # check_statistics + need_approve (as newly added page)
                }
            );
            $app->outdoor->do_action(
                $outdoor_page,
                'edit',
                caption_approved => undef,
                moderation       => {
                    caption => {
                        'changed OutDoor page caption -1'      => {request_id => '2', verdict => -1},
                        'changed OutDoor page caption weeeeed' => {request_id => '3', verdict => 0},
                    },
                },
            );
            # edit to change field value back to prev rejected value
            $app->outdoor->do_action($outdoor_page, 'edit', 'caption' => 'changed OutDoor page caption -1',);
        }
        'edit page moderation 3';
        # check that page is approved without on_verdict_received, since we already have a positive verdict
        my $outdoor_page3_obj = $app->outdoor->get($outdoor_page, fields => ['multistate']);
        ok(
            !$app->outdoor->check_multistate_flag($outdoor_page3_obj->{multistate}, 'need_approve'),
            'page is not in need_approve since we already have a positive verdict for a changed value'
          );
        ok($app->outdoor->check_multistate_flag($outdoor_page3_obj->{multistate}, 'rejected'), 'page is rejected');
        ###
    },
    do_not_die_on_fail  => 0,
    fill_databases      => 0,
    fixtures            => ['partner_outdoor_page_with_moderation', 'user_cron'],
    application_package => 'Cron',
    user                => 'cron',
);
