#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock;

use Test::Most;
use Test::More;
use Test::Deep;
use Test::MockObject::Extends;
use Test::Partner::Utils qw(
  mock_curdate
  get_new_context_on_site_adblock
  get_test_data_and_update_if_needed
  );

use qbit;
use PiConstants qw($IS_PRECISE);

$Test::Partner2::Mock::PAGE_ID_START = 153746;

run_tests(
    sub {
        my ($app) = @_;

        mock_app($app);
        mock_utils_partner2($app);

        cmp_deeply(
            $app->context_on_site_adblock->get_types_direct_blocks(),
            [
                {
                    'max_banners'   => 9,
                    'min_banners'   => 1,
                    'caption'       => re(qr(\w)),
                    'id'            => 'vertical',
                    'short_caption' => re(qr(\w)),
                }
            ],
            'get_types_direct_blocks'
        );

        cmp_deeply($app->context_on_site_adblock->get_types_media_blocks(), [], 'get_types_media_blocks');

        cmp_deeply($app->context_on_site_adblock->get_types_dsp_blocks(), [], 'get_types_dsp_blocks');

        my $id = $app->context_on_site_campaign->add(domain_id => 1, caption => 'Test campaign');

        my $campaign = $app->context_on_site_campaign->get($id, fields => ['id', 'page_id']);

        my $page_id = $campaign->{'page_id'};

        cmp_deeply(
            $campaign,
            {
                id      => $id,
                page_id => $page_id,
            },
            'check campaign'
        );

        $app->partner_db->context_on_site_campaign->edit($id, {tag_id => 1});

        cmp_deeply([sort keys(%{$app->context_on_site_adblock->get_strategies()})], [0], 'check get_strategies');

        cmp_deeply(
            [grep {$_->{'page_id'} == $page_id} @{$app->context_on_site_adblock->get_all_campaigns_for_adding()}],
            [
                {
                    'domain'  => 'showmethemoney.ru',
                    'caption' => 'Test campaign',
                    'page_id' => $page_id
                }
            ],
            'Check get_all_campaigns_for_adding for partner'
        );

        my $public_id = $app->context_on_site_adblock->add(%{get_new_context_on_site_adblock({page_id => $page_id})});

        ok($public_id =~ m/R-AB-$page_id-(\d+)/, 'check public_id');
        my $block_id = $1;

        cmp_deeply(
            $app->context_on_site_adblock->get($public_id, fields => ['*']),
            {
                'bk_data' => undef,
                'dsps'    => [1],
                'actions' => {
                    'delete'                 => re(qr(\w)),
                    'delete_with_page'       => re(qr(\w)),
                    'reset_check_statistics' => re(qr(\w)),
                    'edit'                   => re(qr(\w)),
                    'duplicate'              => re(qr(\w)),
                },
                'page' => {
                    'domain'     => 'showmethemoney.ru',
                    'caption'    => 'Test campaign',
                    'client_id'  => '11016',
                    'page_id'    => $page_id,
                    'owner_id'   => '1016',
                    'login'      => 'mocked-context-adblock-partner',
                    'multistate' => '2112',                             # 64 - check_statistics, 2048 - need_update
                    'owner'      => {
                        'business_unit'    => '0',
                        'client_id'        => '11016',
                        'current_currency' => 'RUB',
                        'email'            => 'devnull@yandex.ru',
                        'features'         => ['turbo_desktop_available'],
                        'id'               => '1016',
                        'is_assessor'      => 0,
                        'lastname'         => '',
                        'login'            => 'mocked-context-adblock-partner',
                        'midname'          => '',
                        'name'             => ''
                    },
                    'cur_user_is_read_assistant' => '',
                    'blocks_limit'               => 200,
                    'blocks_count'               => 1,
                    'id'                         => $id,
                    'is_protected'               => '',
                    'is_tutby'                   => undef,
                },
                'fields_depends'   => {},
                'strategy'         => '0',
                'owner_client_id'  => '11016',
                'public_id'        => $public_id,
                'domain'           => 'showmethemoney.ru',
                'caption'          => 'Test AdBlock',
                'mincpm'           => '50',
                'blocks_limit'     => 200,
                'blocks_count'     => 1,
                'is_deleted'       => '',
                'is_protected'     => '',
                'available_fields' => {
                    'actions'                    => 1,
                    'available_fields'           => 1,
                    'caption'                    => 1,
                    'cur_user_is_read_assistant' => 1,
                    'direct_block'               => 1,
                    'domain'                     => 1,
                    'editable_fields'            => 1,
                    'fields_depends'             => 1,
                    'id'                         => 1,
                    'internal_campaign_id'       => 1,
                    'is_custom_bk_data'          => 1,
                    'blocks_limit'               => 1,
                    'blocks_count'               => 1,
                    'is_deleted'                 => 1,
                    'is_page_deleted'            => 1,
                    'is_protected'               => 1,
                    'limit'                      => 1,
                    'mincpm'                     => 1,
                    'multistate'                 => 1,
                    'multistate_name'            => 1,
                    'owner_id'                   => 1,
                    'page'                       => 1,
                    'page_id'                    => 1,
                    'public_id'                  => 1,
                    'statistics_exists'          => 1,
                    'status'                     => 1,
                    'strategy'                   => 1,
                    'campaign'                   => 1,
                    'campaign_id'                => 1,
                    'is_campaign_deleted'        => 1,
                    'strategy_name'              => 1,
                },
                'id'              => $block_id,
                'limit'           => 5,
                'editable_fields' => {
                    'caption'      => 1,
                    'strategy'     => 1,
                    'mincpm'       => 1,
                    'direct_block' => 1,
                    'limit'        => 1,
                },
                'is_custom_bk_data'    => '0',
                'page_id'              => $page_id,
                'internal_campaign_id' => $id,
                'direct_block'         => 'vertical',
                'multistate'           => '20',                               # 4 - check_statistics, 16 - need_update
                'multistate_name'      => ignore(),
                'login'                => 'mocked-context-adblock-partner',
                'statistics_exists'    => '',
                'is_page_deleted'      => '',
                'owner_id'             => '1016',
                'cur_user_is_read_assistant' => '',
                'status'                     => 'pending',
                'show_video'                 => 0,
                'is_tutby'                   => undef,
                'campaign'                   => {
                    'is_tutby'   => undef,
                    'client_id'  => '11016',
                    'page_id'    => '153746',
                    'login'      => 'mocked-context-adblock-partner',
                    'multistate' => '2112',
                    'owner'      => {
                        'business_unit'    => '0',
                        'client_id'        => '11016',
                        'current_currency' => 'RUB',
                        'email'            => 'devnull@yandex.ru',
                        'features'         => ['turbo_desktop_available'],
                        'id'               => '1016',
                        'is_assessor'      => 0,
                        'lastname'         => '',
                        'login'            => 'mocked-context-adblock-partner',
                        'midname'          => '',
                        'name'             => ''
                    },
                    'domain'                     => 'showmethemoney.ru',
                    'caption'                    => 'Test campaign',
                    'is_protected'               => '',
                    'blocks_limit'               => 200,
                    'blocks_count'               => 1,
                    'id'                         => '153746',
                    'owner_id'                   => '1016',
                    'cur_user_is_read_assistant' => '',
                },
                'campaign_id'         => '153746',
                'is_campaign_deleted' => '',
                'strategy_name'       => 'Minimum CPM',
            },
            'check new adblock'
        );

        $app->context_on_site_adblock->do_action(
            $public_id, 'edit',
            caption  => 'Fix caption',
            strategy => 0,
            mincpm   => 10,
            limit    => 3,
        );

        cmp_deeply(
            $app->context_on_site_adblock->get($public_id, fields => [qw(caption strategy mincpm limit)]),
            {
                'strategy' => '0',
                'caption'  => 'Fix caption',
                'mincpm'   => 10,
                'limit'    => 3,
            },
            'check edit adblock'
        );

        $app->context_on_site_adblock->do_action($public_id, 'delete');

        is(
            $app->context_on_site_adblock->check_multistate_flag(
                $app->context_on_site_adblock->get($public_id, fields => ['multistate'])->{'multistate'}, 'deleted'
            ),
            TRUE,
            'adblock - deleted'
          );

        $app->context_on_site_adblock->do_action($public_id, 'restore');

        is(
            $app->context_on_site_adblock->check_multistate_flag(
                $app->context_on_site_adblock->get($public_id, fields => ['multistate'])->{'multistate'}, 'deleted'
            ),
            FALSE,
            'adblock - restored'
          );

        my $tmp_rights = $app->add_tmp_rights('do_context_on_site_register_in_balance');
        $app->context_on_site_campaign->do_action($id, 'register_in_balance');
        undef($tmp_rights);
        $app->context_on_site_campaign->do_action($id, 'start');

        cmp_deeply(
            {$app->context_on_site_adblock->get_bk_data($page_id)},
            {
                'rtb_blocks' => {
                    $block_id => {
                        'BlockModel' => 'context_on_site_adblock',
                        'DSPInfo'    => [
                            {
                                'DSPID' => '1',
                                'CPM'   => 10000
                            }
                        ],
                        'DirectLimit' => '3',
                        'AdTypeSet'   => {
                            'video-performance' => 0,
                            'video'             => 0,
                            'media'             => 0,
                            'text'              => 1,
                            'media-performance' => 0
                        },
                        'OptimizeType' => 0,
                        'DSPType'      => 1,
                        'AdBlockBlock' => 1,
                        'BlockCaption' => 'Fix caption',
                        'MultiState'   => 22,
                    }
                }
            },
            'check get_bk_data'
        );

        #login manager: mocked-yan-manager
        $app->set_cur_user({id => 1015});

        cmp_deeply(
            [grep {$_->{'page_id'} == $page_id} @{$app->context_on_site_adblock->get_all_campaigns_for_adding()}],
            [
                {
                    'domain'  => 'showmethemoney.ru',
                    'caption' => 'Test campaign',
                    'page_id' => $page_id
                }
            ],
            'Check get_all_campaigns_for_adding for manager'
        );

        my $got = $app->context_on_site_adblock->get($public_id, fields => ['*']);
        my $expected = get_test_data_and_update_if_needed('get.json', $got);

        cmp_deeply($got, $expected, 'adblock for manager');

    },
    user => 'mocked-context-adblock-partner',
    init => [qw(api_bk api_balance agreement_checker kv_store)],
);

sub mock_app {
    my ($app) = @_;

    mock_curdate('2014-03-01 00:00:00');

    mock_mailer($app);

    my $PAGE_ID =
      $app->partner_db->_get_all('SELECT MAX(page_id) as max_page_id FROM context_on_site_campaign')
      ->[0]{'max_page_id'};

    $app->{'api_bk'} = Test::MockObject::Extends->new($app->{'api_bk'});
    $app->api_bk->mock('create_or_update_campaign', sub {++$PAGE_ID});

    $app->{'api_balance'} = Test::MockObject::Extends->new($app->{'api_balance'});
    $app->api_balance->mock('create_or_update_place', sub {1});

    $app->{'agreement_checker'} = Test::MockObject::Extends->new($app->{'agreement_checker'});
    $app->agreement_checker->mock('has_agreement_for_any_product_for_today', sub {1});

    my $ID = $app->partner_db->_get_all('SELECT MAX(id) as max_id FROM context_on_site_campaign')->[0]{'max_id'};
    $app->{'kv_store'} = Test::MockObject::Extends->new($app->{'kv_store'});
    $app->kv_store->mock('get', sub {++$ID});
}
