#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs restore_subs mock_curdate);
use Test::Partner::Utils qw(get_clickhouse_statistics_test_data);

use Test::Deep;
use Test::Differences qw(eq_or_diff);

use qbit;

use PiConstants qw($DSP_DIRECT_ID $DSP_OWN_ADV_ID $DSP_UNSOLD_ID);

my $DATE = '2018-10-30';

run_tests(
    sub {
        my ($app) = @_;

        local $$ = 100500;

        mock_curdate('2019-02-27 15:32:12');

        mock_data($app);

        my @sql = qw();
        mock_subs(
            {
                'QBit::Application::Model::DB::_do' => sub {push(@sql, $_[1])}
            }
        );

        $app->statistics_mobile_mediation_block->update_statistics_by_rsya_monetizer(
            from    => $DATE,
            to      => $DATE,
            storage => 'clickhouse'
        );

        restore_subs(['QBit::Application::Model::DB::_do']);

        eq_or_diff(join("\n\n", @sql) . "\n", do {local $/; <DATA>}, 'Check sql');

        mock_subs(
            {
                'Application::Model::Statistics::MobileMediation::get_clickhouse_expressions' => sub {
                    my $ch_expressions = $Test::Partner2::Mock::original_subs->{
                        'Application::Model::Statistics::MobileMediation::get_clickhouse_expressions'}->(@_);

                    $ch_expressions->{'place_id'} = 'block_id';

                    return $ch_expressions;
                  }
            }
        );

        $app->statistics_mobile_mediation_block->update_statistics_by_rsya_monetizer(
            from    => $DATE,
            to      => $DATE,
            storage => 'clickhouse'
        );

        cmp_deeply(
            $app->clickhouse_db->statistics->get_all(
                fields => [
                    qw(dt page_id block_id monetizer_id hits impressions currency_id calculated_revenue calculated_revenue_original)
                ],
                filter   => {dt => $DATE, product_id => 'mobile_mediation_block'},
                order_by => [qw(block_id)]
            ),
            [
                {
                    'block_id'                    => 3,
                    'currency_id'                 => 2,
                    'calculated_revenue'          => 4527,
                    'calculated_revenue_original' => 4527,
                    'dt'                          => $DATE,
                    'hits'                        => '1320',
                    'impressions'                 => '564',
                    'monetizer_id'                => 0,
                    'page_id'                     => 654397
                },
                {
                    'block_id'                    => 5,
                    'currency_id'                 => 1,
                    'calculated_revenue'          => 1899,
                    'calculated_revenue_original' => 2847,
                    'dt'                          => $DATE,
                    'hits'                        => '440',
                    'impressions'                 => '1212',
                    'monetizer_id'                => 14,
                    'page_id'                     => 124224
                }
            ],
            'check stat for level "mobile_mediation_block"'
        );
    },
    init                 => [qw(statistics_mobile_mediation_block)],
    create_clickhouse_db => TRUE,
);

sub mock_data {
    my ($app) = @_;

    $app->clickhouse_db->statistics->add_multi(
        get_clickhouse_statistics_test_data(
            $app,
            pk_data => [
                {
                    dt           => $DATE,
                    page_id      => 124224,
                    block_id     => 5,
                    currency_id  => 1,
                    monetizer_id => 14,
                }
            ],
            accessors => [qw(statistics_mobile_mediation_block)]
        )
    );

    $app->clickhouse_db->statistics->add_multi(
        get_clickhouse_statistics_test_data(
            $app,
            pk_data => [
                map {{dt => $DATE, page_id => 654397, block_id => 3, dsp_id => $_,}}
                  ($DSP_DIRECT_ID, $DSP_OWN_ADV_ID, $DSP_UNSOLD_ID)
            ],
            accessors => [qw(statistics_mobile_app_rtb)]
        )
    );
}

__DATA__
CREATE TABLE IF NOT EXISTS `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212` (
    `dt` Date,
    `product_id` Enum8('context_on_site_adblock' = 1, 'context_on_site_direct' = 2, 'context_on_site_market' = 3, 'context_on_site_market_api' = 4, 'context_on_site_mcb' = 5, 'context_on_site_rtb' = 6, 'context_on_site_stripe' = 7, 'dsp' = 8, 'internal_context_on_site_direct' = 9, 'internal_context_on_site_rtb' = 10, 'internal_context_on_site_stripe' = 11, 'internal_mobile_app_rtb' = 12, 'internal_search_on_site_direct' = 13, 'internal_search_on_site_premium' = 14, 'mobile_app_rtb' = 15, 'search_on_site_direct' = 16, 'search_on_site_market' = 17, 'search_on_site_market_api' = 18, 'search_on_site_mcb' = 19, 'search_on_site_premium' = 20, 'ssp_context_on_site_rtb' = 21, 'ssp_mobile_app_rtb' = 22, 'ssp_video_an_site_rtb' = 23, 'video_an_site_fullscreen' = 24, 'video_an_site_inpage' = 25, 'video_an_site_instream' = 26, 'context_on_site_adfox' = 27, 'context_on_site_campaign' = 28, 'search_on_site_campaign' = 29, 'internal_context_on_site_campaign' = 30, 'internal_search_on_site_campaign' = 31, 'mobile_app_settings' = 32, 'video_an_site' = 33, 'ssp_context_on_site_campaign' = 34, 'ssp_mobile_app_settings' = 35, 'ssp_video_an_site' = 36, 'context_on_site_content' = 37, 'mobile_mediation_block' = 38, 'internal_context_on_site_content' = 39, 'indoor' = 40, 'indoor_block' = 41, 'outdoor' = 42, 'outdoor_block' = 43, 'context_on_site_natural' = 44, 'internal_context_on_site_natural' = 45),
    `page_id` UInt32,
    `block_id` UInt32,
    `dsp_id` UInt32,
    `currency_id` UInt16,
    `tag_id` UInt32,
    `geo_id` Int64,
    `category_id` UInt32,
    `monetizer_id` UInt32,
    `shows` Int64,
    `clicks` Int64,
    `hits` Int64,
    `impressions` Int64,
    `direct_clicks` Int64,
    `direct_shows` Int64,
    `all_w_nds` Int64,
    `all_wo_nds` Int64,
    `partner_w_nds` Int64,
    `partner_wo_nds` Int64,
    `bad_shows` Int64,
    `bad_hits` Int64,
    `bad_win_price_w_nds` Int64,
    `bad_win_price_wo_nds` Int64,
    `bad_win_partner_price_w_nds` Int64,
    `bad_win_partner_price_wo_nds` Int64,
    `win_max_positions_count` Int64,
    `an_fraud_shows` Int64,
    `an_fraud_clicks` Int64,
    `an_fraud_hits` Int64,
    `an_cover_hits` Int64,
    `an_cover_direct_hits` Int64,
    `an_cover_market_hits` Int64,
    `an_cover_mcb_hits` Int64,
    `an_cover_senthits` Int64,
    `an_rtb_cover_hits` Int64,
    `an_rtb_cover_senthits` Int64,
    `an_rtb_cover_direct_hits` Int64,
    `direct_page_ad_shows` Int64,
    `premium_page_ad_shows` Int64,
    `direct_hits` Int64,
    `market_hits` Int64,
    `mcb_hits` Int64,
    `premium_hits` Int64,
    `view` Int64,
    `open_player` Int64,
    `calculated_revenue` Int64,
    `calculated_revenue_original` Int64
) ENGINE = SummingMergeTree(`dt`, (`dt`, `product_id`, `page_id`, `block_id`, `dsp_id`, `currency_id`, `tag_id`, `category_id`, `monetizer_id`, `geo_id`), 8192);


INSERT INTO `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212` (`all_w_nds`, `all_wo_nds`, `an_cover_direct_hits`, `an_cover_hits`, `an_cover_market_hits`, `an_cover_mcb_hits`, `an_cover_senthits`, `an_fraud_clicks`, `an_fraud_hits`, `an_fraud_shows`, `an_rtb_cover_direct_hits`, `an_rtb_cover_hits`, `an_rtb_cover_senthits`, `bad_hits`, `bad_shows`, `bad_win_partner_price_w_nds`, `bad_win_partner_price_wo_nds`, `bad_win_price_w_nds`, `bad_win_price_wo_nds`, `block_id`, `calculated_revenue`, `calculated_revenue_original`, `category_id`, `clicks`, `currency_id`, `direct_clicks`, `direct_hits`, `direct_page_ad_shows`, `direct_shows`, `dsp_id`, `dt`, `geo_id`, `hits`, `impressions`, `market_hits`, `mcb_hits`, `monetizer_id`, `open_player`, `page_id`, `partner_w_nds`, `partner_wo_nds`, `premium_hits`, `premium_page_ad_shows`, `product_id`, `shows`, `tag_id`, `view`, `win_max_positions_count`) SELECT
    (-1 * SUM(`statistics`.`all_w_nds`)) AS `all_w_nds`,
    (-1 * SUM(`statistics`.`all_wo_nds`)) AS `all_wo_nds`,
    (-1 * SUM(`statistics`.`an_cover_direct_hits`)) AS `an_cover_direct_hits`,
    (-1 * SUM(`statistics`.`an_cover_hits`)) AS `an_cover_hits`,
    (-1 * SUM(`statistics`.`an_cover_market_hits`)) AS `an_cover_market_hits`,
    (-1 * SUM(`statistics`.`an_cover_mcb_hits`)) AS `an_cover_mcb_hits`,
    (-1 * SUM(`statistics`.`an_cover_senthits`)) AS `an_cover_senthits`,
    (-1 * SUM(`statistics`.`an_fraud_clicks`)) AS `an_fraud_clicks`,
    (-1 * SUM(`statistics`.`an_fraud_hits`)) AS `an_fraud_hits`,
    (-1 * SUM(`statistics`.`an_fraud_shows`)) AS `an_fraud_shows`,
    (-1 * SUM(`statistics`.`an_rtb_cover_direct_hits`)) AS `an_rtb_cover_direct_hits`,
    (-1 * SUM(`statistics`.`an_rtb_cover_hits`)) AS `an_rtb_cover_hits`,
    (-1 * SUM(`statistics`.`an_rtb_cover_senthits`)) AS `an_rtb_cover_senthits`,
    (-1 * SUM(`statistics`.`bad_hits`)) AS `bad_hits`,
    (-1 * SUM(`statistics`.`bad_shows`)) AS `bad_shows`,
    (-1 * SUM(`statistics`.`bad_win_partner_price_w_nds`)) AS `bad_win_partner_price_w_nds`,
    (-1 * SUM(`statistics`.`bad_win_partner_price_wo_nds`)) AS `bad_win_partner_price_wo_nds`,
    (-1 * SUM(`statistics`.`bad_win_price_w_nds`)) AS `bad_win_price_w_nds`,
    (-1 * SUM(`statistics`.`bad_win_price_wo_nds`)) AS `bad_win_price_wo_nds`,
    `statistics`.`block_id` AS `block_id`,
    (-1 * SUM(`statistics`.`calculated_revenue`)) AS `calculated_revenue`,
    (-1 * SUM(`statistics`.`calculated_revenue_original`)) AS `calculated_revenue_original`,
    `statistics`.`category_id` AS `category_id`,
    (-1 * SUM(`statistics`.`clicks`)) AS `clicks`,
    `statistics`.`currency_id` AS `currency_id`,
    (-1 * SUM(`statistics`.`direct_clicks`)) AS `direct_clicks`,
    (-1 * SUM(`statistics`.`direct_hits`)) AS `direct_hits`,
    (-1 * SUM(`statistics`.`direct_page_ad_shows`)) AS `direct_page_ad_shows`,
    (-1 * SUM(`statistics`.`direct_shows`)) AS `direct_shows`,
    `statistics`.`dsp_id` AS `dsp_id`,
    `statistics`.`dt` AS `dt`,
    `statistics`.`geo_id` AS `geo_id`,
    (-1 * SUM(`statistics`.`hits`)) AS `hits`,
    (-1 * SUM(`statistics`.`impressions`)) AS `impressions`,
    (-1 * SUM(`statistics`.`market_hits`)) AS `market_hits`,
    (-1 * SUM(`statistics`.`mcb_hits`)) AS `mcb_hits`,
    `statistics`.`monetizer_id` AS `monetizer_id`,
    (-1 * SUM(`statistics`.`open_player`)) AS `open_player`,
    `statistics`.`page_id` AS `page_id`,
    (-1 * SUM(`statistics`.`partner_w_nds`)) AS `partner_w_nds`,
    (-1 * SUM(`statistics`.`partner_wo_nds`)) AS `partner_wo_nds`,
    (-1 * SUM(`statistics`.`premium_hits`)) AS `premium_hits`,
    (-1 * SUM(`statistics`.`premium_page_ad_shows`)) AS `premium_page_ad_shows`,
    `statistics`.`product_id` AS `product_id`,
    (-1 * SUM(`statistics`.`shows`)) AS `shows`,
    `statistics`.`tag_id` AS `tag_id`,
    (-1 * SUM(`statistics`.`view`)) AS `view`,
    (-1 * SUM(`statistics`.`win_max_positions_count`)) AS `win_max_positions_count`
FROM `statistics`
WHERE (
    `statistics`.`product_id` = 'mobile_mediation_block'
    AND `statistics`.`dt` >= '2018-10-30'
    AND `statistics`.`dt` <= '2018-10-30'
    AND `statistics`.`monetizer_id` = 0
)
GROUP BY `dt`, `product_id`, `page_id`, `block_id`, `dsp_id`, `currency_id`, `tag_id`, `category_id`, `monetizer_id`, `geo_id`

INSERT INTO `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212` (`dt`, `product_id`, `page_id`, `block_id`, `monetizer_id`, `hits`, `impressions`, `calculated_revenue`, `calculated_revenue_original`, `currency_id`) SELECT
    `statistics`.`dt` AS `dt`,
    'mobile_mediation_block' AS `mediation_product_id`,
    `statistics`.`page_id` AS `page_id`,
    `statistics`.`block_id` AS `block_id`,
    toUInt32(0) AS `monetizer_id`,
    SUM(`statistics`.`hits`) AS `all_hits`,
    sumIf(`statistics`.`shows`, `statistics`.`dsp_id` NOT IN (5, 10)) AS `impressions`,
    SUM(`statistics`.`partner_wo_nds`) AS `calculated_revenue`,
    SUM(`statistics`.`partner_wo_nds`) AS `calculated_revenue_original`,
    toUInt16(2) AS `currency_id`
FROM `statistics`
WHERE (
    `statistics`.`dt` = '2018-10-30'
    AND `statistics`.`product_id` = 'mobile_app_rtb'
    AND dictGetUInt32('blocks_dict', 'place_id', tuple('mobile_app_rtb', toUInt32(`statistics`.`page_id`), toUInt32(`statistics`.`block_id`))) <> 0
)
GROUP BY `dt`, `page_id`, `block_id`, `mediation_product_id`, `monetizer_id`, `currency_id`

INSERT INTO `statistics` (`all_w_nds`, `all_wo_nds`, `an_cover_direct_hits`, `an_cover_hits`, `an_cover_market_hits`, `an_cover_mcb_hits`, `an_cover_senthits`, `an_fraud_clicks`, `an_fraud_hits`, `an_fraud_shows`, `an_rtb_cover_direct_hits`, `an_rtb_cover_hits`, `an_rtb_cover_senthits`, `bad_hits`, `bad_shows`, `bad_win_partner_price_w_nds`, `bad_win_partner_price_wo_nds`, `bad_win_price_w_nds`, `bad_win_price_wo_nds`, `block_id`, `calculated_revenue`, `calculated_revenue_original`, `category_id`, `clicks`, `currency_id`, `direct_clicks`, `direct_hits`, `direct_page_ad_shows`, `direct_shows`, `dsp_id`, `dt`, `geo_id`, `hits`, `impressions`, `market_hits`, `mcb_hits`, `monetizer_id`, `open_player`, `page_id`, `partner_w_nds`, `partner_wo_nds`, `premium_hits`, `premium_page_ad_shows`, `product_id`, `shows`, `tag_id`, `view`, `win_max_positions_count`) SELECT
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`all_w_nds`) AS `all_w_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`all_wo_nds`) AS `all_wo_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_cover_direct_hits`) AS `an_cover_direct_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_cover_hits`) AS `an_cover_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_cover_market_hits`) AS `an_cover_market_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_cover_mcb_hits`) AS `an_cover_mcb_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_cover_senthits`) AS `an_cover_senthits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_fraud_clicks`) AS `an_fraud_clicks`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_fraud_hits`) AS `an_fraud_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_fraud_shows`) AS `an_fraud_shows`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_rtb_cover_direct_hits`) AS `an_rtb_cover_direct_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_rtb_cover_hits`) AS `an_rtb_cover_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`an_rtb_cover_senthits`) AS `an_rtb_cover_senthits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_hits`) AS `bad_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_shows`) AS `bad_shows`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_win_partner_price_w_nds`) AS `bad_win_partner_price_w_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_win_partner_price_wo_nds`) AS `bad_win_partner_price_wo_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_win_price_w_nds`) AS `bad_win_price_w_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`bad_win_price_wo_nds`) AS `bad_win_price_wo_nds`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`block_id` AS `block_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`calculated_revenue`) AS `calculated_revenue`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`calculated_revenue_original`) AS `calculated_revenue_original`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`category_id` AS `category_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`clicks`) AS `clicks`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`currency_id` AS `currency_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`direct_clicks`) AS `direct_clicks`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`direct_hits`) AS `direct_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`direct_page_ad_shows`) AS `direct_page_ad_shows`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`direct_shows`) AS `direct_shows`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`dsp_id` AS `dsp_id`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`dt` AS `dt`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`geo_id` AS `geo_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`hits`) AS `hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`impressions`) AS `impressions`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`market_hits`) AS `market_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`mcb_hits`) AS `mcb_hits`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`monetizer_id` AS `monetizer_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`open_player`) AS `open_player`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`page_id` AS `page_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`partner_w_nds`) AS `partner_w_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`partner_wo_nds`) AS `partner_wo_nds`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`premium_hits`) AS `premium_hits`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`premium_page_ad_shows`) AS `premium_page_ad_shows`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`product_id` AS `product_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`shows`) AS `shows`,
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`tag_id` AS `tag_id`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`view`) AS `view`,
    SUM(`tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`win_max_positions_count`) AS `win_max_positions_count`
FROM `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`
WHERE (
    `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`product_id` = 'mobile_mediation_block'
    AND `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`dt` >= '2018-10-30'
    AND `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`.`dt` <= '2018-10-30'
)
GROUP BY `dt`, `product_id`, `page_id`, `block_id`, `dsp_id`, `currency_id`, `tag_id`, `category_id`, `monetizer_id`, `geo_id`

DROP TABLE `tmp_statistics_mobile_mediation_block_pid_100500_date_20190227_153212`
