#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use lib::abs;

use Data::Rmap qw(rmap_all);

use Test::Partner2::Simple;
use Test::More;
use Test::Differences qw(eq_or_diff);

use qbit;

$ENV{'LAZY_LOAD'} = FALSE;

my $FIELDS_COUNT = 700;

run_tests(
    sub {
        my ($app) = @_;

        my @levels = grep {blessed($app->{$_}) && $app->$_->isa('Application::Model::Statistics::Product')} keys(%$app);

        my %product_ids = ();
        foreach my $level (@levels) {
            $product_ids{$app->$level->product->accessor} = TRUE if $app->$level->can('product');
        }

        my @product_ids = sort keys(%product_ids);

        my $query =
          $app->clickhouse_db->query(without_table_alias => TRUE)->select(table => $app->clickhouse_db->statistics);

        no warnings 'once';

        my @fields = ();
        foreach my $field (sort {$a->{'id'} cmp $b->{'id'}} @Application::Model::Statistics::Product::FIELD_TYPES) {
            foreach (qw(shared forced)) {
                $field->{$_} += 0 if exists($field->{$_});
            }

            foreach my $field_name (qw( hint  short_title  title)) {
                if (exists($field->{$field_name})) {
                    # "title" -  как в интерефейсе (по-русски), остальное - ключи
                    if ($field_name eq 'title') {
                        $app->set_app_locale('ru');
                    } else {
                        QBit::GetText::set_locale(
                            project => $app->get_option('locale_domain', 'application'),
                            path    => $app->get_option('ApplicationPath') . 'locale',
                            lang    => 'en',
                        );
                    }

                    $field->{$field_name} = $field->{$field_name}->();
                    $field->{$field_name} =~ s/&nbsp;/ /g;
                }
            }

            if (exists($field->{'depends_on'}) && ref($field->{'get'}) eq 'CODE') {
                my @dep_fields = sort @{$field->{'depends_on'}};

                my $val = 0;
                my $data = {map {$val += 100; ($_ => $val)} @dep_fields};

                $field->{'get'} = $field->{'get'}->($data) + 0;

                $field->{'depends_on'} = \@dep_fields;
            }

            if (exists($field->{'clickhouse_expression'})) {
                if (ref($field->{'clickhouse_expression'}) eq 'CODE') {
                    $field->{'clickhouse_expression'} =
                      $field->{'clickhouse_expression'}->($field, \@product_ids, $query, money_format => TRUE);
                }

                unless (
                    in_array(
                        $field->{'id'},
                        [
                            qw(
                              all_real_price_w_nds
                              all_real_price_wo_nds
                              average_rate_all_real_price_w_nds
                              average_rate_all_real_price_wo_nds
                              bk_partner_price_w_nds
                              bk_partner_price_wo_nds
                              bk_price_w_nds
                              bk_price_wo_nds
                              charging_w_nds
                              charging_wo_nds
                              dsp_charging_w_nds
                              dsp_charging_wo_nds
                              dsp_cpm_charging_w_nds
                              dsp_cpm_charging_wo_nds
                              rs_all_w_nds
                              rs_all_wo_nds
                              )
                        ]
                    )
                  )
                {
                    #fields not exists in stsatistics
                    ($field->{'clickhouse_sql'}) =
                      $query->_field_to_sql(undef, $field->{'clickhouse_expression'}, $query->{'__TABLES__'}[0]);

                    $field->{'clickhouse_sql'} =~ s/\n\s*/ /g;
                }

                rmap_all {
                    rmap_all {$_ = ref($_) eq 'REF' || ref($_) eq 'SCALAR' ? $$_ : $_} $_;
                }
                $field->{'clickhouse_expression'};
            }

            push @fields, $field;

            # Usage: > ./prove *.t :: --self_update
            if (need_self_update()) {
                writefile(lib::abs::path("./fields/$field->{id}.json"), to_json($field, pretty => TRUE));
            }

            my $expected_field = from_json(readfile(lib::abs::path("./fields/$field->{id}.json")));

            eq_or_diff($field, $expected_field, $field->{'id'} // 'not exists', {context => 1});
        }

        ok(@fields == $FIELDS_COUNT, "fields count got:" . @fields . " expected:" . $FIELDS_COUNT);
    },
    dont_create_database => TRUE,
    do_not_die_on_fail   => 1,
);
