use Test::Partner2::Simple;

use Test::More;
use Test::Deep qw(cmp_deeply ignore re bag);
use Test::Partner2::Mock;
use Test::Partner::Utils qw(mock_curdate get_hash_values get_new_user restore_table add_bookmaker_filter);

use Capture::Tiny ':all';

use qbit;
use Utils::PublicID;

sub insert {
    my ($self, $table, $fields, @row_hashes) = @_;
    @$fields = keys %{$row_hashes[0]} unless $fields;

    $self->do(
        "INSERT INTO $table (" . join(q{, }, @$fields) . ')
           VALUES  '
          . join(q{, }, ('(' . join(q{, }, ('?') x @$fields) . ')') x @row_hashes) . '
        ;',
        {},
        map {(get_hash_values($_, $fields)),} @row_hashes
    );
}

run_tests(
    sub {
        my ($app) = @_;

        my $manager = get_new_user();

        mock_curdate('2013-11-04 23:59:59');
        mock_utils_partner2($app);
        mock_moneymap($app);

        $app->set_option(
            cur_user => {
                rights => {
                    do_internal_context_on_site_add                     => TRUE,
                    internal_context_on_site_stripe_view_all            => TRUE,
                    do_internal_context_on_site_stripe_add              => TRUE,
                    internal_context_on_site_stripe_view_field__comment => TRUE,
                    internal_context_on_site_campaign_view_all          => TRUE,
                    internal_context_on_site_campaign_edit_all          => TRUE,
                    internal_context_on_site_stripe_edit                => TRUE,
                },
                id => $manager->{'id'},
            }
        );

        my $dbh             = $app->partner_db->get_dbh;
        my $table_dumps_dir = $app->get_option('ApplicationPath') . 't_old_test_data/statistics/2014_03_01/table_dumps';
        restore_table($dbh, $table_dumps_dir, 'tns_dict_article');

        $dbh->do('DELETE FROM users_action_log');
        $dbh->do('DELETE FROM users');
        insert($dbh, 'users', [qw(id login client_id email)], $manager);
        insert($dbh, 'currencies', [qw(id code)], {id => 2, code => 'RUB'});

        $app->api_bk;
        $app->{api_bk} = Test::MockObject::Extends::Easy->new($app->{api_bk});

        $app->api_balance;
        $app->{api_balance} = Test::MockObject::Extends::Easy->new($app->{api_balance});
        $app->api_balance->set_always('create_or_update_campaign', TRUE);
        $app->api_balance->mock(
            'get_internal_pages_stat',
            sub {
                [
                    map {
                        {%$_, DT => "$_[2] 00:00:00",}
                      } @{from_json readfile $app->get_option('ApplicationPath')
                          . "t_old_test_data/application-model-product-internal-context-on-site-stripe.get_pages_stat.json"
                      }
                ];
            }
        );

        add_bookmaker_filter($app);

        my $campaign_domain = 'auto.yandex.ru';

        {
            my @public_ids = (
                add_campaign_with_stripe($app, 46784, $campaign_domain, 3),
                add_campaign_with_stripe($app, 5989,  $campaign_domain),
                add_campaign_with_stripe($app, 87385, $campaign_domain, 2),
                add_campaign_with_stripe($app, 88769, $campaign_domain, 4),
            );

            my @stripes = ();

            foreach my $public_id (@public_ids) {
                my $stripe = {};
                @$stripe{qw(campaign_id id)} = (split_block_public_id($public_id))[1, 2];
                push @stripes, $stripe;
            }

            my %shared_fields = (
                dt                => '2013-11-01',
                currency_id       => 2,
                stripe_shows      => re(qr([0-9]+)),
                stripe_hits       => re(qr([0-9]+)),
                stripe_clicks     => re(qr([0-9]+)),
                stripe_all_w_nds  => re(qr([0-9]+)),
                stripe_all_wo_nds => re(qr([0-9]+)),
                all_w_nds         => re(qr([0-9]+)),
                all_wo_nds        => re(qr([0-9]+)),
            );

            my $stderr = capture_stderr {
                cmp_deeply(
                    $app->statistics_internal_advnet_context_on_site_stripe->_get_stat_from_balance(
                        from => '2013-11-01',
                        to   => '2013-11-01'
                    ),
                    bag(
                        map {
                            {
                                %shared_fields,
                                  campaign_id => $_->{'campaign_id'},
                                  block_id    => $_->{'id'},
                            },
                          } @stripes
                       ),
                    'Get stat from balance: gotten statisitcs is correct.'
                );
            };

            $stderr .= capture_stderr {
                $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics(
                    for_month => TRUE,
                    storage   => 'clickhouse'
                );
                $app->statistics_internal_advnet_context_on_site_stripe->regular_update_statistics(
                    storage => 'clickhouse');
            };

            delete(@shared_fields{qw(dt all_w_nds all_wo_nds)});
            $shared_fields{'date'} = '2013-11-01';

            cmp_deeply(
                $app->statistics->get_statistics(
                    'dimension_fields' => ['date|day', 'currency_id',],
                    'dimension_filter' => undef,
                    'entity_fields'    => ['public_id'],
                    'fields' => [grep {$_ ne 'date' && $_ ne 'currency_id'} keys(%shared_fields)],
                    'levels' => [
                        {
                            'filter' => undef,
                            'id'     => 'internal_advnet_context_on_site_stripe',
                        },
                    ],
                    'order_by' => [{field => 'date', dir => 'desc'}],
                    'period' => ['2013-11-01', '2013-11-01'],
                    'total'  => 0,
                    'vat'    => -1,
                  )->{'data'},
                bag(
                    map {
                        {
                            %shared_fields,
                              public_id => $app->internal_context_on_site_stripe->public_id($_),
                        },
                      } @stripes
                   ),
                'Regular update: gotten statisitcs is correct.'
            );
            is(scalar(@{$app->clickhouse_db->statistics->get_all()}),
                4 * 4, 'Regular update: right statistics record number is gotten.');
            # is( $stderr, '', 'No warnings are gotten.' );
        }
    },
    locale               => "en",
    fill_databases       => 0,
    create_clickhouse_db => TRUE,
    mocks                => [qw(mock_check_statistics_by_blocks)]
);

sub add_campaign_with_stripe {
    my ($app, $campaign_id, $campaign_domain, $stripe_id) = @_;

    $app->api_utils_partner2->set_always('get_next_page_id', $campaign_id);
    $campaign_id = $app->internal_context_on_site_campaign->add(
        domain       => $campaign_domain,
        fast_context => 1
    );

    $app->partner_db->managers->add(
        {
            page_id    => $campaign_id,
            manager_id => $app->get_option('cur_user')->{'id'}
        }
    );

    if ($stripe_id--) {
        $app->partner_db->transaction(
            sub {
                $app->internal_context_on_site_campaign->get_next_block_id($campaign_id);
                1;
            }
        ) while ($stripe_id--);
    }
    return $app->internal_context_on_site_stripe->add(campaign_id => $campaign_id);
}
