use strict;
use utf8;
use warnings FATAL => 'all';
use feature qw(say state);

use Clone qw(clone);
use Test::More;
use Test::Deep;
use Test::Differences qw(eq_or_diff);

use Test::MockObject::Extends::Easy;
use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs);
use Test::Partner::Utils qw(get_test_data_and_update_if_needed);

our $MODE = 'standard';

my $FIXTURE = [
    {
        casename    => 'thismonth && pastyear period parsing test',
        mode        => 'empty',
        digest_data => {
            id => 'test_period_parsing',
            params =>
'{"chartLevel":null,"isLevelsDifferent":false,"tableLevel":[{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":"thismonth","total":0,"vat":-1},{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":"pastyear","total":0,"vat":-1}],"tableViewModelData":{"sortBy":"date","sortOrder":"desc"}}',
        },
        params => {
            sort_by    => 'date',
            sort_order => 'asc',
        },
    },
    {
        casename    => 'empty output for empty input',
        mode        => 'empty',
        digest_data => {
            id => 'single_period',
            params =>
'{"chartLevel":null,"isLevelsDifferent":false,"tableLevel":[{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":null,"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":["2018-03-01","2018-03-04"],"total":0,"vat":-1}],"tableViewModelData":{"sortBy":"date","sortOrder":"desc"}}'
        },
        params => {
            sort_by    => 'date',
            sort_order => 'asc',
        },
    },
    {
        casename    => 'sorted_stat_data',
        mode        => 'standard',
        digest_data => {id => 'single_period',},
        params      => {
            sort_by    => 'date',
            sort_order => 'asc',
        },
    },
    {
        casename    => 'compare stat periods with second period - missing',
        mode        => 'first_full_second_empty',
        digest_data => {
            id => 'cmp_periods_2018-2017',
            params =>
'{"chartLevel":null,"isLevelsDifferent":false,"tableLevel":[{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":["2018-03-01","2018-03-04"],"total":0,"vat":-1},{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":["2017-03-01","2017-03-04"],"total":0,"vat":-1}],"tableViewModelData":{"sortBy":"date","sortOrder":"desc"}}'
        },
        params => {
            sort_by    => 'adblock_all_wo_nds',
            sort_order => 'desc'
        },
    },
    {
        casename    => 'compare stat periods with first period - missing',
        mode        => 'first_empty_second_full',
        digest_data => {id => 'cmp_periods_2018-2017',},
        params      => {
            sort_by    => 'domain',
            sort_order => 'desc'
        },
    },
    {
        casename    => 'compare stat periods with sparse first and second periods',
        mode        => 'first_sparse_second_sparse',
        digest_data => {
            id => 'cmp_periods_2017-2018',
            params =>
'{"chartLevel":null,"isLevelsDifferent":false,"tableLevel":[{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":["2017-03-01","2017-03-04"],"total":0,"vat":-1},{"dimension_fields":["date|day","currency_id"],"dimension_filter":null,"entity_fields":["domain","page_id"],"fields":["adblock_all_wo_nds"],"levels":[{"filter":["AND",[["login","IN",["rambler-p"]]]],"id":"advnet_context_on_site_adblock"}],"order_by":[],"period":["2018-03-01","2018-03-04"],"total":0,"vat":-1}],"tableViewModelData":{"sortBy":"date","sortOrder":"desc"}}'
        },
        params => {
            sort_by    => 'page_id',
            sort_order => 'asc'
        },
    },
    ,
    {
        casename    => 'compare stat periods with sparse second and first periods',
        mode        => 'second_sparse_first_sparse',
        digest_data => {id => 'cmp_periods_2018-2017',},
        params      => {
            sort_by    => 'date',
            sort_order => 'asc'
        },
    },
];

my $MOCK_DATA = {
    currencies => [
        {
            'code' => 'EUR',
            'id'   => '3'
        },
        {
            'code' => 'RUB',
            'id'   => '2'
        },
        {
            'code' => 'USD',
            'id'   => '1'
        }
    ],
    data => [
        {
            'adblock_all_wo_nds' => 0.16,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'passion.ru',
            'page_id'            => '242962'
        },
        {
            'adblock_all_wo_nds' => 1154.27,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'afisha.rambler.ru',
            'page_id'            => '251152'
        },
        {
            'adblock_all_wo_nds' => 0.01,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'infox.sg',
            'page_id'            => '127611'
        },
        {
            'adblock_all_wo_nds' => 9448.99,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'mail.rambler.ru',
            'page_id'            => '229092'
        },
        {
            'adblock_all_wo_nds' => 74738.63,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'rambler.ru',
            'page_id'            => '206075'
        },
        {
            'adblock_all_wo_nds' => 967.23,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'quto.ru',
            'page_id'            => '242471'
        },
        {
            'adblock_all_wo_nds' => 1091.22,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'afisha.rambler.ru',
            'page_id'            => '251152'
        },
        {
            'adblock_all_wo_nds' => 0.05,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'wmj.ru',
            'page_id'            => '242963'
        },
        {
            'adblock_all_wo_nds' => 66542.21,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'lenta.ru',
            'page_id'            => '231411'
        },
        {
            'adblock_all_wo_nds' => 859.95,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'rns.online',
            'page_id'            => '242964'
        },
        {
            'adblock_all_wo_nds' => 31907.14,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'www.gazeta.ru',
            'page_id'            => '188274'
        },
        {
            'adblock_all_wo_nds' => 10845.55,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'weekend.rambler.ru',
            'page_id'            => '188273'
        },
        {
            'adblock_all_wo_nds' => 17403.32,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'www.gazeta.ru',
            'page_id'            => '188274'
        },
        {
            'adblock_all_wo_nds' => 1471.89,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'sport.rambler.ru',
            'page_id'            => '251149'
        },
        {
            'adblock_all_wo_nds' => 59352.48,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'rambler.ru',
            'page_id'            => '206075'
        },
        {
            'adblock_all_wo_nds' => 18504.55,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'championat.com',
            'page_id'            => '188733'
        },
        {
            'adblock_all_wo_nds' => 0.18,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'autorambler.ru',
            'page_id'            => '242965'
        },
        {
            'adblock_all_wo_nds' => 72120.16,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'news.rambler.ru',
            'page_id'            => '188734'
        },
        {
            'adblock_all_wo_nds' => 3288.57,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'weather.rambler.ru',
            'page_id'            => '188272'
        },
        {
            'adblock_all_wo_nds' => 1137.41,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'quto.ru',
            'page_id'            => '242471'
        },
        {
            'adblock_all_wo_nds' => 1448.11,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'afisha.rambler.ru',
            'page_id'            => '251152'
        },
        {
            'adblock_all_wo_nds' => 6995.49,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'weekend.rambler.ru',
            'page_id'            => '188273'
        },
        {
            'adblock_all_wo_nds' => 113777.09,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'news.rambler.ru',
            'page_id'            => '188734'
        },
        {
            'adblock_all_wo_nds' => 3932.22,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'weather.rambler.ru',
            'page_id'            => '188272'
        },
        {
            'adblock_all_wo_nds' => 922.99,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'quto.ru',
            'page_id'            => '242471'
        },
        {
            'adblock_all_wo_nds' => 0.17,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'passion.ru',
            'page_id'            => '242962'
        },
        {
            'adblock_all_wo_nds' => 5794.59,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'mail.rambler.ru',
            'page_id'            => '229092'
        },
        {
            'adblock_all_wo_nds' => 0.14,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'passion.ru',
            'page_id'            => '242962'
        },
        {
            'adblock_all_wo_nds' => 1034.56,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'quto.ru',
            'page_id'            => '242471'
        },
        {
            'adblock_all_wo_nds' => 33088.16,
            'currency_id'        => '2',
            'date'               => '2018-03-04',
            'domain'             => 'championat.com',
            'page_id'            => '188733'
        },
        {
            'adblock_all_wo_nds' => 720.75,
            'currency_id'        => '2',
            'date'               => '2018-03-02',
            'domain'             => 'auto.rambler.ru',
            'page_id'            => '251151'
        },
        {
            'adblock_all_wo_nds' => 0.00,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'autorambler.ru',
            'page_id'            => '242965'
        },
        {
            'adblock_all_wo_nds' => 401.88,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'rns.online',
            'page_id'            => '242964'
        },
        {
            'adblock_all_wo_nds' => 78275.52,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'news.rambler.ru',
            'page_id'            => '188734'
        },
        {
            'adblock_all_wo_nds' => 2343.32,
            'currency_id'        => '2',
            'date'               => '2018-03-03',
            'domain'             => 'sport.rambler.ru',
            'page_id'            => '251149'
        },
        {
            'adblock_all_wo_nds' => 7216.13,
            'currency_id'        => '2',
            'date'               => '2018-03-01',
            'domain'             => 'sport.rambler.ru',
            'page_id'            => '251149'
        },
    ]
};

sub fix_numbers {
    my ($data) = @_;
    foreach my $row (@$data) {
        map {$row->{$_} += 0}
          grep {defined $row->{$_} && $row->{$_} ne '-'} qw(adblock_all_wo_nds);
    }
}

sub filter_test_data {
    my ($mode, $data) = @_;

    $mode //= $MODE;

    fix_numbers($data);

    state $runs = 0;
    $runs %= 2;

    my $get_odds = sub {
        my $n = 1;
        [grep {$n++ % 2} @$data];
    };
    my $get_evens = sub {
        my $n = 0;
        [grep {$n++ % 2} @$data];
    };
    my $change_year = sub {
        [map {$_->{'date'} =~ s/2018/2017/; $_} @$data];
    };

    if ($mode eq 'empty') {
        return [];
    } elsif ($mode eq 'first_full_second_empty') {
        if ($runs++ == 0) {
            return $data;
        } else {
            return [];
        }
    } elsif ($mode eq 'first_empty_second_full') {
        if ($runs++ == 0) {
            return [];
        } else {
            return $change_year->($data);
        }
    } elsif ($mode eq 'first_sparse_second_sparse') {
        if ($runs++ == 0) {
            return $change_year->($get_odds->($data));
        } else {
            return $get_evens->($data);
        }
    } elsif ($mode eq 'second_sparse_first_sparse') {
        if ($runs++ == 0) {
            return $get_odds->($data);
        } else {
            return $change_year->($get_evens->($data));
        }
    } else {
        return $data;
    }
}

run_tests(
    sub {
        my ($app) = @_;

        change_cur_user($app, 'yndx-developer');
        $app->set_option('use_coro' => 0);

        mock_subs({'Application::Model::Statistics::Product::is_available' => sub {1},});

        my $tree = $app->statistics->get_tree();
        $app->statistics;
        $app->{'statistics'} = Test::MockObject::Extends::Easy->new($app->{'statistics'});
        $app->statistics->mock('get_tree', sub {$tree});

        my $got_data = [];
        foreach my $fixture (@$FIXTURE) {
            my ($case_name, $digest_data, $mode, $params) = @$fixture{qw( casename  digest_data  mode  params)};

            $app->partner_db->stat_report_params_digest->add($digest_data) if $digest_data->{params};

            $app->statistics->mock(
                'get_statistics',
                sub {
                    my $_data = clone($MOCK_DATA);
                    $_data->{data} = filter_test_data($mode, $_data->{data});
                    return $_data;
                }
            );

            my $mol_data = $app->statistics->get_data_from_hash(
                report_hash => $digest_data->{id},
                %$params
            );

            fix_numbers($mol_data->{sorted_stat_data});

            push @$got_data,
              {
                casename => $case_name,
                mol_data => $mol_data,
              };
        }

        my $expected = get_test_data_and_update_if_needed('data.json', $got_data);
        eq_or_diff($got_data, $expected, 'check data', {context => 3});

    },
    init => [qw(statistics_advnet_context_on_site_adblock)],
);
