use strict;
use warnings FATAL => 'all';

use feature qw(say state);

use Test::More;
use Test::Deep qw(cmp_deeply);
use Test::Exception;
use Test::MockObject::Extends::Easy;

use qbit;

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs);

our $MODE = 'standard';

sub filter_test_data {
    state $runs = 0;
    $runs %= 2;

    my $change_year = sub {
        [map {$_->{'date'} =~ s/2018/2017/; $_} @{$_[0]}];
    };

    if ($MODE eq 'empty') {
        return [];
    } elsif ($MODE eq 'first_full_second_empty') {
        if ($runs++ == 0) {
            return $_[0];
        } else {
            return [];
        }
    } elsif ($MODE eq 'first_empty_second_full') {
        if ($runs++ == 0) {
            return [];
        } else {
            return $change_year->($_[0]);
        }
    } elsif ($MODE eq 'first_full_second_full') {
        if ($runs++ == 0) {
            return $_[0];
        } else {
            return $change_year->($_[0]);
        }
    } elsif ($MODE eq 'first_full_second_full_group_by_week') {
        if ($runs++ == 0) {
            return [map {$_->{'date'} = '2018-02-26'; $_} @{$_[0]}];
        } else {
            return [map {$_->{'date'} = '2017-02-27'; $_} @{$_[0]}];
        }
    } elsif ($MODE eq 'first_full_second_full_group_by_week_mysql_format') {
        if ($runs++ == 0) {
            return [map {$_->{'date'} = '2018-W09'; $_} @{$_[0]}];
        } else {
            return [map {$_->{'date'} = '2017-W09'; $_} @{$_[0]}];
        }
    } else {
        return $_[0];
    }
}

run_tests(
    sub {
        my ($app) = @_;
        my $yndx_developer_user_id = 1022;
        $app->{'__CURRENT_USER_RIGHTS__'} = {};
        $app->set_cur_user({id => $yndx_developer_user_id});
        $app->set_option('use_coro' => 0);

        throws_ok(
            sub {
                $app->statistics->get_statistics_cmp_periods(period => ['2017-07-06', '2017-07-06']);
            },
            'Exception::Validation::BadArguments',
            "single period"
        );

        throws_ok(
            sub {
                $app->statistics->get_statistics_cmp_periods(period => [['2017-07-07', '2017-07-06'], []]);
            },
            'Exception::Validation::BadArguments',
            "end before start"
        );

        throws_ok(
            sub {
                $app->statistics->get_statistics_cmp_periods(
                    period => [['2017-07-06', '2017-07-06'], ['2018-07-06', '2018-07-07']]);
            },
            'Exception::Validation::BadArguments',
            "different lengths"
        );

        throws_ok(
            sub {
                $app->statistics->get_statistics_cmp_periods(period => [['2017-07-06', '2017-07-06'], ['', '']]);
            },
            'Exception::Validation::BadArguments',
            'Wrong date format'
        );

        mock_subs({'Application::Model::Statistics::Product::is_available' => sub {1},});

        my $tree = $app->statistics->get_tree();
        $app->statistics;
        $app->{'statistics'} = Test::MockObject::Extends::Easy->new($app->{'statistics'});
        $app->statistics->mock('get_tree', sub {$tree});

        my $stat_request = {
            'dimension_fields' => ['date|day', 'currency_id',],
            'dimension_filter' => undef,
            'entity_fields'    => ['domain',   'page_id',],
            'fields'           => ['adblock_all_wo_nds',],
            'levels'           => [
                {
                    'filter' => ['AND', [['login', 'IN', ['rambler-p',],],],],
                    'id' => 'advnet_context_on_site_adblock',
                },
            ],
            'order_by' => [{field => 'date', dir => 'desc'}],
            'period' => [['2018-03-01', '2018-03-04',], ['2017-03-01', '2017-03-04',]],
            'total'  => 0,
            'vat'    => -1,
        };

        $app->statistics->mock(
            'get_statistics',
            sub {
                return {
                    'currencies' => [
                        {
                            'code' => 'EUR',
                            'id'   => '3'
                        },
                        {
                            'code' => 'RUB',
                            'id'   => '2'
                        },
                        {
                            'code' => 'USD',
                            'id'   => '1'
                        }
                    ],
                    'data' => filter_test_data(
                        [
                            {
                                'adblock_all_wo_nds' => 0.16,
                                'currency_id'        => '2',
                                'date'               => '2018-03-03',
                                'domain'             => 'passion.ru',
                                'page_id'            => '242962'
                            },
                            {
                                'adblock_all_wo_nds' => 1154.27,
                                'currency_id'        => '2',
                                'date'               => '2018-03-02',
                                'domain'             => 'afisha.rambler.ru',
                                'page_id'            => '251152'
                            },
                            {
                                'adblock_all_wo_nds' => 0.05,
                                'currency_id'        => '2',
                                'date'               => '2018-03-04',
                                'domain'             => 'wmj.ru',
                                'page_id'            => '242963'
                            },
                            {
                                'adblock_all_wo_nds' => 31907.14,
                                'currency_id'        => '2',
                                'date'               => '2018-03-01',
                                'domain'             => 'www.gazeta.ru',
                                'page_id'            => '188274'
                            },
                        ]
                    ),
                };
            }
        );

        $MODE = 'empty';
        my $data = $app->statistics->get_statistics_cmp_periods(%$stat_request);
        cmp_deeply($data->{'data'}, [[], []], "empty result");

        $MODE = 'first_full_second_empty';
        $data = $app->statistics->get_statistics_cmp_periods(%$stat_request);
        cmp_deeply(
            $data->{'data'},
            [
                [
                    {
                        'adblock_all_wo_nds' => 0.05,
                        'currency_id'        => 2,
                        'date'               => '2018-03-04',
                        'domain'             => 'wmj.ru',
                        'page_id'            => 242963,
                    },
                    {
                        'adblock_all_wo_nds' => 0.16,
                        'currency_id'        => 2,
                        'date'               => '2018-03-03',
                        'domain'             => 'passion.ru',
                        'page_id'            => 242962,
                    },
                    {
                        'adblock_all_wo_nds' => 1154.27,
                        'currency_id'        => 2,
                        'date'               => '2018-03-02',
                        'domain'             => 'afisha.rambler.ru',
                        'page_id'            => 251152,
                    },
                    {
                        'adblock_all_wo_nds' => 31907.14,
                        'currency_id'        => 2,
                        'date'               => '2018-03-01',
                        'domain'             => 'www.gazeta.ru',
                        'page_id'            => 188274,
                    },
                ],
                [
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2017-03-04',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2017-03-03',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2017-03-02',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2017-03-01',
                        'domain'             => '-',
                        'page_id'            => '-'
                    }
                ],
            ],
            $MODE
        );

        $MODE = 'first_empty_second_full';
        $data = $app->statistics->get_statistics_cmp_periods(%$stat_request);
        cmp_deeply(
            $data->{'data'},
            [
                [
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2018-03-04',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2018-03-03',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2018-03-02',
                        'domain'             => '-',
                        'page_id'            => '-'
                    },
                    {
                        'adblock_all_wo_nds' => '-',
                        'currency_id'        => '-',
                        'date'               => '2018-03-01',
                        'domain'             => '-',
                        'page_id'            => '-'
                    }
                ],
                [
                    {
                        'adblock_all_wo_nds' => 0.05,
                        'currency_id'        => 2,
                        'date'               => '2017-03-04',
                        'domain'             => 'wmj.ru',
                        'page_id'            => 242963,
                    },
                    {
                        'adblock_all_wo_nds' => 0.16,
                        'currency_id'        => 2,
                        'date'               => '2017-03-03',
                        'domain'             => 'passion.ru',
                        'page_id'            => 242962,
                    },
                    {
                        'adblock_all_wo_nds' => 1154.27,
                        'currency_id'        => 2,
                        'date'               => '2017-03-02',
                        'domain'             => 'afisha.rambler.ru',
                        'page_id'            => 251152,
                    },
                    {
                        'adblock_all_wo_nds' => 31907.14,
                        'currency_id'        => 2,
                        'date'               => '2017-03-01',
                        'domain'             => 'www.gazeta.ru',
                        'page_id'            => 188274,
                    },
                ]
            ],
            $MODE
        );

        $MODE = 'first_full_second_full';
        $data = $app->statistics->get_statistics_cmp_periods(%$stat_request);
        cmp_deeply(
            $data->{'data'},
            [
                [
                    {
                        'adblock_all_wo_nds' => 0.05,
                        'currency_id'        => 2,
                        'date'               => '2018-03-04',
                        'domain'             => 'wmj.ru',
                        'page_id'            => 242963,
                    },
                    {
                        'adblock_all_wo_nds' => 0.16,
                        'currency_id'        => 2,
                        'date'               => '2018-03-03',
                        'domain'             => 'passion.ru',
                        'page_id'            => 242962,
                    },
                    {
                        'adblock_all_wo_nds' => 1154.27,
                        'currency_id'        => 2,
                        'date'               => '2018-03-02',
                        'domain'             => 'afisha.rambler.ru',
                        'page_id'            => 251152,
                    },
                    {
                        'adblock_all_wo_nds' => 31907.14,
                        'currency_id'        => 2,
                        'date'               => '2018-03-01',
                        'domain'             => 'www.gazeta.ru',
                        'page_id'            => 188274,
                    },
                ],
                [
                    {
                        'adblock_all_wo_nds' => 0.05,
                        'currency_id'        => 2,
                        'date'               => '2017-03-04',
                        'domain'             => 'wmj.ru',
                        'page_id'            => 242963,
                    },
                    {
                        'adblock_all_wo_nds' => 0.16,
                        'currency_id'        => 2,
                        'date'               => '2017-03-03',
                        'domain'             => 'passion.ru',
                        'page_id'            => 242962,
                    },
                    {
                        'adblock_all_wo_nds' => 1154.27,
                        'currency_id'        => 2,
                        'date'               => '2017-03-02',
                        'domain'             => 'afisha.rambler.ru',
                        'page_id'            => 251152,
                    },
                    {
                        'adblock_all_wo_nds' => 31907.14,
                        'currency_id'        => 2,
                        'date'               => '2017-03-01',
                        'domain'             => 'www.gazeta.ru',
                        'page_id'            => 188274,
                    },
                ]
            ],
            $MODE
        );

        $MODE = 'first_full_second_full_group_by_week';
        $data =
          $app->statistics->get_statistics_cmp_periods(%$stat_request,
            'dimension_fields' => ['date|week', 'currency_id',]);

        my $expected_data = [
            [
                {
                    'adblock_all_wo_nds' => '31907.14',
                    'currency_id'        => '2',
                    'date'               => '2018-02-26',
                    'domain'             => 'www.gazeta.ru',
                    'page_id'            => '188274'
                },
                {
                    'adblock_all_wo_nds' => '0.05',
                    'currency_id'        => '2',
                    'date'               => '2018-02-26',
                    'domain'             => 'wmj.ru',
                    'page_id'            => '242963'
                },
                {
                    'adblock_all_wo_nds' => '0.16',
                    'currency_id'        => '2',
                    'date'               => '2018-02-26',
                    'domain'             => 'passion.ru',
                    'page_id'            => '242962'
                },
                {
                    'adblock_all_wo_nds' => '1154.27',
                    'currency_id'        => '2',
                    'date'               => '2018-02-26',
                    'domain'             => 'afisha.rambler.ru',
                    'page_id'            => '251152'
                }
            ],
            [
                {
                    'adblock_all_wo_nds' => '31907.14',
                    'currency_id'        => '2',
                    'date'               => '2017-02-27',
                    'domain'             => 'www.gazeta.ru',
                    'page_id'            => '188274'
                },
                {
                    'adblock_all_wo_nds' => '0.05',
                    'currency_id'        => '2',
                    'date'               => '2017-02-27',
                    'domain'             => 'wmj.ru',
                    'page_id'            => '242963'
                },
                {
                    'adblock_all_wo_nds' => '0.16',
                    'currency_id'        => '2',
                    'date'               => '2017-02-27',
                    'domain'             => 'passion.ru',
                    'page_id'            => '242962'
                },
                {
                    'adblock_all_wo_nds' => '1154.27',
                    'currency_id'        => '2',
                    'date'               => '2017-02-27',
                    'domain'             => 'afisha.rambler.ru',
                    'page_id'            => '251152'
                }
            ]
        ];

        cmp_deeply($data->{'data'}, $expected_data, $MODE);

        $MODE = 'first_full_second_full_group_by_week_mysql_format';
        $data =
          $app->statistics->get_statistics_cmp_periods(%$stat_request,
            'dimension_fields' => ['date|week', 'currency_id',]);

        cmp_deeply($data->{'data'}, $expected_data, $MODE);
    },
    init => [qw(statistics_advnet_context_on_site_adblock)],
);
