use strict;
use warnings FATAL => 'all';

use qbit;

use Test::Partner2::Simple;
use Test::Partner2::Fixture;
use Test::Partner::Utils qw(get_clickhouse_statistics_test_data);
use Test::Partner2::Mock qw(mock_subs);
use Test::More;
use Test::Differences qw(eq_or_diff);

run_tests(
    sub {
        my ($app) = @_;

        my $block = get_fixture('partner_instream_video_block');
        my $user  = get_fixture('mocked_video_partner');

        my $public_id = $app->video_an_site_instream->public_id($block);

        change_cur_user($app, $user->{'login'});
        my $tmp_rights = $app->add_tmp_rights('statistics_from_ch');

        mock_ch_table(
            $app,
            {
                page_id     => $block->{'page_id'},
                block_id    => $block->{'id'},
                dsp_id      => 1,
                currency_id => 2,
                tag_id      => 0,
            }
        );

        mock_subs(
            {
                'Application::Model::Statistics::get_query_for_clickhouse' => sub {
                    my $query =
                      $Test::Partner2::Mock::original_subs->{'Application::Model::Statistics::get_query_for_clickhouse'}
                      ->(@_);

                    my ($sql) = $query->get_sql_with_data();

                    eq_or_diff($sql . "\n", do {local $/; <DATA>}, 'query corrected');

                    return $query;
                  }
            }
        );

        my $data = $app->statistics->get_statistics(
            period => ['2017-08-16', '2017-12-31'],
            fields => [
                'video_an_site_sum_partner_wo_nds', 'partner_wo_nds',
                'video_an_site_all_hits',           'instream_block_partner_wo_nds',
                'instream_block_hits'
            ],
            dimension_fields => ['date|wo', 'currency_id'],
            dimension_filter => undef,
            entity_fields    => [],
            vat              => -1,
            levels           => [
                {
                    'filter' => undef,
                    'id'     => 'video_an_site'
                }
            ],
            order_by => [{'field' => 'partner_wo_nds', 'dir' => 'asc'}],
        );

        # sorting here bc ordering is not implemented atm
        my $fn_sort_by_key_date = sub {$a->{'date'} cmp $b->{'date'}};
        $data->{'data'} = [sort $fn_sort_by_key_date @{$data->{'data'}}];

        eq_or_diff(
            $data,
            {
                'currencies' => [],
                'data'       => [
                    sort $fn_sort_by_key_date (
                        {
                            'currency_id'                      => 2,
                            'date'                             => '2017-08-16',
                            'instream_block_hits'              => 440,
                            'partner_wo_nds'                   => '3420.34',
                            'video_an_site_all_hits'           => 440,
                            'video_an_site_sum_partner_wo_nds' => '3420.34'
                        },
                        {
                            'currency_id'                      => 2,
                            'date'                             => '2017-08-17',
                            'instream_block_hits'              => 440,
                            'partner_wo_nds'                   => '3420.34',
                            'video_an_site_all_hits'           => 440,
                            'video_an_site_sum_partner_wo_nds' => '3420.34'
                        },
                        {
                            'currency_id'                      => 2,
                            'date'                             => '2017-08-19',
                            'instream_block_hits'              => 440,
                            'partner_wo_nds'                   => '3420.34',
                            'video_an_site_all_hits'           => 440,
                            'video_an_site_sum_partner_wo_nds' => '3420.34'
                        }
                    )
                ],
                'measures' => {
                    'instream_block_hits' => {
                        'index' => 4,
                        'title' => 'instream_block_hits__title',
                        'type'  => 'number',
                        'unit'  => 'count'
                    },
                    'partner_wo_nds' => {
                        'index' => 2,
                        'title' => 'partner__title_manager (w/o&nbsp;VAT)',
                        'type'  => 'money',
                        'unit'  => 'money'
                    },
                    'video_an_site_all_hits' => {
                        'index' => 3,
                        'title' => 'video_an_site_all_hits__title',
                        'type'  => 'number',
                        'unit'  => 'count'
                    },
                    'video_an_site_sum_partner_wo_nds' => {
                        'index' => 1,
                        'title' => 'video_an_site_sum_partner__title (w/o&nbsp;VAT)',
                        'type'  => 'money',
                        'unit'  => 'money'
                    }
                },
                'report_title' => 'Report for period 16.08.2017 - 31.12.2017',
                'total'        => {
                    '2' => {
                        'instream_block_hits'              => 1320,
                        'partner_wo_nds'                   => '10261.02',
                        'video_an_site_all_hits'           => 1320,
                        'video_an_site_sum_partner_wo_nds' => '10261.02'
                    }
                }
            },
            'get_statistics() for partner from CH with total',
            {context => 5}
        );
    },
    create_clickhouse_db => TRUE,
    fixtures             => [qw(partner_instream_video_block)],
    fill_databases       => FALSE,
    init                 => [qw(statistics)],
);

sub mock_ch_table {
    my ($app, $pk) = @_;

    my $data = get_clickhouse_statistics_test_data(
        $app,
        accessors => [qw(statistics_video_an_site_instream)],
        pk_data   => [map {{dt => $_, partner_wo_nds => 34203400045208, %$pk}} qw(2017-08-16 2017-08-17 2017-08-19)]
    );

    $app->clickhouse_db->statistics->add_multi($data);
}

__DATA__
SELECT
    `_currency_id` AS `currency_id`,
    `_date` AS `date`,
    `_instream_block_hits` AS `instream_block_hits`,
    `_partner_wo_nds` AS `partner_wo_nds`,
    `_video_an_site_all_hits` AS `video_an_site_all_hits`,
    `_video_an_site_sum_partner_wo_nds` AS `video_an_site_sum_partner_wo_nds`
FROM (
    SELECT
        `currency_id` AS `_currency_id`,
        `dt` AS `_date`,
        sumIf(`hits`, (
        `dsp_id` NOT IN (5, 10)
        AND `product_id` = 'video_an_site_instream'
    )) AS `_instream_block_hits`,
        round((SUM((`partner_wo_nds` / 1000)) / 10000000), 2) AS `_partner_wo_nds`,
        (sumIf(`hits`, (
            `dsp_id` NOT IN (5, 10)
            AND `product_id` = 'video_an_site_instream'
        )) + sumIf(`hits`, (
            `dsp_id` NOT IN (5, 10)
            AND 0
        )) + sumIf(`hits`, (
            `dsp_id` NOT IN (5, 10)
            AND 0
        ))) AS `_video_an_site_all_hits`,
        round(((sumIf((`partner_wo_nds` / 1000), `product_id` = 'video_an_site_instream') + sumIf((`partner_wo_nds` / 1000), 0) + sumIf((`partner_wo_nds` / 1000), 0)) / 10000000), 2) AS `_video_an_site_sum_partner_wo_nds`
    FROM `statistics`
    WHERE (
        `dt` >= '2017-08-16'
        AND `dt` <= '2017-12-31'
        AND `product_id` IN ('video_an_site', 'video_an_site_instream')
        AND `page_id` IN (1)
    )
    GROUP BY `_currency_id`, `_date`
) `statistics`
WHERE (
    (if(isNaN(`video_an_site_all_hits`), 0, `video_an_site_all_hits`) + if(isNaN(`instream_block_hits`), 0, `instream_block_hits`) + if(isNaN(`partner_wo_nds`), 0, `partner_wo_nds`) + if(isNaN(`video_an_site_sum_partner_wo_nds`), 0, `video_an_site_sum_partner_wo_nds`)) > 0
)
ORDER BY `partner_wo_nds` FORMAT JSONCompact
