#!/usr/bin/perl

use lib::abs;

use Test::Partner2::Simple;
use Test::MockObject::Extends;
use Test::Partner2::Mock qw(mock_subs);
use Test::Partner2::Utils qw($SKIP_MODELS);

use Test::More;
use Test::Differences qw(eq_or_diff);

use qbit;

my $IGNORE = {map {$_ => TRUE} qw(charging dsp)};

my %ONLY_BLOCK_FIELDS = ();

run_tests(
    sub {
        my ($app) = @_;

        %ONLY_BLOCK_FIELDS = map {$_ => TRUE} $app->statistics->get_fields_for_block_levels_only();

        mock_app($app);

        my $tree = $app->statistics->get_tree();

        my @levels = _get_levels($tree);

        my $path = lib::abs::path('./get_statistics_from_clickhouse');
        foreach my $level (grep {!$IGNORE->{$_->{'id'}}} @levels) {
            my $level_accessor = 'statistics_' . $level->{'id'};
            if ($app->$level_accessor->can('product')) {
                my $model_name = $app->$level_accessor->product->accessor;
                next if $SKIP_MODELS->{$model_name};
            }

            my @requests = _get_requests_for_level($app, $level);

            my @sql = ();
            foreach my $request (@requests) {
                my $query = $app->statistics->get_statistics(
                    %$request,
                    storage    => 'clickhouse',
                    only_query => TRUE,
                );

                push(@sql, $query->get_sql_with_data);
            }

            if (need_self_update()) {
                writefile("$path/$level->{'id'}.sql", join("\n\n", @sql) . "\n");
            }

            eq_or_diff(join("\n\n", @sql) . "\n", readfile("$path/$level->{'id'}.sql"), $level->{'id'}, {context => 1});
        }
    },
    init => [qw(context_on_site_rtb)],
);

sub mock_app {
    my ($app) = @_;

    $app->api_balance;
    $app->{'api_balance'} = Test::MockObject::Extends->new($app->{'api_balance'});

    $app->{'api_balance'}->mock(
        'get_partner_contracts',
        sub {
            my ($self, $param, $value) = @_;

            return [{Person => {client_id => $value}}];
        }
    );

    mock_subs(
        {
            'QBit::Application::Model::DB::Query::get_all' => sub {
                my ($self) = @_;

                return _get_query_data($self);
            },
            'QBit::Application::Model::Multistate::get_multistates_by_filter' => sub {
                return ['mock_multistate'];
            },
            'Application::Model::Statistics::_get_value_by_expression_type' => sub {
                $_[1];
            },
        }
    );
}

sub _get_query_data {
    my ($query) = @_;

    my $table = $query->{'__TABLES__'}[0];

    my $name   = $table->{'table'}->name;
    my @fields = keys(%{$table->{'fields'}});

    my $filter_expression = $table->{'filter'}->expression();
    _ref_to_scalar($filter_expression);
    my $filter = to_json($filter_expression);

    return [{map {$_ => "${_}__from_table__${name}__with_filter__$filter"} @fields}];
}

sub _ref_to_scalar {
    if (blessed($_[0])) {
        my $data = _get_query_data($_[0]);

        $_[0] = [values(%{$data->[0]})];
    } elsif (ref($_[0]) eq 'HASH') {
        map {_ref_to_scalar($_)} values(%{$_[0]});
    } elsif (ref($_[0]) eq 'ARRAY') {
        map {_ref_to_scalar($_)} @{$_[0]};
    } elsif (ref($_[0]) eq 'SCALAR' || ref($_[0]) eq 'REF') {
        $_[0] = ${$_[0]};
    }
}

sub _get_levels {
    my ($tree) = @_;

    my @res = ();
    foreach my $level (@$tree) {
        my @children = ();
        if ($level->{'children'}) {
            @children = _get_levels(delete($level->{'children'}));
        }

        push(@res, $level, @children);
    }

    return @res;
}

sub _get_requests_for_level {
    my ($app, $level) = @_;

    my $stat_accessor = "statistics_$level->{'id'}";
    my $stat_level    = $app->$stat_accessor;
    my $product       = $stat_level->get_product;

    my $conflicts = {};
    my @res       = ();
    if (exists($level->{'conflict_fields'}) && @{$level->{'conflict_fields'}} > 0) {
        # есть конфликтные поля
        foreach my $conflict (@{$level->{'conflict_fields'}}) {
            $conflicts->{$conflict->[0]}{$conflict->[1]} = TRUE;
            $conflicts->{$conflict->[1]}{$conflict->[0]} = TRUE;
        }

        foreach my $field_name (sort keys(%$conflicts)) {
            push(@res, _get_request($level, $stat_level, $product, $conflicts, $field_name));
        }
    } else {
        push(@res, _get_request($level, $stat_level, $product, $conflicts));
    }

    if ($stat_level->can('children') && $stat_level->children) {
        # Добавляем запрос без группировки по блочным полям
        push(
            @res,
            {
                period           => ['2017-08-16',                '2017-12-31'],
                fields           => _get_fields($level,           \%ONLY_BLOCK_FIELDS),
                dimension_fields => _get_dimension_fields($level, \%ONLY_BLOCK_FIELDS),
                dimension_filter => _get_dimension_filter($level, \%ONLY_BLOCK_FIELDS),
                entity_fields    => _get_entity_fields($level,    \%ONLY_BLOCK_FIELDS),
                total            => 0,
                vat              => -1,
                levels           => [
                    {
                        'filter' => _get_level_filter($level, \%ONLY_BLOCK_FIELDS),
                        'id'     => $level->{'id'}
                    }
                ],
            }
        );
    }

    return @res;
}

sub _get_request {
    my ($level, $stat_level, $product, $conflicts, $field_name) = @_;

    my $ignore_fields =
      %$conflicts ? {%{$conflicts->{$field_name}}, map {$_ => TRUE} grep {$_ ne $field_name} %$conflicts} : {};

    my $dimension_fields = _get_dimension_fields($level, $ignore_fields);
    my $entity_fields = _get_entity_fields($level, $ignore_fields);

    # для некоторых уровней приходит conflict но таких полей на этом уровне нет
    return () if %$conflicts && !grep {$field_name eq $_} @$dimension_fields, @$entity_fields;

    if (!$product->isa('Application::Model::Block') && grep {$ONLY_BLOCK_FIELDS{$_}} @$dimension_fields,
        @$entity_fields)
    {
        # убираем поля уровня пейдж если группировка по блочным полям
        my $page_fields = _get_page_fields($stat_level);

        $ignore_fields->{$_} = TRUE foreach keys(%$page_fields);
    }

    return {
        period           => ['2017-08-16',                '2017-12-31'],
        fields           => _get_fields($level,           $ignore_fields),
        dimension_fields => $dimension_fields,
        dimension_filter => _get_dimension_filter($level, $ignore_fields),
        entity_fields    => $entity_fields,
        total            => 0,
        vat              => -1,
        levels           => [
            {
                'filter' => _get_level_filter($level, $ignore_fields),
                'id'     => $level->{'id'}
            }
        ],
    };
}

sub _get_page_fields {
    my ($stat_level) = @_;

    if ($stat_level->get_product->isa('Application::Model::Page')) {
        return $stat_level->fields();
    }

    my $res = {};
    foreach ($stat_level->children) {
        $res = {%$res, %{_get_page_fields($_)}};
    }

    return $res;
}

sub _get_fields {
    my ($level, $ignore_fields) = @_;

    return [map {$_->{'id'}} grep {!$ignore_fields->{$_->{'id'}}} @{$level->{'fields'}}];
}

sub _get_dimension_fields {
    my ($level, $ignore_fields) = @_;

    my @res = ();
    foreach my $field (@{$level->{'dimension_fields'}}) {
        next if $ignore_fields->{$field->{'id'}};

        if ($field->{'id'} eq 'date') {
            push(@res, 'date|day');
        } else {
            push(@res, $field->{'id'});
        }
    }

    return \@res;
}

sub _get_dimension_filter {
    my ($level, $ignore_fields) = @_;

    my @res = ();
    foreach my $field (@{$level->{'dimension_fields'}}) {
        next if $ignore_fields->{$field->{'id'}} || $field->{'only_group'} || $field->{'id'} eq 'date';

        if ($field->{'type'} eq 'boolean') {
            push(@res, [$field->{'id'}, '=', JSON::XS::true]);
        } else {
            if ($field->{'filter_values'}) {
                push(@res, [$field->{'id'}, 'IN', $field->{'filter_values'}[0]{'id'}]);
            } else {
                push(@res, [$field->{'id'}, '=', "mock__$field->{'id'}"]);
            }
        }
    }

    return ['AND' => \@res];
}

sub _get_entity_fields {
    my ($level, $ignore_fields) = @_;

    return [map {$_->{'id'}} grep {!$ignore_fields->{$_->{'id'}}} @{$level->{'entity_fields'}}];
}

sub _get_level_filter {
    my ($level, $ignore_fields) = @_;

    my ($entity_filter_fields, $entity_filter_simple_fields) =
      @$level{qw(entity_filter_fields entity_filter_simple_fields)};

    if (@{$entity_filter_simple_fields} == 2 && ref($entity_filter_simple_fields->[0]) eq 'ARRAY') {
        $entity_filter_simple_fields = [map {@$_} @$entity_filter_simple_fields];
    }

    my @filter = ();

    foreach my $simple_field (@$entity_filter_simple_fields) {
        my @names = split(/\./, $simple_field->{'name'});

        my $sub_filter;
        if ($ignore_fields->{$names[0]}) {
            next;
        } elsif ($entity_filter_fields->{$names[0]}) {
            my ($type, $value) = _get_type($entity_filter_fields, clone(\@names));

            @names = reverse(@names);
            my $name = shift(@names);

            if ($type eq 'boolean') {
                # to check boolean filter on false/0
                $sub_filter = [$name, '=', ($level->{id} ne 'mobile_app_rtb' ? JSON::XS::true : JSON::XS::false)];
            } elsif ($type eq 'multistate') {
                $sub_filter = [$name, '=', $value];
            } elsif ($type eq 'publicid') {
                #TODO: устанавливать корректный public_id
                # для тестрования можно для уровня context_on_site_rtb
                # подставить тут 'R-A-1-2'
                $sub_filter = [$name, '=', 1];
            } elsif ($type eq 'dictionary') {
                $sub_filter = [$name, '=', $value];
            } else {
                $sub_filter = [$name, '=', "mock__$simple_field->{'name'}"];
            }
        } else {
            throw sprintf('unknown simple field: %s', $simple_field->{'name'});
        }

        foreach (@names) {
            $sub_filter = [$_, 'MATCH', $sub_filter];
        }

        push(@filter, $sub_filter);
    }

    return ['AND', \@filter];
}

sub _get_type {
    my ($tree, $path) = @_;

    my $field = shift(@$path);

    my $data = $tree->{$field};

    if (@$path) {
        return _get_type($data->{'subfields'}, $path);
    } else {
        my $data_values = $data->{'values'} // {};

        my $value;
        if (ref($data_values) eq 'HASH') {
            #"blocked" ignore
            $value = [sort grep {$_ ne 'blocked'} keys(%$data_values)]->[0];
        } elsif (ref($data_values) eq 'ARRAY') {
            $value = [sort map {$_->{'id'}} @$data_values];
        } else {
            throw 'Can not find value';
        }

        return ($data->{'type'}, $value);
    }
}
