#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use Test::Partner2::Simple;

use Test::Deep;
use Test::Differences qw(eq_or_diff);
use Test::Partner2::Mock qw(mock_curdate mock_subs);
use Test::Partner::Utils qw(get_clickhouse_statistics_test_data);

use qbit;

use Application::Model::Statistics::_Utils::Query;

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate('2019-03-20 18:11:23');

        mock_stat($app);

        my @queries = ();
        mock_subs(
            {
                'QBit::Application::Model::DB::_get_all' => sub {
                    push(@queries, $_[1]);

                    return $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::_get_all'}->(@_);
                  }
            }
        );

        my $data = $app->statistics->get_statistics(
            "levels"           => [{"id"  => "advnet_context_on_site_rtb", "filter" => undef}],
            "fields"           => ["rtb_all_wo_nds"],
            "entity_fields"    => ["campaign_id"],
            "dimension_fields" => ["date|day"],
            "storage"          => "clickhouse",
            "period"           => "yesterday",
            "top_keys"         => 2,
            "order_by"         => [{field => "rtb_all_wo_nds",             "dir"    => "asc"}]
        );

        eq_or_diff(
            $queries[0], qq[SELECT
    `_rtb_all_wo_nds` AS `rtb_all_wo_nds`,
    `_top_key` AS `top_key`
FROM (
    SELECT
        round((sumIf((`all_wo_nds` / 1000), `product_id` = 'context_on_site_rtb') / 10000000), 2) AS `_rtb_all_wo_nds`,
        cityHash64(`currency_id`, `page_id`) AS `_top_key`
    FROM `statistics`
    WHERE (
        `dt` >= '2019-03-19'
        AND `dt` <= '2019-03-19'
        AND `product_id` IN (\'context_on_site_rtb\')
    )
    GROUP BY `_top_key`
) `statistics`
WHERE (
    (if(isNaN(`rtb_all_wo_nds`), 0, `rtb_all_wo_nds`)) > 0
)
ORDER BY `rtb_all_wo_nds` DESC
LIMIT 0, 2 FORMAT JSONCompact], 'get top keys', {context => 1}
        );

        eq_or_diff(
            $queries[1], qq[SELECT
    `_campaign_id` AS `campaign_id`,
    `_currency_id` AS `currency_id`,
    `_date` AS `date`,
    `_rtb_all_wo_nds` AS `rtb_all_wo_nds`
FROM (
    SELECT
        `page_id` AS `_campaign_id`,
        `currency_id` AS `_currency_id`,
        `dt` AS `_date`,
        round((sumIf((`all_wo_nds` / 1000), `product_id` = 'context_on_site_rtb') / 10000000), 2) AS `_rtb_all_wo_nds`
    FROM `statistics`
    WHERE (
        `dt` >= '2019-03-19'
        AND `dt` <= '2019-03-19'
        AND `product_id` IN ('context_on_site_rtb')
        AND cityHash64(`currency_id`, `page_id`) IN (4616339735982655829, 5211490184136037803)
    )
    GROUP BY `_campaign_id`, `_currency_id`, `_date`
) `statistics`
WHERE (
    (if(isNaN(`rtb_all_wo_nds`), 0, `rtb_all_wo_nds`)) > 0
) FORMAT JSONCompact], 'get data', {context => 1}
        );

        eq_or_diff(
            $queries[2], qq[SELECT
    `_currency_id` AS `currency_id`,
    `_rtb_all_wo_nds` AS `rtb_all_wo_nds`
FROM (
    SELECT
        `currency_id` AS `_currency_id`,
        round((sumIf((`all_wo_nds` / 1000), `product_id` = 'context_on_site_rtb') / 10000000), 2) AS `_rtb_all_wo_nds`
    FROM `statistics`
    WHERE (
        `dt` >= '2019-03-19'
        AND `dt` <= '2019-03-19'
        AND `product_id` IN ('context_on_site_rtb')
    )
    GROUP BY `_currency_id`
) `statistics`
WHERE (
    (if(isNaN(`rtb_all_wo_nds`), 0, `rtb_all_wo_nds`)) > 0
) FORMAT JSONCompact], 'get total', {context => 1}
        );

        cmp_deeply(
            $data,
            {
                'currencies' => [
                    {
                        'code' => 'EUR',
                        'id'   => '3'
                    },
                    {
                        'code' => 'RUB',
                        'id'   => '2'
                    },
                    {
                        'code' => 'USD',
                        'id'   => '1'
                    }
                ],
                'data' => bag(
                    {
                        'campaign_id'    => 2755,
                        'currency_id'    => 0,
                        'date'           => '2019-03-19',
                        'rtb_all_wo_nds' => '842734922.35'
                    },
                    {
                        'campaign_id'    => 56393,
                        'currency_id'    => 0,
                        'date'           => '2019-03-19',
                        'rtb_all_wo_nds' => '63493402.46'
                    }
                ),
                'measures' => {
                    'rtb_all_wo_nds' => {
                        'index' => 1,
                        'title' => 'Turnover for RTB-blocks (w/o&nbsp;VAT)',
                        'type'  => 'money',
                        'unit'  => 'money'
                    }
                },
                'report_title' => 'Report for period 19.03.2019 - 19.03.2019',
                'total'        => {'0' => {'rtb_all_wo_nds' => '907026893.27'}}
            },
            'data'
        );
    },
    create_clickhouse_db => TRUE,
);

sub mock_stat {
    my ($app) = @_;

    my $data = get_clickhouse_statistics_test_data(
        $app,
        pk_data => [
            {dt => '2019-03-19', page_id => 2345,   all_wo_nds => 4532342342342342},
            {dt => '2019-03-19', page_id => 496124, all_wo_nds => 3453342342342342},
            {dt => '2019-03-19', page_id => 56393,  all_wo_nds => 634934024563546456},
            {dt => '2019-03-19', page_id => 2755,   all_wo_nds => 8427349223493534543},
        ],
        accessors => [qw(statistics_advnet_context_on_site_rtb)]
    );

    $app->clickhouse_db->statistics->add_multi($data);
}
