#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use qbit;
use PiConstants qw($SERVICE_NAME_IN_MODERATION);

use Test::Differences;
use Test::More;    # tests => 5;

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs mock_curdate restore_subs);
use Test::Partner::Utils qw(get_test_data_and_update_if_needed);
use Test::Partner2::Fixture;

my $block_name = 'partner_indoor_video_block_with_photos';

my $mock_date    = '2018-07-03';
my @dates_before = (map {sprintf($mock_date . 'T11:%02d:00', $_ * 5)} 0 .. 3);
my $date_last    = $mock_date . 'T12:00:00';
my @dates_after  = (map {sprintf($mock_date . 'T12:%02d:00', $_ * 5)} 1 .. 3);
my @dates        = (@dates_before, $date_last, @dates_after);

my %test_data;
my $cnt = 1000;

sub add_data {
    my (@data) = @_;
    my $msg = pop @data;
    $test_data{$cnt . '_' . $msg} = \@data;
    $cnt++;
}

sub check_data {
    my ($name) = @_;
    my $has_data = get_test_data_and_update_if_needed("$name.json", \%test_data);
    eq_or_diff(from_json(to_json(\%test_data)), $has_data, $name);

    $cnt       = 1000;
    %test_data = ();
}

sub block_data {
    my ($app, $id) = @_;
    return $app->indoor_block->get($id, fields => ['moderation', 'multistate', 'multistate_name']);
}

my $avatars = [
    {
        id      => 10001,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f018'
    }
];

my $verdict_table = [];

sub get_verdict_row {
    my ($field, $id, $verdict) = @_;
    return {
        data => [{"caption" => "Mocked Indoor Block_new",},],
        meta => {
            page_id    => 1,
            imp_id     => 1,
            model      => 'indoor_block',
            request_id => $id,
        },
        type        => 'indoor_block_' . $field,
        service     => $SERVICE_NAME_IN_MODERATION,
        mod_results => {
            verdict => ($verdict ? 'Yes' : 'No'),
            reasons => [],
        },
    };
}

sub get_request_id {
    my ($moderation_data, $field_name) = @_;

    foreach my $r (values %{$moderation_data->{moderation}{$field_name}}) {
        return $r->{request_id} unless $r->{verdict};
    }

    return undef;
}

my $hires = 0;

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($mock_date . ' 12:00:00');
        mock_subs(
            {
                'Time::HiRes::time' => sub() {
                    return time() + $hires++ / 1000;
                },
                'Application::Model::MDSAvatars::get_all' => sub {
                    return $avatars,;
                },
                'Cron::Methods::Moderation::INFO' => sub {
                    add_data(@_, 'Moderation::INFO');
                },
                'Cron::Methods::Moderation::WARN' => sub {
                    add_data(@_, 'Moderation::WARN');
                },
                'Application::Model::Block::Dooh::WARN' => sub {
                    add_data(@_, 'Dooh::WARN');
                },
                'Application::Model::Role::Has::FieldsModeration::WARN' => sub {
                    add_data(@_, 'FieldsModeration::WARN');
                },
                'Application::Model::API::Yandex::SelfService::logbroker' => sub {
                    my ($app, @in) = @_;
                    add_data(@in, 'logbroker in');
                    my $out =
                      $Test::Partner2::Mock::original_subs->{'Application::Model::API::Yandex::SelfService::logbroker'}
                      ->(@_);
                    add_data($out, 'logbroker out');
                    return $out;
                },
                'QBit::Application::Model::DB::mysql::Table::edit' => sub {
                    my ($self, $id, @in) = @_;
                    my $lid = $id;
                    if (ref $id eq 'HASH') {
                        $lid = [@{$id}{qw(page_id id)}];
                    }
                    add_data($lid, @in, $self->name . ' edit in');
                    my $out =
                      $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::mysql::Table::edit'}
                      ->($self, $id, @in);
                    add_data($out, 'edit out');
                    return $out;
                },
                'QBit::Application::Model::API::Yandex::YT::read_table_from_any_replica' => sub {
                    return join("\n", map {to_json $_} @$verdict_table);
                },
                'QBit::Application::Model::API::Yandex::YT::list_tables_from_any_replica' => sub {
                    return to_json([shift @dates]);
                },
            }
        );

        my $blocks       = get_fixtures_hash($block_name);
        my $indoor_block = $blocks->{$block_name};

        my $block_data = $app->indoor_block->get($indoor_block, fields => ['*']);
        add_data($block_data, 'init block');
        add_data(block_data($app, $indoor_block), 'init');
        check_data('init');

        add_data('repeat photo');
        $app->indoor_block->do_action($indoor_block, 'edit', photo_id_list => [10001]);
        add_data(block_data($app, $indoor_block), 'repeat photo');

        add_data('change photo & caption');
        $avatars->[0]{id} = 10002;
        $app->indoor_block->do_action($indoor_block, 'edit', photo_id_list => [10002]);
        add_data(block_data($app, $indoor_block), 'edit photo');
        check_data('edit');

        $app->do('moderation', 'check_need_approve');
        my $moderation_data = block_data($app, $indoor_block);
        add_data($moderation_data, 'edit photo');
        check_data('check_need_approve');

        $verdict_table = [get_verdict_row(image => get_request_id($moderation_data, 'photo_id_list') => TRUE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $indoor_block), 'get verdict photo');
        check_data('check_verdicts');

        add_data('reject photo');
        $avatars->[0]{id} = 10003;
        $app->indoor_block->do_action($indoor_block, 'edit', photo_id_list => [10003]);
        add_data(block_data($app, $indoor_block), 'change photo_id_list to 10003');

        $app->do('moderation', 'check_need_approve');
        $moderation_data = block_data($app, $indoor_block);
        add_data($moderation_data, 'change photo_id_list to 10003');
        check_data('check_need_approve_rejecting');

        $verdict_table = [get_verdict_row(image => get_request_id($moderation_data, 'photo_id_list') => FALSE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $indoor_block), 'get verdict photo rejected');
        check_data('check_verdicts_rejecting');

    },
    application_package => 'Cron',
    fill_databases      => 0,
    fixtures            => [$block_name, 'user_cron'],
    init                => [qw(api_selfservice)],
);
