#!/usr/bin/perl -w

use strict;
use warnings FATAL => 'all';

use qbit;
use PiConstants qw($SERVICE_NAME_IN_MODERATION);

use Test::Differences;
use Test::More;    # tests => 5;

use Test::Partner2::Simple;
use Test::Partner2::Mock qw(mock_subs mock_curdate restore_subs);
use Test::Partner::Utils qw(get_test_data_and_update_if_needed);
use Test::Partner2::Fixture;
use Test::Exception;

my $block_name = 'partner_outdoor_video_block_with_photos';

my $mock_date    = '2018-07-03';
my @dates_before = (map {sprintf($mock_date . 'T11:%02d:00', $_ * 5)} 0 .. 3);
my $date_last    = $mock_date . 'T12:00:00';
my @dates_after  = (map {sprintf($mock_date . 'T12:%02d:00', $_ * 5)} 1 .. 3);
my @dates        = (@dates_before, $date_last, @dates_after);

my %test_data;
my $cnt = 1000;

sub add_data {
    my (@data) = @_;
    my $msg = pop @data;
    $test_data{$cnt . '_' . $msg} = \@data;
    $cnt++;
}

sub check_data {
    my ($name) = @_;
    my $has_data = get_test_data_and_update_if_needed("$name.json", \%test_data);
    eq_or_diff(from_json(to_json(\%test_data)), $has_data, $name);

    $cnt       = 1000;
    %test_data = ();
}

sub block_data {
    my ($app, $id) = @_;
    return $app->outdoor_block->get($id, fields => [qw(moderation multistate multistate_name waiting_moderation)]);
}

my $avatars = [
    {
        id      => 10001,
        'links' => {
            'orig' => {
                'height' => 853,
                'path'   => '/get-partner/1574764/2a0000016b46c75181f829bd03e3f33ff3fe/orig',
                'width'  => 1280
            }
        },
        'md5' => '912e231b0c3d100479d28ef17eb7f018'
    }
];

my $verdict_table = [];

sub get_verdict_row {
    my ($field, $id, $verdict) = @_;
    return {
        data => [{"caption" => "Mocked Outdoor Block_new",},],
        meta => {
            page_id    => 1,
            imp_id     => 1,
            model      => 'outdoor_block',
            request_id => $id,
        },
        type        => 'outdoor_block_' . $field,
        service     => $SERVICE_NAME_IN_MODERATION,
        mod_results => {
            verdict => ($verdict ? 'Yes' : 'No'),
            reasons => [],
        },
    };
}

sub get_request_id {
    my ($moderation_data, $field_name) = @_;

    foreach my $r (values %{$moderation_data->{moderation}{$field_name}}) {
        return $r->{request_id} unless $r->{verdict};
    }

    return undef;
}

my $hires = 0;

run_tests(
    sub {
        my ($app) = @_;

        mock_curdate($mock_date . ' 12:00:00');
        mock_subs(
            {
                'Time::HiRes::time' => sub() {
                    return time() + $hires++ / 1000;
                },
                'Application::Model::MDSAvatars::get_all' => sub {
                    return $avatars,;
                },
                'Cron::Methods::Moderation::INFO' => sub {
                    add_data(@_, 'Moderation::INFO');
                },
                'Cron::Methods::Moderation::WARN' => sub {
                    add_data(@_, 'Moderation::WARN');
                },
                'Application::Model::Block::Dooh::WARN' => sub {
                    add_data(@_, 'Dooh::WARN');
                },
                'Application::Model::Role::Has::FieldsModeration::WARN' => sub {
                    add_data(@_, 'FieldsModeration::WARN');
                },
                'Application::Model::API::Yandex::SelfService::logbroker' => sub {
                    my ($app, @in) = @_;
                    add_data(@in, 'logbroker in');
                    my $out =
                      $Test::Partner2::Mock::original_subs->{'Application::Model::API::Yandex::SelfService::logbroker'}
                      ->(@_);
                    add_data($out, 'logbroker out');
                    return $out;
                },
                'QBit::Application::Model::DB::mysql::Table::edit' => sub {
                    my ($self, $id, @in) = @_;
                    my $lid = $id;
                    if (ref $id eq 'HASH') {
                        $lid = [@{$id}{qw(page_id id)}];
                    }
                    add_data($lid, @in, $self->name . ' edit in');
                    my $out =
                      $Test::Partner2::Mock::original_subs->{'QBit::Application::Model::DB::mysql::Table::edit'}
                      ->($self, $id, @in);
                    add_data($out, 'edit out');
                    return $out;
                },
                'QBit::Application::Model::API::Yandex::YT::read_table_from_any_replica' => sub {
                    return join("\n", map {to_json $_} @$verdict_table);
                },
                'QBit::Application::Model::API::Yandex::YT::list_tables_from_any_replica' => sub {
                    return to_json([shift @dates]);
                },
            }
        );

        my $blocks        = get_fixtures_hash($block_name);
        my $outdoor_block = $blocks->{$block_name};

        my $block_data = $app->outdoor_block->get($outdoor_block, fields => ['*']);
        add_data($block_data, 'init block');
        add_data(block_data($app, $outdoor_block), 'init');
        check_data('init');

        add_data('repeat caption');
        $app->outdoor_block->do_action($outdoor_block, 'edit', caption => $block_data->{caption});
        add_data(block_data($app, $outdoor_block), 'repeat caption');

        add_data('repeat photo');
        $app->outdoor_block->do_action($outdoor_block, 'edit', photo_id_list => [10001]);
        add_data(block_data($app, $outdoor_block), 'repeat photo');

        add_data('repeat gps');
        $app->outdoor_block->do_action($outdoor_block, 'edit', gps => $block_data->{gps});
        add_data(block_data($app, $outdoor_block), 'repeat gps');

        add_data('repeat caption');
        $app->outdoor_block->do_action($outdoor_block, 'edit', address => $block_data->{address});
        add_data(block_data($app, $outdoor_block), 'repeat address');

        add_data('change photo & caption & gps & adress');
        $avatars->[0]{id} = 10002;
        $app->outdoor_block->do_action(
            $outdoor_block, 'edit',
            caption       => $block_data->{caption} . '_new',
            address       => $block_data->{address} . '_new',
            gps           => "-81,124",
            photo_id_list => [10002]
        );
        add_data(block_data($app, $outdoor_block), 'edit caption & photo & gps & adress');
        check_data('edit');

        $app->do('moderation', 'check_need_approve');
        my $moderation_data = block_data($app, $outdoor_block);
        add_data($moderation_data, 'changed all');
        check_data('check_need_approve');

        $verdict_table = [get_verdict_row(caption => get_request_id($moderation_data, 'caption') => TRUE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict caption');

        $verdict_table = [get_verdict_row(image => get_request_id($moderation_data, 'photo_id_list') => TRUE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict photo');

        $verdict_table = [get_verdict_row(address => get_request_id($moderation_data, 'gps') => TRUE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict gps');

        $verdict_table = [get_verdict_row(address => get_request_id($moderation_data, 'address') => TRUE)];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict address');

        check_data('check_verdicts');

        add_data('reject caption');
        $app->outdoor_block->do_action($outdoor_block, 'edit', caption => 'FAIL caption', address => 'FAIL address');
        add_data(block_data($app, $outdoor_block), 'change caption & adress to FAIL');

        $app->do('moderation', 'check_need_approve');
        $moderation_data = block_data($app, $outdoor_block);
        add_data($moderation_data, 'change caption & address to FAIL');
        check_data('check_need_approve_rejecting');

        $verdict_table = [
            get_verdict_row(caption => get_request_id($moderation_data, 'caption') => FALSE),
            get_verdict_row(address => get_request_id($moderation_data, 'address') => FALSE),
        ];
        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict caption & address rejected');

        check_data('check_verdicts_rejecting');

        add_data('approve caption');
        $app->outdoor_block->do_action($outdoor_block, 'edit', caption => 'NEW caption', address => 'NEW address');
        add_data(block_data($app, $outdoor_block), 'change caption & adress to NEW');
        $app->do('moderation', 'check_need_approve');
        $moderation_data = block_data($app, $outdoor_block);
        add_data($moderation_data, 'change caption & address to NEW');

        $verdict_table = [
            get_verdict_row(caption => get_request_id($moderation_data, 'caption') => TRUE),
            get_verdict_row(address => get_request_id($moderation_data, 'address') => TRUE),
        ];

        $app->do('moderation', 'check_verdicts');
        add_data(block_data($app, $outdoor_block), 'get verdict caption & address approve');

        check_data('check_verdicts_approving');
    },
    application_package => 'Cron',
    fill_databases      => 0,
    fixtures            => [$block_name, 'user_cron'],
    init                => [qw(api_selfservice)],
);
