package Exception::Partner::AlterDB::Table;
use base qw(Exception);

package Partner::AlterDB::Table;
use base qw(QBit::Class);

use strict;
use warnings FATAL => 'all';

use qbit;
use Partner::AlterDB::Table::Comparator;

sub init {
    my ($self) = @_;

    throw Exception::Partner::AlterDB::Table 'Table option must be gotten'
      unless $self->{'table'};

    $self->SUPER::init();
    $self->_set_sql_alter_query('');
    $self->_set_last_command('');
    $self->_cleanup_used_strings();

    return $self;
}

sub get_sql_alter_query {
    my ($self, $description_copy, $description_etalon) = @_;

    return $self->_get_sql_alter_query()
      if $description_copy eq $description_etalon;

    my $comparator = Partner::AlterDB::Table::Comparator->new(etalon => $description_etalon, copy => $description_copy);

    foreach my $string (sort $comparator->get_strings_to_drop()) {
        next if $self->_string_is_used($string);

        if (my ($key, $fk_columns) = $string =~ /^\s*(KEY `\S+`) (\(.+\))/) {
            my ($fk_string, $fk_name) =
              $self->_parse_not_used_string(qr{^(\s*CONSTRAINT (`\S+`) FOREIGN KEY \Q$fk_columns\E.*)$},
                $comparator->get_copy_strings());
            $self->_drop($string, "FOREIGN KEY $fk_name")
              if ($fk_name);

            $self->_drop($string, $key);
        } elsif (my ($foreign_key) = $string =~ /^\s*CONSTRAINT (`\S+`) FOREIGN KEY (\(.+\))/) {
            $self->_drop($string, "FOREIGN KEY $foreign_key");
        } elsif (my ($property) = $string =~ /^\s*(.+?`\S+`)/) {
            $self->_drop($string, $property);
        }

    }

    foreach my $string (sort $comparator->get_columns_to_drop()) {
        my ($column) = $string =~ /^\s*(`\S+`)/;
        $self->_drop($string, $column);
    }
    foreach my $string (sort $comparator->get_columns_to_modify()) {
        my ($column_description) = $string =~ /^\s*(`\S+` .+?),?$/;
        $self->_modify($string, $column_description);
    }
    foreach my $string (sort $comparator->get_columns_to_create()) {
        my ($column_description) = $string =~ /^\s*(`\S+` .+?),?$/;
        $self->_add($string, $column_description);
    }

    {
        foreach my $string (reverse sort $comparator->get_strings_to_create()) {
            next if $self->_string_is_used($string);

            if (my ($key_with_description, $fk_columns) = $string =~ /^\s*(KEY `\S+` (\(.+\)).*?),?$/) {
                $self->_add($string, $key_with_description);

                my ($fk_string, $fk_description) =
                  $self->_parse_not_used_string(qr{^(\s*(CONSTRAINT `\S+` FOREIGN KEY \Q$fk_columns\E.*?),?)$},
                    $comparator->get_etalon_strings());

                $self->_add($fk_string, $fk_description)
                  if $fk_description;
            } elsif (my ($property_with_description) = $string =~ /^\s*(.+?),?$/) {
                $self->_add($string, $property_with_description);
            }
        }
    }

    return $self->_get_sql_alter_query();
}

sub _set_sql_alter_query {
    my ($self, $sql_alter_query) = @_;

    $self->{sql_alter_query} = $sql_alter_query;
}

sub _append_sql_alter_query {
    my ($self, $sql_alter_query_appendix) = @_;

    $self->{sql_alter_query} .= $sql_alter_query_appendix;
}

sub _drop {
    my ($self, $string, $property) = @_;

    $self->_remember_used_string($self->_get_current_method_name(), $string);
    $self->_append_sql_alter_query("ALTER TABLE `$self->{'table'}` DROP $property;\n");
}

sub _modify {
    my ($self, $string, $property_with_description) = @_;

    $self->_remember_used_string($self->_get_current_method_name(), $string);
    $self->_append_sql_alter_query("ALTER TABLE `$self->{'table'}` MODIFY $property_with_description;\n");
}

sub _add {
    my ($self, $string, $property_with_description) = @_;

    $self->_remember_used_string($self->_get_current_method_name(), $string);
    $self->_append_sql_alter_query("ALTER TABLE `$self->{'table'}` ADD $property_with_description;\n");
}

sub _get_current_method_name {
    return (caller(1))[3];
}

sub _set_last_command {
    my ($self, $command) = @_;

    $self->{'last_command'} = $command;
}

sub _get_last_command {
    my ($self) = @_;

    return $self->{'last_command'};
}

sub _cleanup_used_strings {
    my ($self) = @_;

    $self->{'used_strings'} = {};
}

sub _remember_used_string {
    my ($self, $command, $string) = @_;

    $self->_cleanup_used_strings()
      if $command ne $self->_get_last_command();

    $self->{'used_strings'}{$string} = undef;
}

sub _get_sql_alter_query {
    my ($self) = @_;

    return $self->{'sql_alter_query'};
}

sub _string_is_used {
    my ($self, $string) = @_;

    return exists($self->{'used_strings'}{$string});
}

sub _parse_not_used_string {
    my ($self, $regexp, @strings) = @_;

    my @not_used_strings =
      map {my @matches = /$regexp/; @matches ? \@matches : ()}
      @{arrays_difference(\@strings, [keys %{$self->{'used_strings'}}])};
    return @not_used_strings ? @{$not_used_strings[0]} : ();
}

TRUE;
