package SQL::Translator::Parser::QBit;

use strict;
use warnings;

our $VERSION = '1.59';

our $DEBUG;
$DEBUG = 0 unless defined $DEBUG;

use Storable qw(dclone);

use constant TRUE  => 1;
use constant FALSE => '';

use base qw(Exporter);
our @EXPORT_OK = qw(generate_meta parse);

sub generate_meta {
    my ($db, @table_names) = @_;

    return {tables => {map {$_ => eval('$db->' . $_) // die("Table $_ doesn't exist.\n"),} sort @table_names}};
}

sub parse {
    my ($translator, $data, $locales) = @_;
    $locales //= [qw(en ru)];

    local $::RD_TRACE = $translator->trace ? 1 : undef;
    local $DEBUG = $translator->debug;

    my $schema = $translator->schema;

    foreach my $table_name (keys %{$data->{'tables'}}) {
        my $tdata = $data->{tables}{$table_name};
        my $table = $schema->add_table(name => $table_name,) or die $schema->error;

        my @fields = @{$tdata->{'fields'}};
        foreach my $fdata (@fields) {
            if ($fdata->{'i18n'}) {
                foreach my $locale (@$locales) {
                    my $field = $table->add_field(
                        name              => $fdata->{'name'} . '_' . $locale,
                        data_type         => lc($fdata->{'type'}),
                        size              => $fdata->{'length'},
                        default_value     => _get_field_default_value($fdata, $tdata),
                        is_auto_increment => $fdata->{'autoincrement'},
                        ($fdata->{'not_null'} ? (is_nullable => !$fdata->{'not_null'}) : ()),
                    );
                    $field->extra(unsigned => 1) if $fdata->{'unsigned'};
                }
            } else {
                my $field = $table->add_field(
                    name      => $fdata->{'name'},
                    data_type => lc($fdata->{'type'}),
                    size      => (
                        lc($fdata->{'type'}) eq 'decimal'
                        ? [$fdata->{'length'}, $fdata->{'decimals'}]
                        : $fdata->{'length'}
                    ),
                    default_value     => _get_field_default_value($fdata, $tdata),
                    is_auto_increment => $fdata->{'autoincrement'},
                    ($fdata->{'not_null'} ? (is_nullable => !$fdata->{'not_null'}) : ()),
                );
                $field->extra(unsigned => 1) if $fdata->{'unsigned'};
            }
        }

        if ($tdata->{'primary_key'}) {
            foreach my $fname (@{$tdata->{'primary_key'}}) {
                $table->primary_key($fname);
            }
        }

        if (exists($tdata->{'foreign_keys'})) {
            my $fk_counter = 1;
            foreach my $foreign_key (@{$tdata->{'foreign_keys'}}) {
                my $constraint = $table->add_constraint(
                    name             => $table_name . '_ibfk_' . $fk_counter++,
                    type             => 'foreign_key',
                    fields           => $foreign_key->[0],
                    reference_table  => $foreign_key->[1],
                    reference_fields => $foreign_key->[2],
                ) or die $table->error;

                unless (_foreign_key_fields_already_in_index($foreign_key->[0], $tdata->{'indexes'})) {
                    my $index_name = _restrict_length(
                        join('__',
                            'fk_' . $table_name,
                            join('_', @{$foreign_key->[0]}),
                            '_' . $foreign_key->[1],
                            join('_', @{$foreign_key->[2]}))
                      )
                      unless (@{$foreign_key->[0]} == 1
                        && defined($tdata->{'primary_key'})
                        && $foreign_key->[0][0] eq $tdata->{'primary_key'}[0]);

                    my $index = $table->add_index(
                        name   => $index_name,
                        type   => 'NORMAL',
                        fields => $foreign_key->[0],
                    ) or die $table->error;
                }
            }
        }

        if ($tdata->{'indexes'}) {
            foreach my $index (@{$tdata->{'indexes'}}) {
                if ($index->{'unique'}) {
                    my $index = $table->add_constraint(
                        name   => _restrict_length(join('__', 'uniq_' . $table_name, join('_', @{$index->{'fields'}}))),
                        type   => 'UNIQUE',
                        fields => $index->{'fields'},
                    ) or die $table->error;
                } else {
                    my $index = $table->add_index(
                        name   => _restrict_length(join('__', $table_name, join('_', @{$index->{'fields'}}))),
                        type   => 'NORMAL',
                        fields => $index->{'fields'},
                    ) or die $table->error;
                }
            }
        }

        $table->options([{ENGINE => 'InnoDB'}, {'CHARACTER SET' => 'utf8'}]);
    }

    return 1;
}

sub _restrict_length {
    my ($string) = @_;
    my $max_length = 64;

    $string = substr($string, 0, $max_length)
      if defined $string && length($string) > $max_length;

    return $string;
}

sub _foreign_key_fields_already_in_index {
    my ($fk_fields, $indexes) = @_;

    return FALSE unless $indexes;

    foreach my $index (@$indexes) {
        my $index_fields = $index->{'fields'};
        my $is_in_index  = TRUE;
        foreach my $i (0 .. (@$fk_fields - 1)) {
            unless (defined($index_fields->[$i]) && $fk_fields->[$i] eq $index_fields->[$i]) {
                $is_in_index = FALSE;
                last;
            }
        }
        return TRUE if $is_in_index;
    }

    return FALSE;
}

sub _get_field_default_value {
    my ($fdata, $tdata) = @_;

    if (exists($fdata->{'default'})) {
        if (defined($fdata->{'default'}) && lc($fdata->{'type'}) eq 'decimal') {
            return sprintf("%.$fdata->{'decimals'}f", $fdata->{'default'});

        } else {
            return $fdata->{'default'};
        }
    }
    if (!exists($fdata->{'not_null'}) && grep {$fdata->{'name'} eq $_} @{$tdata->{'primary_key'}}) {
        return '' if lc($fdata->{'type'}) eq 'varchar';
        return 0
          if lc($fdata->{'type'}) eq 'int'
              || lc($fdata->{'type'}) eq 'bigint'
              || lc($fdata->{'type'}) eq 'smallint'
              || lc($fdata->{'type'}) eq 'tinyint';
    }

    return undef;
}

1;
