#!/usr/bin/perl -w

use qbit;

use FindBin qw($Bin);
use Test::More;
use Test::Differences;

use SQL::Translator;
use SQL::Translator::Parser::MySQL;

use lib "$Bin/../../lib/";
use Application;
use lib "$Bin/../lib/";
use SQL::Translator::Parser::QBit;

use Partner::DB qw(get_db_partner2);
my $db = get_db_partner2(server => 'test');

my $app = Application->new();
$app->pre_run();

my $meta = $app->partner_db->get_all_meta();
foreach my $table (values %{$meta->{'tables'}}) {
    foreach my $field (@{$table->{'fields'}}) {
        $field->{'type'} = lc($field->{'type'});
    }
}
my $tables = $meta->{'tables'};

foreach my $table_name ((sort keys %$tables)) {
    my $table_description =
      get_table_description_from_code($table_name, eval('$app->partner_db->' . $table_name . '->create_sql()'));
    $table_description = clean_mysql($table_description);
    my $table_meta = $tables->{$table_name};

    my $translator = SQL::Translator->new();
    my $meta = SQL::Translator::Parser::QBit::generate_meta($app->partner_db, $table_name);
    SQL::Translator::Parser::QBit::parse($translator, $meta);
    my $table_description_mysql_gotten = $translator->translate(producer => 'MySQL');
    $table_description_mysql_gotten =~ s/.*(CREATE .+?;).*/$1/s;

    test_eq_or_diff($table_description_mysql_gotten,
        $table_description, "$table_name - Correct MySQL is produced from QBit.");
    #    eq_or_diff($translator->translate(producer => 'QBit'), $meta, 'Correct QBit is produced from schema');
}

#print $translator->translate(producer => 'PostgreSQL'), "\n";
#print $translator->translate(producer => 'MySQL'), "\n";
#print $translator->translate(producer => 'QBit'), "\n";
#print Dumper $translator->translate(producer => 'SQLite');

done_testing();
exit;

sub clean_mysql {
    my ($sql) = @_;

    $sql =~ s/;?$/;/;

    my $translator = SQL::Translator->new();
    SQL::Translator::Parser::MySQL::parse($translator, $sql);
    $sql = $translator->translate(producer => 'MySQL');
    $sql =~ s/.*(CREATE .+?;).*/$1/s;
    $sql =~ s/ on update SCALAR\(\S+\)//g;
    $sql =~ s/ DEFAULT CURRENT_TIMESTAMP//g;
    $sql =~ s/\Qbigint(20)/bigint/g;
    $sql =~ s/\Qinteger(10)/integer/g;
    $sql =~ s/\Qinteger(11)/integer/g;
    $sql =~ s/\Qtinyint(3)/tinyint/g;
    $sql =~ s/\Qsmallint(5)/smallint/g;
    $sql =~ s/\Qfloat(8, 2)/float/g;
    $sql =~ s/\Q DEFAULT NULL//g;

    return $sql;
}

sub test_eq_or_diff {
    my ($gotten_sql, $expected_sql, $comment) = @_;

    $gotten_sql   = sort_mysql($gotten_sql);
    $expected_sql = sort_mysql($expected_sql);
    eq_or_diff($gotten_sql, $expected_sql, $comment) || &open_with_vimdiff($gotten_sql, $expected_sql);
}

sub open_with_vimdiff {
    my ($gotten_sql, $expected_sql) = @_;

    my $gotten_file   = 'gotten.tmp';
    my $expected_file = 'expected.tmp';
    writefile($gotten_file,   $gotten_sql);
    writefile($expected_file, $expected_sql);

    system(qw(vimdiff), $gotten_file, $expected_file);
    unlink $gotten_file;
    unlink $expected_file;

    exit;
}

sub sort_mysql {
    my ($sql) = @_;

    $sql =~ s/enum\('0','1'\)(?: unsigned)?((?: NOT)? NULL) DEFAULT '(\d)'/tinyint(1)$1 DEFAULT $2/g;
    $sql =~ s/enum\('0','1'\)(?: unsigned)?((?: NOT)? NULL)/tinyint(1)$1/g;
    $sql =~ s/(INDEX .+)\Q(255)`)/$1`)/g;

    my @sql = split("\n", $sql);
    my %sql;
    @sql{@sql} = 0 .. @sql - 1;
    my $regexp = qr/^\s+(?:PRIMARY KEY|CONSTRAINT|INDEX|UNIQUE)/;
    $sql =
      join("\n", map {s/,?$/,/; $_;} sort {($a =~ $regexp && $b =~ $regexp) ? $a cmp $b : $sql{$a} <=> $sql{$b}} @sql);

    return $sql;
}

sub get_table_description_from_code {
    my ($table, $temporary_table_description) = @_;

    my $temporary_table = 'to_drop';
    $temporary_table_description =~ s/$table/$temporary_table/g;

    $db->execute("DROP TABLE IF EXISTS `to_drop`");
    $db->execute($temporary_table_description);
    my $description_from_code = get_table_description_from_db($temporary_table);
    $db->execute("DROP TABLE IF EXISTS `$temporary_table`");

    $description_from_code =~ s/$temporary_table/$table/g;

    return $description_from_code;
}

sub get_table_description_from_db {
    my ($table) = @_;
    return @{$db->get_data("SHOW CREATE TABLE $table")}[0]->{'Create Table'};
}
