# -*- coding: utf-8 -*-
import logging
from logging.config import dictConfig

from flask import (
    Blueprint,
    Flask,
)
from passport.backend.adm_api import (
    common,
    settings as adm_api_settings,
)
from passport.backend.adm_api.common.grants import get_grants_loader
from passport.backend.core.conf import settings
from passport.backend.core.dbmanager.manager import get_dbm
from passport.backend.core.dbmanager.sharder import (
    build_range_shard_function,
    get_sharder,
)


log = logging.getLogger('passport_adm_api.initialize')


def create_app():
    from passport.backend.adm_api.views import (
        restore as restore_controllers,
        meltingpot as meltingpot_controllers,
        account as account_controllers,
    )

    api = Blueprint('api', __name__)

    api.add_url_rule(
        '/restore/account_state/',
        view_func=restore_controllers.RestoreAccountStateView.as_view(),
        methods=['GET'],
    )

    api.add_url_rule(
        '/restore/attempt/',
        view_func=restore_controllers.RestoreSemiAutoAttemptView.as_view(),
        methods=['GET'],
    )

    api.add_url_rule(
        '/restore/attempt/exists/',
        view_func=restore_controllers.RestoreSemiAutoAttemptExistsView.as_view(),
        methods=['GET'],
    )

    api.add_url_rule(
        '/restore/attempts/',
        view_func=restore_controllers.RestoreSemiAutoAttemptsView.as_view(),
        methods=['GET'],
    )

    api.add_url_rule(
        '/restore/support_decision/',
        view_func=restore_controllers.RestoreSemiAutoSupportDecisionView.as_view(),
        methods=['POST'],
    )

    api.add_url_rule(
        '/meltingpot/<path:path>',
        view_func=meltingpot_controllers.MeltingpotProxyView.as_view(),
        methods=['GET', 'POST', 'PUT', 'DELETE'],
    )

    api.add_url_rule(
        '/account/auths_aggregated/',
        view_func=account_controllers.AuthsView.as_view(),
        methods=['GET', 'POST'],
    )

    api.add_url_rule(
        '/account/profile/',
        view_func=account_controllers.ProfileView.as_view(),
        methods=['GET', 'POST'],
    )

    api.add_url_rule(
        '/account/botnetia_stats/',
        view_func=account_controllers.BotnetiaStatsView.as_view(),
        methods=['GET', 'POST'],
    )

    api.add_url_rule(
        '/account/public_id/remove_all/',
        view_func=account_controllers.RemoveAllPublicIdView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/public_id/remove/',
        view_func=account_controllers.RemovePublicIdView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/public_id/set/',
        view_func=account_controllers.SetPublicIdView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/set_is_verified/',
        view_func=account_controllers.IsVerifiedView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/set_takeout_subscription/',
        view_func=account_controllers.TakeoutSubscriptionView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/set_sms_2fa/',
        view_func=account_controllers.Sms2FAToggleView.as_view(),
        methods=['POST'],
    )
    api.add_url_rule(
        '/account/set_takeout_delete_subscription/',
        view_func=account_controllers.TakeoutDeleteSubscriptionView.as_view(),
        methods=['POST'],
    )

    api.add_url_rule(
        '/account/phonish/disable_auth/',
        view_func=account_controllers.AccountPhonishDisableAuth.as_view(),
        methods=['POST'],
    )

    app = Flask(__name__)
    app.register_blueprint(api, url_prefix='/1')

    app.before_request(common.add_env)
    app.before_request(common.setup_log_prefix)
    app.before_request(common.log_request)

    app.after_request(common.log_response)

    return app


def execute_app():  # pragma: no cover
    # настраиваем settings
    settings.configure(adm_api_settings)
    # настраиваем логгинг
    dictConfig(settings.LOGGING)

    # создаем подключения к базе
    for db_name, config in settings.DB_CONFIG.items():
        get_dbm(db_name).configure(config)

    shard_function = build_range_shard_function(settings.DB_SHARDING_RANGES)
    for table_name, config in settings.DB_SHARDING_CONFIG.items():
        get_sharder(table_name).configure(config, shard_function),

    # заранее подгружаем файл с ролями и грантами
    get_grants_loader()

    application = create_app()
    application.config['LOGGER_NAME'] = settings.APP_LOGGER_NAME

    return application
