# -*- coding: utf-8 -*-

from passport.backend.core.validators.utils import convert_formencode_invalid_to_error_list
from passport.backend.utils.string import smart_text


class BaseAdmError(Exception):
    """Базовый класс исключения для всех ошибок, которые будут
    сконвертированы в простой текстовый код ошибки"""
    error = None

    @property
    def errors(self):
        if not self.error:
            return []
        return [self.error]

    def __str__(self):
        result = 'errors: [%s]' % ', '.join(self.errors)
        args = self.args
        if args:
            result += '; args: %s' % ', '.join(map(smart_text, args))
        return result

    def __repr__(self):
        return '<%s: %s>' % (self.__class__.__name__, str(self))


class ValidationFailedError(BaseAdmError):
    """Ошибки в форме валидации"""
    def __init__(self, invalid_exception, *args, **kwargs):
        self._invalid_exception = invalid_exception
        super(ValidationFailedError).__init__(*args, **kwargs)

    @property
    def errors(self):
        return convert_formencode_invalid_to_error_list(self._invalid_exception, None)


class BlackboxUnavailableError(BaseAdmError):
    """Временная ошибка от ЧЯ"""
    error = 'backend.blackbox_failed'


class BlackboxPermanentError(BaseAdmError):
    """Постоянная ошибка от ЧЯ"""
    error = 'backend.blackbox_permanent_error'


class HistoryDBApiUnavailableError(BaseAdmError):
    """Временная ошибка от HistoryDB"""
    error = 'backend.historydb_api_failed'


class HistoryDBPermanentError(BaseAdmError):
    """Постоянная ошибка от historydb-api"""
    error = 'backend.historydb_api_permanent_error'


class SessionidInvalidError(BaseAdmError):
    """Значение Session_id / sessionid2 не прошло проверку в ЧЯ"""
    error = 'sessionid.invalid'


class UnhandledError(BaseAdmError):
    error = 'exception.unhandled'


class InternalError(BaseAdmError):
    """Случилось что-то плохое, и мы ничего сделать не можем"""
    error = 'internal'


class AccessDeniedError(BaseAdmError):
    """У пользователя не хватает грантов для выполнения операции"""
    error = 'access.denied'


class AuthorizationHeaderError(BaseAdmError):
    """Невалидный заголовок авторизации"""
    error = 'authorization.invalid'


class AccountDisabledError(BaseAdmError):
    """Аккаунт заблокирован"""
    error = 'account.disabled'


class OAuthTokenValidationError(BaseAdmError):
    """Невалидный токен"""
    error = 'oauth_token.invalid'


class AccountNotFoundError(BaseAdmError):
    """Аккаунт не найден/не существует/никогда не существовал"""
    error = 'account.not_found'


class AccountInvalidTypeError(BaseAdmError):
    """Аккаунт не поддерживает выполнение операции"""
    error = 'account.invalid_type'


class RestoreIdNotFoundError(BaseAdmError):
    """Событие в restore-логе с заданным идентификатором не найдено"""
    error = 'restore_id.not_found'


class RestoreIdUnknownVersionError(BaseAdmError):
    """Версия анкеты, соответствующей данному restore_id, не поддерживается"""
    error = 'restore_id.unknown_version'


class ActionNotRequiredError(BaseAdmError):
    """Какое-либо действие нельзя выполнить, т.к. оно уже выполнено"""
    error = 'action.not_required'


class ActionNotAllowedError(BaseAdmError):
    """Какое-либо действие запрещено"""
    error = 'action.not_allowed'


class HeadersEmptyError(BaseAdmError):
    """Отсутствуют необходимые заголовки запроса"""
    def __init__(self, aliases, *args, **kwargs):
        # Список коротких алиасов
        self.aliases = aliases
        super(HeadersEmptyError).__init__(*args, **kwargs)

    @property
    def errors(self):
        return [('%s.empty' % a) for a in self.aliases]


class MeltingpotUnavailableError(BaseAdmError):
    """Временная ошибка от котла"""
    error = 'backend.meltingpot_failed'


class MeltingpotHandlerNotFound(BaseAdmError):
    """Временная ошибка от котла"""
    error = 'backend.meltingpot_handler_not_found'


class ProxyError(BaseAdmError):
    """Ошибка для проксирования ошибок"""

    def __init__(self, errors):
        self._errors = errors

    @property
    def errors(self):
        return self._errors


class PassportUnavailableError(BaseAdmError):
    """Временная ошибка от Паспорта"""
    error = 'backend.passport_failed'


class PassportPermanentError(BaseAdmError):
    """Постоянная ошибка от Паспорта"""
    error = 'backend.passport_permanent_error'


class UfoUnavailableError(BaseAdmError):
    """Временная ошибка от UfoApi"""
    error = 'backend.ufo_api_failed'


class YtDtUnavailableError(BaseAdmError):
    """Ошибка от Yt"""
    error = 'backend.yt_error'


class PublicIdNotFoundError(BaseAdmError):
    """у аккаунта нет такого public_id"""
    error = 'public_id.not_found'


class PublicIdNotAvailable(BaseAdmError):
    """public_id не доступен"""
    error = 'public_id.not_available'
